/* $Id: pftop.c,v 1.65 2002/09/10 18:57:32 canacar Exp $	 */
/*
 * Copyright (c) 2001 Can Erkin Acar
 * Copyright (c) 2001 Daniel Hartmeier
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <net/if.h>
#include <netinet/in.h>
#define TCPSTATES
#include <netinet/tcp_fsm.h>
#include <net/pfvar.h>
#include <arpa/inet.h>

#include <ctype.h>
#include <curses.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>

#define MIN_NUM_STATES 1024
#define NUM_STATE_INC  1024

#define DEFAULT_WIDTH  80
#define DEFAULT_HEIGHT 25

#define HEADER_LINES	3
#define HOME_X	0
#define HOME_Y	1

#define CTRL_A  1
#define CTRL_B  2
#define CTRL_E  5
#define CTRL_F  6
#define CTRL_H  8
#define CTRL_L  12
#define CTRL_N  14
#define CTRL_P  16
#define CTRL_V  22

#define META_V  246

#if OS_LEVEL < 32
/* UDP state enumeration */
#define PFUDPS_NSTATES		3	/* number of state levels */

#define PFUDPS_NAMES { \
	"NO TRAFFIC", \
	"SINGLE", \
	"MULTIPLE", \
	NULL \
}

/* Other protocol state enumeration */
#define PFOTHERS_NSTATES	3	/* number of state levels */

#define PFOTHERS_NAMES { \
	"NO TRAFFIC", \
	"SINGLE", \
	"MULTIPLE", \
	NULL \
}
#endif

/* view management */
int read_states(int dev);
void sort_states(void);
void print_states(void);
int read_rules(int dev);
void print_rules(void);

/* qsort callbacks */
int sort_size_callback(const void *s1, const void *s2);
int sort_exp_callback(const void *s1, const void *s2);
int sort_pkt_callback(const void *s1, const void *s2);
int sort_age_callback(const void *s1, const void *s2);
int sort_sa_callback(const void *s1, const void *s2);
int sort_sp_callback(const void *s1, const void *s2);
int sort_da_callback(const void *s1, const void *s2);
int sort_dp_callback(const void *s1, const void *s2);

struct pf_state *state_buf = NULL;
int state_buf_len = 0;
u_int32_t *state_ord = NULL;
u_int32_t num_states = 0;
u_int32_t num_disp = 0;
u_int32_t num_rules = 0;
int columns, lines;

int delay = 5;
int interactive = 1;
int sortdir = 1;
int rawmode = 0;
int maxprint = 0;
int dispstart = 0;
int rawwidth = DEFAULT_WIDTH;

volatile sig_atomic_t gotsig_close = 0;
volatile sig_atomic_t gotsig_resize = 0;
volatile sig_atomic_t gotsig_alarm = 0;
int need_update = 0;
int need_sort = 0;

#define FLD_ALIGN_LEFT   0
#define FLD_ALIGN_RIGHT  1
#define FLD_ALIGN_CENTER 2
#define FLD_ALIGN_COLUMN 3

SCREEN *screen;
typedef struct {
	char *title;
	int norm_width;
	int max_width;
	int increment;
	int align;
	int start;
	int width;
} field_def;
/* for states */
#define FLD_SRC     0
#define FLD_DEST    1
#define FLD_GW      2
#define FLD_STATE   3
#define FLD_AGE     4
#define FLD_EXP     5
/* common */
#define FLD_PROTO   6
#define FLD_DIR     7
#define FLD_PKTS    8
#define FLD_BYTES   9
#define FLD_RULE   10
/* for rules */
#define FLD_LABEL  11
#define FLD_STATS  12
#define FLD_EVAL   13
#define FLD_ACTION 14
#define FLD_LOG    15
#define FLD_QUICK  16
#define FLD_KST    17
#define FLD_IF     18
#define FLD_RINFO  19

#if OS_LEVEL < 31
#define PF_RULE_LABEL_SIZE 20
#endif

field_def fields[] = {
	{"SRC", 20, 45, 1, FLD_ALIGN_LEFT, -1, 0},
	{"DEST", 20, 45, 1, FLD_ALIGN_LEFT, -1, 0},
	{"GW", 20, 45, 1, FLD_ALIGN_LEFT, -1, 0},
	{"STATE", 5, 23, 18, FLD_ALIGN_COLUMN, -1, 0},
	{"AGE", 5, 9, 4, FLD_ALIGN_RIGHT, -1, 0},
	{"EXP", 5, 9, 4, FLD_ALIGN_RIGHT, -1, 0},
	{"PR ", 4, 4, 1, FLD_ALIGN_LEFT, -1, 0},
	{"DIR", 1, 3, 2, FLD_ALIGN_CENTER, -1, 0},
	{"PKTS", 5, 8, 1, FLD_ALIGN_RIGHT, -1, 0},
	{"BYTES", 5, 8, 1, FLD_ALIGN_RIGHT, -1, 0},
	{"RULE", 2, 4, 1, FLD_ALIGN_RIGHT, -1, 0},
	{"LABEL", 20, PF_RULE_LABEL_SIZE, 1, FLD_ALIGN_LEFT, -1, 0},
	{"STATES", 5, 8, 1, FLD_ALIGN_RIGHT, -1, 0},
	{"EVAL", 5, 8, 1, FLD_ALIGN_RIGHT, -1, 0},
	{"ACT", 1, 5, 4, FLD_ALIGN_LEFT, -1, 0},
	{"LOG", 1, 3, 2, FLD_ALIGN_LEFT, -1, 0},
	{"QUICK", 1, 1, 1, FLD_ALIGN_LEFT, -1, 0},
	{"KS", 1, 1, 1, FLD_ALIGN_LEFT, -1, 0},
	{"IF", 4, 6, 1, FLD_ALIGN_LEFT, -1, 0},
	{"INFO", 40, 80, 1, FLD_ALIGN_LEFT, -1, 0}
};

int *fieldseq = 0;

int view0[] = {
	FLD_PROTO, FLD_DIR, FLD_SRC, FLD_DEST, FLD_STATE,
	FLD_AGE, FLD_EXP, FLD_PKTS, FLD_BYTES, -1
};
int view1[] = {
	FLD_PROTO, FLD_DIR, FLD_SRC, FLD_DEST, FLD_GW, FLD_STATE,
	FLD_AGE, FLD_EXP, FLD_PKTS, FLD_BYTES, FLD_RULE, -1
};
int view2[] = {
	FLD_PROTO, FLD_DIR, FLD_SRC, FLD_DEST, FLD_STATE,
	FLD_AGE, FLD_EXP, FLD_PKTS, FLD_BYTES, FLD_RULE, FLD_GW, -1
};
int view3[] = {
	FLD_PROTO, FLD_DIR, FLD_SRC, FLD_DEST, FLD_AGE, FLD_EXP,
	FLD_PKTS, FLD_BYTES, FLD_STATE, FLD_RULE, FLD_GW, -1
};
int view4[] = {
	FLD_PROTO, FLD_DIR, FLD_SRC, FLD_DEST, FLD_PKTS, FLD_BYTES,
	FLD_STATE, FLD_AGE, FLD_EXP, FLD_RULE, FLD_GW, -1
};

int view5[] = {
	FLD_RULE, FLD_ACTION, FLD_DIR, FLD_LOG, FLD_QUICK, FLD_IF,
	FLD_PROTO, FLD_KST, FLD_PKTS, FLD_BYTES, FLD_STATS, FLD_RINFO, -1
};

int view6[] = {
	FLD_RULE, FLD_LABEL, FLD_PKTS, FLD_BYTES, FLD_STATS,
	FLD_ACTION, FLD_DIR, FLD_LOG, FLD_QUICK, FLD_IF, FLD_PROTO,
	FLD_KST,  -1
};

struct view_manager {
	char *name;
	int  (*read_fn) (int);
	void (*sort_fn) (void);
	void (*print_fn) (void);
};

struct view_manager state_mgr = {
	"State", read_states, sort_states, print_states
};

struct view_manager rule_mgr = {
	"Rule", read_rules, NULL, print_rules
};

typedef struct {
	int *view;
	char *name;
	int hotkey;
	struct view_manager *mgr;
} field_view;

field_view views[] = {
	{view0, "default", '0', &state_mgr},
	{view1, "long", '1', &state_mgr},
	{view2, "state", '2', &state_mgr},
	{view3, "time", '3', &state_mgr},
	{view4, "size", '4', &state_mgr},
	{view5, "rules", '5', &rule_mgr},
	{view6, "label", '6', &rule_mgr},
	{NULL, NULL, 0, NULL}
};

field_view *curr_view = NULL;
struct view_manager *curr_mgr = NULL;

int num_fields = sizeof(fields) / sizeof(field_def);
int curr_line = 0;

/* line buffer for raw mode */
#define MAX_LINE_BUF 1024
char linebuf[MAX_LINE_BUF];
int linepos = 0;

/* temp storage for state printing */
char tmp_buf[MAX_LINE_BUF];

/* ordering */
typedef struct {
	char *name;
	char *match;
	int hotkey;
	int (*func) (const void *, const void *);
} order_type;

order_type order_list[] = {
	{"none", "none", 'N', NULL},
	{"bytes", "bytes", 'B', sort_size_callback},
	{"expiry", "exp", 'E', sort_exp_callback},
	{"packets", "pkt", 'P', sort_pkt_callback},
	{"age", "age", 'A', sort_age_callback},
	{"source addr", "src", 'F', sort_sa_callback},
	{"dest. addr", "dest", 'T', sort_da_callback},
	{"source port", "sport", 'S', sort_sp_callback},
	{"dest. port", "dport", 'D', sort_dp_callback},
	{NULL, NULL, 0, NULL}
};

order_type *ordering = NULL;

/* command prompt */

struct command {
	char *prompt;
	void ( *exec)(void);
};

char cmdbuf[MAX_LINE_BUF];
int cmd_len = -1;
struct command *curr_cmd = NULL;

void print_cmdline(void);
void cmd_delay(void);
void cmd_count(void);

struct command cm_delay = {"Seconds to delay", cmd_delay};
struct command cm_count = {"Number of lines to display", cmd_count};

/* screen output functions */

char * tb_ptr = NULL;
int tb_len = 0;

void
tb_start(void)
{
	tb_ptr = tmp_buf;
	tb_len = sizeof(tmp_buf);
}

void
tb_end(void)
{
	tb_ptr = NULL;
	tb_len = 0;
}

int
tbprintf(char *format, ...)
	GCC_PRINTFLIKE(1,2)       /* defined in curses.h */
{
	int len;
	va_list arg;

	if (tb_ptr == NULL || tb_len <= 0)
		return 0;

	va_start(arg, format);
	len=vsnprintf(tb_ptr, tb_len, format, arg);
	va_end(arg);
	
	if (len > tb_len)
		tb_end();
	else if (len > 0) {
		tb_ptr += len;
		tb_len -= len;
	}
	
	return len;
}

void
move_horiz(int offset)
{
	if (rawmode) {
		if (offset <= 0)
			linepos = 0;
		/* XXX: Check against rawwidth instead? */
		else if (offset >= MAX_LINE_BUF)
			linepos = MAX_LINE_BUF - 1;
		else
			linepos = offset;
	} else {
		move(curr_line, offset);
	}
}

void
print_str(int len, char *str)
{
	if (len <= 0)
		return;

	if (rawmode) {
		int length = MIN(len, MAX_LINE_BUF - linepos);
		if (length <= 0)
			return;
		bcopy(str, &linebuf[linepos], length);
		linepos += length;
	} else
		addnstr(str, len);
}

void
clear_linebuf(void)
{
	memset(linebuf, ' ', MAX_LINE_BUF);
}

void
end_line(void)
{
	if (rawmode) {
		if (linepos >= MAX_LINE_BUF)
			linepos = MAX_LINE_BUF - 1;
		linebuf[linepos] = '\0';
		printf("%s\n", linebuf);
		clear_linebuf();
	}
	curr_line++;
}

void
end_page(void)
{
	if (rawmode) {
		linepos = 0;
		clear_linebuf();
	} else {
		move(HOME_Y, HOME_X);
		print_cmdline();
		refresh();
	}
	curr_line = 0;
}

void
rawaddstr(char *s)
{
	if (rawmode)
		printf("%s", s);
	else
		addstr(s);
}

/* field output functions */

void
print_fld_str(int fid, char *str)
{
	field_def *fld;
	int len, move;
	char *cpos;

	if (str == NULL)
		return;
	if (fid < 0 || fid >= num_fields)
		return;

	fld = &fields[fid];
	if (fld->start < 0)
		return;

	len = strlen(str);
	if (len >= fld->width) {
		move_horiz(fld->start);
		print_str(fld->width, str);
	} else {
		switch (fld->align) {
		case FLD_ALIGN_RIGHT:
			move_horiz(fld->start + (fld->width - len));
			break;
		case FLD_ALIGN_CENTER:
			move_horiz(fld->start + (fld->width - len) / 2);
			break;
		case FLD_ALIGN_COLUMN:
			if ((cpos = strchr(str, ':')) == NULL) {
				move = (fld->width - len) / 2;
			} else {
				move = (fld->width / 2) - (cpos - str);
				if (move < 0)
					move = 0;
				else if (move > (fld->width - len))
					move = fld->width - len;
			}
			move_horiz(fld->start + move);
			break;
		default:
			move_horiz(fld->start);
			break;
		}
		print_str(len, str);
	}
}

void
print_title(void)
{
	int n;

	for (n = 0; n < num_fields; n++)
		print_fld_str(n, fields[n].title);
	end_line();
}

/* view related functions */

void
field_setup(void)
{
	int *v, n, st, fwid, change;
	int width = columns;

	for (n = 0; n < num_fields; n++) {
		fields[n].start = -1;
		fields[n].width = fields[n].norm_width;
	}

	if (curr_view == NULL)
		return;

	dispstart = 0;
	st = 0;

	for (v = curr_view->view; *v >= 0 && *v < num_fields; v++) {
		if (width <= 1)
			break;
		if (st != 1)
			width--;
		n = *v;

		fields[n].start = 1;
		fwid = fields[n].width;
		st++;
		if (fwid >= width) {
			fields[n].width = width;
			width = 0;
		} else
			width -= fwid;
	}

	change = 0;
	while (width > 0) {
		change = 0;
		for (v = curr_view->view; *v >= 0 && *v < num_fields; v++) {
			n = *v;
			if ((fields[n].width < fields[n].max_width) &&
			    (fields[n].increment <= width)) {
				int w = fields[n].width + fields[n].increment;
				if (w > fields[n].max_width)
					w = fields[n].max_width;
				width += fields[n].width - w;
				fields[n].width = w;
				change = 1;
			}
			if (width <= 0) break;
		}
		if (change == 0) break;
	}

	st = 0;
	for (v = curr_view->view; *v >= 0 && *v < num_fields; v++) {
		n = *v;
		if (fields[n].start < 0) break;
		fields[n].start = st;
		st += fields[n].width + 1;
	}
}

void
set_curr_view(field_view *v)
{
	if (v == NULL) {
		curr_view = NULL;
		curr_mgr = NULL;
		return;
	}

	if ((curr_view != NULL) && (curr_view->mgr != v->mgr))
		gotsig_alarm = 1;

	curr_view = v;
	curr_mgr = v->mgr;
	field_setup();
	need_update = 1;
}

void
set_view(char *opt)
{
	field_view *v;

	set_curr_view(views);

	if (opt == NULL)
		return;

	for (v = views; v->name != NULL; v++) {
		if (strcasecmp(opt, v->name) == 0) {
			set_curr_view(v);
			return;
		}
	}
}

int
set_view_hotkey(int ch)
{
	field_view *v;
	int key = tolower(ch);

	for (v = views; v->name != NULL; v++) {
		if (key == v->hotkey) {
			set_curr_view(v);
			return 1;
		}
	}
	return 0;
}

void
next_view(void)
{
	field_view *v;

	for (v = views; v->name != NULL; v++) {
		if (curr_view == v) {
			v++;
			if (v->name == NULL)
				break;
			set_curr_view(v);
			return;
		}
	}
	set_curr_view(views);
}

void
prev_view(void)
{
	field_view *v, *p, *cv;

	p = NULL;
	cv = curr_view;

	for (v = views; v->name != NULL; v++) {
		if (cv == v)
			cv = p;
		p = v;
	}
	if (cv == NULL)
		cv = p;

	set_curr_view(cv);
}

/* ordering functions */

int
sort_size_callback(const void *s1, const void *s2)
{
	if (state_buf[* (u_int32_t *) s2].bytes >
	    state_buf[* (u_int32_t *) s1].bytes)
		return sortdir;
	return -sortdir;
}

int
sort_pkt_callback(const void *s1, const void *s2)
{
	if (state_buf[* (u_int32_t *) s2].packets >
	    state_buf[* (u_int32_t *) s1].packets)
		return sortdir;
	return -sortdir;
}

int
sort_age_callback(const void *s1, const void *s2)
{
	if (state_buf[* (u_int32_t *) s2].creation >
	    state_buf[* (u_int32_t *) s1].creation)
		return sortdir;
	return -sortdir;
}

int
sort_exp_callback(const void *s1, const void *s2)
{
	if (state_buf[* (u_int32_t *) s2].expire >
	    state_buf[* (u_int32_t *) s1].expire)
		return sortdir;
	return -sortdir;
}

int
compare_addr(int af, const struct pf_addr *a, const struct pf_addr *b)
{
	switch (af) {
	case AF_INET:
		if (ntohl(a->addr32[0]) > ntohl(b->addr32[0]))
			return 1;
		if (a->addr32[0] != b->addr32[0])
			return -1;
		break;
	case AF_INET6:
		if (ntohl(a->addr32[0]) > ntohl(b->addr32[0]))
			return 1;
		if (a->addr32[0] != b->addr32[0])
			return -1;
		if (ntohl(a->addr32[1]) > ntohl(b->addr32[1]))
			return 1;
		if (a->addr32[1] != b->addr32[1])
			return -1;
		if (ntohl(a->addr32[2]) > ntohl(b->addr32[2]))
			return 1;
		if (a->addr32[2] != b->addr32[2])
			return -1;
		if (ntohl(a->addr32[3]) > ntohl(b->addr32[3]))
			return 1;
		if (a->addr32[3] != b->addr32[3])
			return -1;
		break;
	}
	
	return 0;
}

#ifdef __GNUC__
__inline__
#endif
int
sort_addr_callback(const struct pf_state *s1,
		   const struct pf_state *s2, int dir)
{
	const struct pf_state_host *a, *b;
	int af, ret;

	af = s1->af;

	if (af > s2->af)
		return sortdir;
	if (af < s2->af)
		return -sortdir;
	
	if (s1->direction == dir) {
		a = &s1->lan;
	} else {
		a = &s1->ext;
	}

	if (s2->direction == dir) {
		b = &s2->lan;
	} else {
		b = &s2->ext;
	}

	ret = compare_addr(af, &a->addr, &b->addr);
	if (ret)
		return ret * sortdir;

	if (ntohs(a->port) > ntohs(b->port))
		return sortdir;
	return -sortdir;
}

int sort_sa_callback(const void *p1, const void *p2)
{
	struct pf_state *s1 = state_buf + (* (u_int32_t *) p1);
	struct pf_state *s2 = state_buf + (* (u_int32_t *) p2);
	return sort_addr_callback(s1, s2, PF_OUT);
}

int sort_da_callback(const void *p1, const void *p2)
{
	struct pf_state *s1 = state_buf + (* (u_int32_t *) p1);
	struct pf_state *s2 = state_buf + (* (u_int32_t *) p2);
	return sort_addr_callback(s1, s2, PF_IN);
}

#ifdef __GNUC__
__inline__
#endif
int
sort_port_callback(const struct pf_state *s1,
		   const struct pf_state *s2, int dir)
{
	const struct pf_state_host *a, *b;
	int af;

	af = s1->af;

	if (af > s2->af)
		return sortdir;
	if (af < s2->af)
		return -sortdir;
	
	if (s1->direction == dir) {
		a = &s1->lan;
	} else {
		a = &s1->ext;
	}

	if (s2->direction == dir) {
		b = &s2->lan;
	} else {
		b = &s2->ext;
	 }

	if (ntohs(a->port) > ntohs(b->port))
		return sortdir;
	if (ntohs(a->port) < ntohs(b->port))
		return -sortdir;

	if (compare_addr(af, &a->addr, &b->addr) > 0)
		return sortdir;
	return -sortdir;
}

int
sort_sp_callback(const void *p1, const void *p2)
{
	struct pf_state *s1 = state_buf + (* (u_int32_t *) p1);
	struct pf_state *s2 = state_buf + (* (u_int32_t *) p2);
	return sort_port_callback(s1, s2, PF_OUT);
}

int
sort_dp_callback(const void *p1, const void *p2)
{
	struct pf_state *s1 = state_buf + (* (u_int32_t *) p1);
	struct pf_state *s2 = state_buf + (* (u_int32_t *) p2);
	return sort_port_callback(s1, s2, PF_IN);
}

void
set_order(char *opt)
{
	order_type *o;

	ordering = order_list;

	if (opt == NULL)
		return;

	for (o = order_list; o->name != NULL; o++) {
		if (strcasecmp(opt, o->match) == 0) {
			ordering = o;
			return;
		}
	}
}

int
set_order_hotkey(int ch)
{
	order_type *o;
	int key = ch;

	for (o = order_list; o->name != NULL; o++) {
		if (key == o->hotkey) {
			if (ordering == o) {
				sortdir *= -1;
			} else {
				ordering = o;
			}
			return 1;
		}
	}
	return 0;
}

void
next_order(void)
{
	order_type *o;

	for (o = order_list; o->name != NULL; o++) {
		if (ordering == o) {
			o++;
			if (o->name == NULL)
				break;
			ordering = o;
			return;
		}
	}
	ordering = order_list;
}

void
sort_states(void)
{
	if (ordering == NULL)
		return;
	if (ordering->func == NULL)
		return;
	if (state_buf == NULL)
		return;
	if (num_states <= 0)
		return;

	heapsort(state_ord, num_states, sizeof(u_int32_t), ordering->func);
}

/* state management functions */

void
alloc_buf(int ns)
{
	int len;

	if (ns < MIN_NUM_STATES)
		ns = MIN_NUM_STATES;

	len = ns;

	if (len >= state_buf_len) {
		len += NUM_STATE_INC;
		state_buf = realloc(state_buf, len * sizeof(struct pf_state));
		state_ord = realloc(state_ord, len * sizeof(u_int32_t));
		if (state_buf == NULL || state_ord == NULL)
			err(1, "realloc");
		state_buf_len = len;
	}
}

int
read_states(int dev)
{
	struct pfioc_states ps;
	int n;

	for (;;) {
		int sbytes = state_buf_len * sizeof(struct pf_state);

		ps.ps_len = sbytes;
		ps.ps_buf = (char *) state_buf;

		if (ioctl(dev, DIOCGETSTATES, &ps) < 0) {
			errx(1, "DIOCGETSTATES");
		}
		num_states = ps.ps_len / sizeof(struct pf_state);

		if (ps.ps_len < sbytes)
			break;

		alloc_buf(num_states);
	}

	for (n = 0; n<num_states; n++)
		state_ord[n] = n;

	num_disp = num_states;
	return 0;
}

int
unmask(struct pf_addr * m, u_int8_t af)
{
	int i = 31, j = 0, b = 0, msize;
	u_int32_t tmp;

	if (af == AF_INET)
		msize = 1;
	else
		msize = 4;
	while (j < msize && m->addr32[j] == 0xffffffff) {
		b += 32;
		j++;
	}
	if (j < msize) {
		tmp = ntohl(m->addr32[j]);
		for (i = 31; tmp & (1 << i); --i)
			b++;
	}
	return (b);
}

/* display functions */

int
print_header(struct pf_status *status)
{
	struct tm *tp;
	time_t t;

	int start = dispstart + 1;
	int end = dispstart + maxprint;
	
	if (end > num_disp)
		end = num_disp;

	tb_start();
	tbprintf("pfTop: ");

	if (status)
		tbprintf(status->running ? "Up" : "Down");
	else
		tbprintf("Unknown!");

	tbprintf(" %s", curr_mgr ? curr_mgr->name : "???");

	if (num_disp == 0)
		tbprintf(" no entries");
	else
		tbprintf(" %u-%u/%u", start, end, num_disp);
	
	if (curr_view) {
		tbprintf(", View: %s", curr_view->name);
	}

	if (curr_mgr && curr_mgr->sort_fn != NULL && ordering) {
		tbprintf(", Order: %s", ordering->name);
		if (sortdir < 0 && ordering->func != NULL)
			tbprintf(" (rev)");
	}

	if (rawmode)
		printf("\n\n%s\n", tmp_buf);
	else
		mvprintw(0, 0, "%s", tmp_buf);

	time(&t);
	tp = localtime(&t);

	if (tp) {
		int len;

		tb_start();
		tbprintf("%.2u:%.2u:%.2u", tp->tm_hour,
			 tp->tm_min, tp->tm_sec);
		len = columns - strlen(tmp_buf);
		if (len < 0)
			len = 0;
		mvprintw(0, len, "%s", tmp_buf);
	}

	tb_end();

	curr_line = 2;
	print_title();

	return (0);
}

void
tb_print_addr(struct pf_addr * addr, struct pf_addr * mask, int af)
{
	static char buf[48];
	const char *bf;

	bf = inet_ntop(af, addr, buf, sizeof(buf));
	tbprintf("%s", bf);

	if (mask != NULL) {
		if (!PF_AZERO(mask, af))
			tbprintf("/%u", unmask(mask, af));
	}
}

void
print_fld_host(int fid, struct pf_state_host * h, int af)
{
	u_int16_t p = ntohs(h->port);

	if (fid < 0 || fid >= num_fields)
		return;

	if (fields[fid].width < 3) {
		print_fld_str(fid, "*");
		return;
	}

	tb_start();
	tb_print_addr(&h->addr, NULL, af);

	if (af == AF_INET)
		tbprintf(":%u", p);
	else
		tbprintf("[%u]", p);

	print_fld_str(fid, tmp_buf);
	tb_end();
}

void
print_fld_age(int fid, unsigned int age)
{
	int len;
	unsigned int h, m, s;

	if (fid < 0 || fid >= num_fields)
		return;
	len = fields[fid].width;
	if (len < 1)
		return;

	s = age % 60;
	m = age / 60;
	h = m / 60;
	m %= 60;

	tb_start();
	if (tbprintf("%02u:%02u:%02u", h, m, s) <= len)
		goto ok;
	
	tb_start();
	if (tbprintf("%u", age) <= len)
		goto ok;

	age /= 60;
	tb_start();
	if (tbprintf("%um", age) <= len)
		goto ok;
	if (age == 0)
		goto err;
	
	age /= 60;
	tb_start();
	if (tbprintf("%uh", age) <= len)
		goto ok;
	if (age == 0)
		goto err;
	
	age /= 24;
	tb_start();
	if (tbprintf("%ud", age) <= len)
		goto ok;
	
 err:
	print_fld_str(fid, "*");
	tb_end();
	return;
	
 ok:
	print_fld_str(fid, tmp_buf);
	tb_end();
}

void
print_fld_state(int fid, unsigned int proto, unsigned int s1, unsigned int s2)
{
	int len;
	
	if (fid < 0 || fid >= num_fields)
		return;
	len = fields[fid].width;
	if (len < 1)
		return;
	
	tb_start();

	if (proto == IPPROTO_TCP) {
		if (s1 <= TCPS_TIME_WAIT && s2 <= TCPS_TIME_WAIT) {
			tbprintf("%s:%s", tcpstates[s1], tcpstates[s2]);
		} else {
			tbprintf("<BAD STATE LEVELS>");
		}
	} else if (proto == IPPROTO_UDP && s1 < PFUDPS_NSTATES &&
		   s2 < PFUDPS_NSTATES) {
		const char *states[] = PFUDPS_NAMES;
		tbprintf("%s:%s", states[s1], states[s2]);
	} else if (proto != IPPROTO_ICMP && s1 < PFOTHERS_NSTATES &&
		   s2 < PFOTHERS_NSTATES) {
		/* XXX ICMP doesn't really have state levels */
		const char *states[] = PFOTHERS_NAMES;
		tbprintf("%s:%s", states[s1], states[s2]);
	} else {
		tbprintf("%u:%u", s1, s2);
	}

	if (strlen(tmp_buf) > len) {
		tb_start();
		tbprintf("%u:%u", s1, s2);
	}

	print_fld_str(fid, tmp_buf);

	tb_end();
}

void
print_fld_size(int fid, unsigned int size)
{
	int len;

	if (fid < 0 || fid >= num_fields)
		return;
	len = fields[fid].width;
	if (len < 1)
		return;

	tb_start();
	if (tbprintf("%u", size) <= len)
		goto ok;

	size /= 1024;
	tb_start();
	if (tbprintf("%uK", size) <= len)
		goto ok;
	if (size == 0)
		goto err;

	size /= 1024;
	tb_start();
	if (tbprintf("%uM", size) <= len)
		goto ok;
	if (size == 0)
		goto err;

	size /= 1024;
	tb_start();
	if (tbprintf("%uG", size) <= len)
		goto ok;
	
err:
	print_fld_str(fid, "*");
	tb_end();
	return;

ok:
	print_fld_str(fid, tmp_buf);
	tb_end();
}

void
print_fld_uint(int fid, unsigned int size)
{
	int len;

	if (fid < 0 || fid >= num_fields)
		return;

	len = fields[fid].width;
	if (len < 1)
		return;

	tb_start();
	if (tbprintf("%u", size) > len)
		print_fld_str(fid, "*");
	else
		print_fld_str(fid, tmp_buf);
}

int
print_state(struct pf_state * s)
{
	struct pf_state_peer *src, *dst;
	struct protoent *p;

	if (s->direction == PF_OUT) {
		src = &s->src;
		dst = &s->dst;
	} else {
		src = &s->dst;
		dst = &s->src;
	}

	p = getprotobynumber(s->proto);

	if (p != NULL)
		print_fld_str(FLD_PROTO, p->p_name);
	else
		print_fld_uint(FLD_PROTO, s->proto);

	if (s->direction == PF_OUT) {
		print_fld_host(FLD_SRC, &s->lan, s->af);
		print_fld_host(FLD_DEST, &s->ext, s->af);
	} else {
		print_fld_host(FLD_SRC, &s->ext, s->af);
		print_fld_host(FLD_DEST, &s->lan, s->af);
	}

	if (PF_ANEQ(&s->lan.addr, &s->gwy.addr, s->af) ||
	    (s->lan.port != s->gwy.port)) {
		print_fld_host(FLD_GW, &s->gwy, s->af);
	}

	if (s->direction == PF_OUT)
		print_fld_str(FLD_DIR, "Out");
	else
		print_fld_str(FLD_DIR, "In");

	print_fld_state(FLD_STATE, s->proto, src->state, dst->state);
	print_fld_age(FLD_AGE, s->creation);
	print_fld_age(FLD_EXP, s->expire);
	print_fld_size(FLD_PKTS, s->packets);
	print_fld_size(FLD_BYTES, s->bytes);
#if OS_LEVEL > 31
	print_fld_uint(FLD_RULE, s->rule.nr);
#endif
	end_line();
	return 1;
}

void
print_states(void)
{
	int n, count = 0;

	for (n = dispstart; n < num_disp; n++) {
		count += print_state(state_buf + state_ord[n]);
		if (maxprint > 0 && count >= maxprint)
			break;
	}
}

int
disp_update(int dev)
{
	struct pf_status status;

	if (maxprint < 0)
		dispstart = 0;
	else if (dispstart + maxprint > num_disp)
		dispstart = num_disp - maxprint;
	
	if (dispstart < 0)
		dispstart = 0;

	if (ioctl(dev, DIOCGETSTATUS, &status)) {
		warnx("DIOCGETSTATUS");
		return (-1);
	}

	print_header(&status);

	if (curr_view == NULL)
		return 0;

	if (curr_mgr != NULL)
		if (curr_mgr->print_fn != NULL)
		curr_mgr->print_fn();

	return (0);
}

/* rule display */

struct pf_rule *rules = NULL;
u_int32_t alloc_rules = 0;

int
read_rules(int dev)
{
	struct pfioc_rule pr;
	u_int32_t nr;

	if (ioctl(dev, DIOCGETRULES, &pr)) {
		warnx("DIOCGETRULES");
		return (-1);
	}

	num_disp = num_rules = pr.nr;

	if (num_rules == 0) return (0);

	if (rules == NULL) {
		rules = malloc(num_rules * sizeof(struct pf_rule));
		if (rules == NULL)
			err(1, "malloc");
		alloc_rules = num_rules;
	} else if (num_rules > alloc_rules) {
		rules = realloc(rules, num_rules * sizeof(struct pf_rule));
		if (rules == NULL)
			err(1, "realloc");
		alloc_rules = num_rules;
	}

	for (nr = 0; nr < num_rules; ++nr) {
		pr.nr = nr;
		if (ioctl(dev, DIOCGETRULE, &pr)) {
			warnx("DIOCGETRULE");
			return (-1);
		}
		rules[nr] = pr.rule;
	}

	return (0);
}

#if OS_LEVEL > 31
void
tb_print_addrw(struct pf_addr_wrap *addr, struct pf_addr *mask, u_int8_t af)
{
	if (addr->addr_dyn != NULL)
		tbprintf("(%s)", addr->addr.pfa.ifname);
	else
		tb_print_addr(&addr->addr, mask, af);
}
#endif

void
tb_print_op(u_int8_t op, const char *a1, const char *a2)
{
	if (op == PF_OP_IRG)
		tbprintf("%s >< %s ", a1, a2);
	else if (op == PF_OP_XRG)
		tbprintf("%s <> %s ", a1, a2);
	else if (op == PF_OP_EQ)
		tbprintf("= %s ", a1);
	else if (op == PF_OP_NE)
		tbprintf("!= %s ", a1);
	else if (op == PF_OP_LT)
		tbprintf("< %s ", a1);
	else if (op == PF_OP_LE)
		tbprintf("<= %s ", a1);
	else if (op == PF_OP_GT)
		tbprintf("> %s ", a1);
	else if (op == PF_OP_GE)
		tbprintf(">= %s ", a1);
}

void
tb_print_port(u_int8_t op, u_int16_t p1, u_int16_t p2, char *proto)
{
	char a1[6], a2[6];
	struct servent *s = getservbyport(p1, proto);

	p1 = ntohs(p1);
	p2 = ntohs(p2);
	snprintf(a1, sizeof(a1), "%u", p1);
	snprintf(a2, sizeof(a2), "%u", p2);
	tbprintf("port ");
	if (s != NULL && (op == PF_OP_EQ || op == PF_OP_NE))
		tb_print_op(op, s->s_name, a2);
	else
		tb_print_op(op, a1, a2);
}

void
tb_print_fromto(struct pf_rule_addr *src, struct pf_rule_addr *dst,
		u_int8_t af, u_int8_t proto)
{
	if (
#if OS_LEVEL > 31
	    PF_AZERO(&src->addr.addr, AF_INET6) &&
	    PF_AZERO(&dst->addr.addr, AF_INET6) &&
	    !src->noroute && !dst->noroute &&
#else
	    PF_AZERO(&src->addr, AF_INET6) &&
	    PF_AZERO(&dst->addr, AF_INET6) &&
#endif
	    PF_AZERO(&src->mask, AF_INET6) &&
	    PF_AZERO(&dst->mask, AF_INET6) &&
	    !src->port_op && !dst->port_op)
		tbprintf("all ");
	else {
		tbprintf("from ");
#if OS_LEVEL > 30
		if (src->noroute)
			tbprintf("no-route ");
 #if OS_LEVEL > 31
		else if (PF_AZERO(&src->addr.addr, AF_INET6) &&
 #else
		else if (PF_AZERO(&src->addr, AF_INET6) &&
 #endif
#else
		if (PF_AZERO(&src->addr, AF_INET6) &&
#endif
		    PF_AZERO(&src->mask, AF_INET6))
			tbprintf("any ");
		else {
			if (src->not)
				tbprintf("! ");
#if OS_LEVEL > 31
			tb_print_addrw(&src->addr, &src->mask, af);
#else
			tb_print_addr(&src->addr, &src->mask, af);
#endif
			tbprintf(" ");
		}
		if (src->port_op)
			tb_print_port(src->port_op, src->port[0],
			    src->port[1],
			    proto == IPPROTO_TCP ? "tcp" : "udp");

		tbprintf("to ");
#if OS_LEVEL > 30
		if (dst->noroute)
			tbprintf("no-route ");
 #if OS_LEVEL > 31
		else if (PF_AZERO(&dst->addr.addr, AF_INET6) &&
 #else
		else if (PF_AZERO(&dst->addr, AF_INET6) &&
 #endif
#else
		if (PF_AZERO(&dst->addr, AF_INET6) &&
#endif
		    PF_AZERO(&dst->mask, AF_INET6))
			tbprintf("any ");
		else {
			if (dst->not)
				tbprintf("! ");
#if OS_LEVEL > 31
			tb_print_addrw(&dst->addr, &dst->mask, af);
#else
			tb_print_addr(&dst->addr, &dst->mask, af);
#endif
			tbprintf(" ");
		}
		if (dst->port_op)
			tb_print_port(dst->port_op, dst->port[0],
			    dst->port[1],
			    proto == IPPROTO_TCP ? "tcp" : "udp");
	}
}

void
print_rule(struct pf_rule *pr)
{
	if (pr == NULL) return;

#if OS_LEVEL > 30
	print_fld_str(FLD_LABEL, pr->label);
#endif
#if OS_LEVEL > 31
	print_fld_size(FLD_STATS, pr->states);
#endif
	print_fld_size(FLD_PKTS, pr->packets);
	print_fld_size(FLD_BYTES, pr->bytes);
	print_fld_uint(FLD_RULE, pr->nr);
	print_fld_str(FLD_DIR, pr->direction ? "Out" : "In");
	if (pr->quick)
		print_fld_str(FLD_QUICK, "Quick");

	if (pr->keep_state == PF_STATE_NORMAL)
		print_fld_str(FLD_KST, "Keep");
	else if (pr->keep_state == PF_STATE_MODULATE)
		print_fld_str(FLD_KST, "Mod");
	
	if (pr->log == 1)
		print_fld_str(FLD_LOG, "Log");
	else if (pr->log == 2)
		print_fld_str(FLD_LOG, "All");

	if (pr->action == PF_PASS)
		print_fld_str(FLD_ACTION, "Pass");
	else if (pr->action == PF_DROP)
		print_fld_str(FLD_ACTION, "Block");
	else print_fld_str(FLD_ACTION, "Scrub");

	if (pr->proto) {
		struct protoent *p = getprotobynumber(pr->proto);

		if (p != NULL)
			print_fld_str(FLD_PROTO, p->p_name);
		else
			print_fld_uint(FLD_PROTO, pr->proto);
	}

	if (pr->ifname[0]) {
		tb_start();
#if OS_LEVEL > 31
		if (pr->ifnot)
			tbprintf("!");
#endif
		tbprintf("%s", pr->ifname);
		print_fld_str(FLD_IF, tmp_buf);
	}

	tb_start();
	tb_print_fromto(&pr->src, &pr->dst, pr->af, pr->proto);
	print_fld_str(FLD_RINFO, tmp_buf);

	end_line();
}

void
print_rules(void)
{
	u_int32_t n, count = 0;
	
	for (n = dispstart; n < num_rules; n++) {
		print_rule(rules + n);
		count ++;
		if (maxprint > 0 && count >= maxprint)
			break;
	}
}

/* main program functions */

int
read_view(int dev)
{
	if (curr_mgr == NULL)
		return (0);

	if (curr_mgr->read_fn != NULL)
		return (curr_mgr->read_fn(dev));

	return (0);
}

void
sort_view(void)
{
	if (curr_mgr != NULL)
		if (curr_mgr->sort_fn != NULL)
			curr_mgr->sort_fn();
}

void
sig_close(int signal)
{
	gotsig_close = 1;
}

void
sig_resize(int signal)
{
	gotsig_resize = 1;
}

void
sig_alarm(int signal)
{
	gotsig_alarm = 1;
}

void
usage()
{
	extern char *__progname;
	fprintf(stderr, "usage: %s [-abhir] [-d cnt]", __progname);
	fprintf(stderr, " [-o field] [-s time] [-v view] [-w width] [num]\n");
	exit(1);
}

void
show_help(void)
{
	int line = 0;

	if (rawmode)
		return;

	erase();
	mvprintw(line, 2, "pfTop Help");
	line += 2;
	mvprintw(line++, 5, " h  - Help (this page)");
	mvprintw(line++, 5, " n  - Set number of lines");
	mvprintw(line++, 5, " o  - next sort Order");
	mvprintw(line++, 5, " r  - Reverse sort order");
	mvprintw(line++, 5, " s  - Set update interval");
	mvprintw(line++, 5, " v  - next View");
	mvprintw(line++, 5, " q  - Quit");
	line++;
	mvprintw(line++, 5, "0-6 - select view directly");
	mvprintw(line++, 5, "SPC - update immediately");
	mvprintw(line++, 5, "^L  - refresh display");
	line++;
	mvprintw(line++, 5, "cursor keys - scroll display");
	line++;
	mvprintw(line++, 3, "Sorting shortcuts:");
	line++;
	mvprintw(line,    5, " A  - Age");
	mvprintw(line,   25, " B  - Bytes");
	mvprintw(line++, 45, " D  - Dest. port");
	mvprintw(line,    5, " E  - Expiry");
	mvprintw(line,   25, " F  - From");
	mvprintw(line++, 45, " N  - None ");
	mvprintw(line,    5, " P  - Packets");
	mvprintw(line,   25, " S  - Src. port");
	mvprintw(line++, 45, " T  - To");
	line++;
	mvprintw(line++, 6, "press any key to continue ...");

	while (getch() == ERR);
}

void
setup_term(int maxstates)
{
	maxprint = maxstates;

	if (rawmode) {
		columns = rawwidth;
		lines = DEFAULT_HEIGHT;
		clear_linebuf();
	} else {
		if (maxstates < 0)
			maxstates = 0;

		screen = newterm(NULL, stdout, stdin);
		if (screen == NULL) {
			rawmode = 1;
			interactive = 0;
			setup_term(maxstates);
			return;
		}
		columns = COLS;
		lines = LINES;

		if (maxstates > lines - HEADER_LINES)
			maxprint = lines - HEADER_LINES;

		nonl();
		keypad(stdscr, TRUE);
		intrflush(stdscr, FALSE);

		halfdelay(10);
		noecho();
	}

	if (maxstates == 0)
		maxprint = lines - HEADER_LINES;

	field_setup();
}

struct command *
command_set(struct command *cmd)
{
	struct command *prev = curr_cmd;

	if (cmd) {
		cmd_len = 0;
		cmdbuf[0] = 0;
	}
	curr_cmd = cmd;
	need_update = 1;
	return prev;
}

void
print_cmdline(void)
{
	if (curr_cmd == NULL)
		return;

	attron(A_STANDOUT);
	mvprintw(HOME_Y, HOME_X, "%s: ", curr_cmd->prompt);
	attroff(A_STANDOUT);
	printw("%s", cmdbuf);
}

void
cmd_delay(void)
{
	int del;
	del = atoi(cmdbuf);
	if (del > 0) {
		delay = del;
		gotsig_alarm = 1;
	}
}

void
cmd_count(void)
{
	int ms;
	ms = atoi(cmdbuf);

	if (ms <= 0 || ms > lines - HEADER_LINES)
		maxprint = lines - HEADER_LINES;
	else
		maxprint = ms;
}

void
cmd_keyboard(int ch)
{
	if (curr_cmd == NULL)
		return;

	if (ch > 0 && isprint(ch)) {
		if (cmd_len < sizeof(cmdbuf) - 1) {
			cmdbuf[cmd_len++] = ch;
			cmdbuf[cmd_len] = 0;
			need_update = 1;
		} else
			beep();
	}
	
	switch (ch) {
	case KEY_ENTER:
	case 0x0a:
	case 0x0d:
	{
		struct command * c = command_set(NULL);
		c->exec();
		break;
	}
	case KEY_BACKSPACE:
	case KEY_DC:
	case CTRL_H:
		if (cmd_len > 0) {
			cmdbuf[--cmd_len] = 0;
			need_update = 1;
		} else
			beep();
	default:
	}
}

void
keyboard(void)
{
	int ch;

	ch = getch();

	if (curr_cmd) {
		cmd_keyboard(ch);
		return;
	}

	switch (ch) {
	case ' ':
		gotsig_alarm = 1;
		break;
	case '?':
		/* FALLTHROUGH */
	case 'h':
		show_help();
		need_update = 1;
		break;
	case 'n':
		command_set(&cm_count);
		break;
	case 'o':
		next_order();
		need_sort = 1;
		break;
	case 'q':
		gotsig_close = 1;
		break;
	case 'r':
		sortdir *= -1;
		need_sort = 1;
		break;
	case 's':
		command_set(&cm_delay);
		break;
	case 'v':
		/* FALLTHROUGH */
	case KEY_RIGHT:
		/* FALLTHROUGH */
	case CTRL_F:
		next_view();
		break;
	case KEY_LEFT:
		/* FALLTHROUGH */
	case CTRL_B:
		prev_view();
		break;
	case KEY_DOWN:
		/* FALLTHROUGH */
	case CTRL_N:
		dispstart++;
		need_update = 1;
		break;
	case KEY_UP:
		/* FALLTHROUGH */
	case CTRL_P:
		dispstart--;
		need_update = 1;
		break;
	case KEY_NPAGE:
		/* FALLTHROUGH */
	case CTRL_V:
		dispstart += maxprint;
		need_update = 1;
		break;
	case KEY_PPAGE:
		/* FALLTHROUGH */
	case META_V:
		dispstart -= maxprint;
		need_update = 1;
		break;
	case KEY_HOME:
		/* FALLTHROUGH */
	case CTRL_A:
		dispstart = 0;
		need_update = 1;
		break;
	case KEY_END:
		/* FALLTHROUGH */
	case CTRL_E:
		dispstart = num_disp;
		need_update = 1;
		break;
	case CTRL_L:
		clear();
		need_update = 1;
		break;
	default:
		break;
	}

	if (set_order_hotkey(ch))
		need_sort = 1;
	else
		set_view_hotkey(ch);
}

int
main(int argc, char *argv[])
{
	extern char *optarg;
	extern int optind;

	struct pf_status status;

	char *orderstr = NULL;
	char *viewstr = NULL;

	int countmax = 0;
	int count = 0;
	int maxstates = 0;

	int dev = -1;
	int ch;

	while ((ch = getopt(argc, argv, "abhirs:d:o:v:w:")) != -1) {
		switch (ch) {
		case 'a':
			maxstates = -1;
			break;
		case 'd':
			countmax = atoi(optarg);
			if (countmax < 0)
				countmax = 0;
			break;
		case 'i':
			interactive = 1;
			break;
		case 'b':
			rawmode = 1;
			interactive = 0;
			break;
		case 'o':
			orderstr = optarg;
			break;
		case 'r':
			sortdir *= -1;
			break;
		case 's':
			delay = atoi(optarg);
			if (delay < 1)
				delay = 1;
			break;
		case 'v':
			viewstr = optarg;
			break;
		case 'w':
			rawwidth = atoi(optarg);
			if (rawwidth < 1)
				rawwidth = DEFAULT_WIDTH;
			if (rawwidth > MAX_LINE_BUF)
				rawwidth = MAX_LINE_BUF;
			break;
		case 'h':
			/* FALLTHROUGH */
		default:
			usage();
			/* NOTREACHED */
		}
	}

	argc -= optind;
	argv += optind;

	if (argc == 1)
		maxstates = atoi(argv[0]);
	else if (argc > 1)
		usage();

	set_order(orderstr);
	set_view(viewstr);

	if (!isatty(STDOUT_FILENO)) {
		rawmode = 1;
		interactive = 0;
	}
	signal(SIGTERM, sig_close);
	signal(SIGINT, sig_close);
	signal(SIGQUIT, sig_close);
	signal(SIGWINCH, sig_resize);
	signal(SIGALRM, sig_alarm);

#if OS_LEVEL > 30
	dev = open("/dev/pf", O_RDONLY);
#else
	dev = open("/dev/pf", O_RDWR);
#endif
	if (dev == -1)
		err(1, "open(\"/dev/pf\")");

	/* preallocate existing states if possible */
	if (ioctl(dev, DIOCGETSTATUS, &status)) {
		warnx("DIOCGETSTATUS");
		alloc_buf(0);
	} else
		alloc_buf(status.states);

	setup_term(maxstates);

	if (rawmode && countmax == 0)
		countmax = 1;

	gotsig_alarm = 1;
	for (;;) {
		if (gotsig_alarm) {
			read_view(dev);
			need_sort = 1;
			gotsig_alarm = 0;
			alarm(delay);
		}

		if (need_sort) {
			sort_view();
			need_sort = 0;
			need_update = 1;
			
			/* XXX if sort took too long */
			if (gotsig_alarm) {
				gotsig_alarm = 0;
				alarm(delay);
			}
		}

		if (need_update) {
			erase();
			disp_update(dev);
			end_page();
			need_update = 0;
			if (countmax && ++count >= countmax)
				break;
		}

		if (gotsig_close)
			break;
		if (gotsig_resize) {
			if (rawmode == 0)
				endwin();
			setup_term(maxstates);
			gotsig_resize = 0;
			need_update = 1;
		}

		if (interactive && need_update == 0)
			keyboard();
		else if (interactive == 0)
			sleep(delay);
	}

	if (rawmode == 0)
		endwin();

	close(dev);
	return 0;
}
