/*	$OpenBSD: pfstat.c,v 1.1 2002/06/26 14:35:00 dhartmei Exp $ */

/*
 * Copyright (c) 2002, Daniel Hartmeier
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer. 
 *    - Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <net/pfvar.h>
#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <gd.h>
#include <gdfonts.h>

#include "pfstat.h"

const struct descriptor_type descriptor[DESC_SIZE] = {
	{ "bytes_v4_in"		, "bits/s"	, 1, 1 },
	{ "bytes_v4_out"	, "bits/s"	, 1, 1 },
	{ "bytes_v6_in"		, "bits/s"	, 1, 1 },
	{ "bytes_v6_out"	, "bits/s"	, 1, 1 },
	{ "packets_v4_in_pass"	, "packets/s"	, 1, 0 },
	{ "packets_v4_in_drop"	, "packets/s"	, 1, 0 },
	{ "packets_v4_out_pass"	, "packets/s"	, 1, 0 },
	{ "packets_v4_out_drop"	, "packets/s"	, 1, 0 },
	{ "packets_v6_in_pass"	, "packets/s"	, 1, 0 },
	{ "packets_v6_in_drop"	, "packets/s"	, 1, 0 },
	{ "packets_v6_out_pass"	, "packets/s"	, 1, 0 },
	{ "packets_v6_out_drop"	, "packets/s"	, 1, 0 },
	{ "states_entries"	, "entries"	, 0, 0 },
	{ "states_searches"	, "states/s"	, 1, 0 },
	{ "states_inserts"	, "states/s"	, 1, 0 },
	{ "states_removals"	, "states/s"	, 1, 0 },
	{ "counters_match"	, "per second"	, 1, 0 },
	{ "counters_badoffset"	, "per second"	, 1, 0 },
	{ "counters_fragment"	, "per second"	, 1, 0 },
	{ "counters_short"	, "per second"	, 1, 0 },
	{ "counters_normalize"	, "per second"	, 1, 0 },
	{ "counters_memory"	, "per second"	, 1, 0 } };

void	 usage(void);
int	 generate_data(void);
int	 read_file(const char *, struct matrix *);
int	 generate_images(struct matrix *);
double	 intersect(double, double, double, double);
void	 add(struct column *, unsigned, unsigned, unsigned, struct status *);
double	 max(double, double);
void	 find_max(struct matrix *);
void	 normalize(struct matrix *);
void	 draw(gdImagePtr, struct matrix *, unsigned, int, unsigned);
void	 draw_grid(gdImagePtr, struct matrix *);
int	 add_matrix(struct matrix **, const char *, unsigned, unsigned,
	    unsigned, unsigned);
int	 add_graph(struct graph **, const char *, const char *,
	    unsigned, int);

void
usage(void)
{
	extern char *__progname;

	fprintf(stderr, "usage: %s [-q] [-c config] [-d data]\n", __progname);
	exit(1);
}

int
main(int argc, char *argv[])
{
	int ch;
	const char *configopt = NULL;
	const char *dataopt = NULL;
	int query = 0;

	if (argc < 2)
		usage();

	while ((ch = getopt(argc, argv, "c:d:q")) != -1) {
		switch (ch) {
		case 'c':
			configopt = optarg;
			break;
		case 'd':
			dataopt = optarg;
			break;
		case 'q':
			query = 1;
			break;
		default:
			usage();
		}
	}

	if (argc != optind) {
		warnx("unknown command line argument: %s ...", argv[optind]);
		usage();
	}

	if (query)
		if (generate_data())
			return (1);

	if (configopt != NULL && dataopt != NULL) {
		struct matrix *matrices = NULL;

		if (parse_config(configopt, &matrices))
			return (1);
		if (read_file(dataopt, matrices))
			return (1);
		if (generate_images(matrices))
			return (1);
	}

	return (0);
}

int
generate_data(void)
{
	int dev, i;
	struct pf_status s;

	dev = open("/dev/pf", O_RDWR); /* O_RDONLY sufficient for 3.1 and later */
	if (dev == -1) {
		warn("/dev/pf");
		return (1);
	}
	if (ioctl(dev, DIOCGETSTATUS, &s)) {
		err(1, "ioctl(DIOCGETSTATUS)");
		close(dev);
		return (1);
	}
	close(dev);
	if (s.running) {
		printf("%u %u", time(NULL), s.since);
		printf(" %llu %llu %llu %llu",
		    s.bcounters[0][PF_IN],
		    s.bcounters[0][PF_OUT],
		    s.bcounters[1][PF_IN],
		    s.bcounters[1][PF_OUT]);
		printf(" %llu %llu %llu %llu %llu %llu %llu %llu",
		    s.pcounters[0][PF_IN][PF_PASS],
		    s.pcounters[0][PF_IN][PF_DROP],
		    s.pcounters[0][PF_OUT][PF_PASS],
		    s.pcounters[0][PF_OUT][PF_DROP],
		    s.pcounters[1][PF_IN][PF_PASS],
		    s.pcounters[1][PF_IN][PF_DROP],
		    s.pcounters[1][PF_OUT][PF_PASS],
		    s.pcounters[1][PF_OUT][PF_DROP]);
		printf(" %u", s.states);
		for (i = 0; i < FCNT_MAX; ++i)
			printf(" %lld", s.fcounters[i]);
		for (i = 0; i < PFRES_MAX; ++i)
			printf(" %lld", s.counters[i]);
		printf("\n");
	}
	return (0);
}

int
read_file(const char *name, struct matrix *matrices)
{
	struct matrix *matrix;
	char c;
	char line[1024];
	char *p;
	int i, pos;
	struct status s[2];
	FILE *file;

	printf("reading data file %s\n", name);
	file = fopen(name, "r");
	if (file == NULL) {
		warn("%s", name);
		return (1);
	}

	memset(&s, 0, 2 * sizeof(struct status));
	pos = 0;
	while (fread(&c, 1, 1, file)) {
		if (c == '\n') {
			line[pos] = 0;
			pos = 0;
			p = line;
			s[1].time = strtoul(p, &p, 10);
			s[1].since = strtoul(p, &p, 10);
			for (i = 0; i < DESC_SIZE; ++i)
				s[1].n[i] = strtouq(p, &p, 10);
			if (s[1].since == s[0].since && s[1].time > s[0].time)
				for (matrix = matrices; matrix;
				    matrix = matrix->next)
					add(matrix->columns, matrix->w0,
					    matrix->beg, matrix->end, s);
			memcpy(&s[0], &s[1], sizeof(struct status));
		} else {
			if (pos < sizeof(line)-1)
				line[pos++] = c;
			else
				err(1, "data line too long");
		}
	}

	fclose(file);
	return (0);
}

int
generate_images(struct matrix *matrices)
{
	struct matrix *matrix;
	const unsigned fw = gdFontSmall->w, fh = gdFontSmall->h;

	matrix = matrices;
	while (matrix != NULL) {
		gdImagePtr im;
		int white;
		FILE *out;
		struct matrix *old_matrix;
		struct graph *graph, *old_graph;
		unsigned i, x;

		printf("generating image file %s\n", matrix->filename);

		find_max(matrix);
		for (i = 0; i < 2; ++i) {
			double m = 0.0;

			for (graph = matrix->graphs[i]; graph;
			    graph = graph->next) {
				if (matrix->max.v[graph->desc_nr] > m)
					m = matrix->max.v[graph->desc_nr];
			}
			for (graph = matrix->graphs[i]; graph;
			    graph = graph->next)
				matrix->max.v[graph->desc_nr] = m;
			printf("  m[%i] == %f\n", i, m);
		}
		normalize(matrix);

		im = gdImageCreate(matrix->width, matrix->height);
		white = gdImageColorAllocate(im, 255, 255, 255);

		gdImageFilledRectangle(im, 0, 0, matrix->width-1,
		    matrix->height-1, white);

		draw_grid(im, matrix);

		for (i = 0; i < 2; ++i) {
			x = i == 0 ? matrix->x0 : matrix->x0+matrix->w0;
			for (graph = matrix->graphs[i]; graph;
			    graph = graph->next) {
				int color = gdImageColorAllocate(im,
				    (graph->color >> 16) & 0xFF,
				    (graph->color >> 8) & 0xFF,
				    graph->color & 0xFF);

				draw(im, matrix, graph->desc_nr,
				    color, graph->filled);
				if (i)
					x -= strlen(graph->label) * fw;
				gdImageString(im, gdFontSmall, x,
				    matrix->y0+matrix->h0+5+fh,
				    graph->label, color);
				if (!i)
					x += (strlen(graph->label) + 1) * fw;
				else
					x -= fw;
			}
		}

		out = fopen(matrix->filename, "wb");
		if (out == NULL) {
			warn("%s", matrix->filename);
			return (1);
		}
		gdImageJpeg(im, out, 95);
		fclose(out);
		gdImageDestroy(im);

		/* free matrix */
		graph = matrix->graphs[0];
		while (graph != NULL) {
			old_graph = graph;
			graph = graph->next;
			free(old_graph);
		}
		graph = matrix->graphs[1];
		while (graph != NULL) {
			old_graph = graph;
			graph = graph->next;
			free(old_graph);
		}
		free(matrix->columns);
		free(matrix->filename);
		old_matrix = matrix;
		matrix = matrix->next;
		free(old_matrix);
	}

	return (0);
}

inline double
intersect(double sa, double sb, double pa, double pb)
{
	/* sa <= sb && pa <= pb */
	if (sb <= pa || sa >= pb)
		return (0.0);
	if (sa <= pa && sb >= pb)
		return (pb - pa);
	if (sa >= pa && sb <= pb)
		return (sb - sa);
	if (sa < pa)
		return (sb - pa);
	else
		return (pb - sa);
}

void
add(struct column *columns, unsigned width, unsigned beg, unsigned end,
    struct status *status)
{
	double spp = (double)(end - beg) / (double)width;
	double sa, sb, pa, pb, f;
	unsigned int i, j;
	struct column c;

	/* sa < sb */
	sa = (double)status[0].time - (double)beg;
	sb = (double)status[1].time - (double)beg;
	for (j = 0; j < DESC_SIZE; ++j)
		c.v[j] = descriptor[j].cumulative ?
		    (double)(status[1].n[j] - status[0].n[j]) :
		    (double)status[1].n[j];
	for (i = 0; i < width; ++i) {
		pa = i * spp;
		pb = pa + spp;
		f = intersect(sa, sb, pa, pb);
		if (f != 0.0)
			for (j = 0; j < DESC_SIZE; ++j) {
				if (descriptor[j].cumulative)
					columns[i].v[j] += c.v[j] * f / (sb - sa);
				else
					columns[i].v[j] += c.v[j] * f;
			}
	}
}

inline double
max(double a, double b)
{
	if (a > b)
		return (a);
	else
		return (b);
}

void
find_max(struct matrix *m)
{
	unsigned i, j;

	for (i = 0; i < DESC_SIZE; ++i) {
		m->max.v[i] = 0.0;
		for (j = 0; j < m->w0; ++j)
			if (m->columns[j].v[i] > m->max.v[i])
				m->max.v[i] = m->columns[j].v[i];
	}
}

void
normalize(struct matrix *m)
{
	unsigned i, j;

	for (i = 0; i < DESC_SIZE; ++i)
		for (j = 0; j < m->w0; ++j)
			m->columns[j].v[i] /= m->max.v[i];
}

void
draw(gdImagePtr im, struct matrix *m, unsigned i, int color, unsigned filled)
{
	unsigned x = m->x0, y = m->y0, w = m->w0, h = m->h0;
	unsigned dx, dy0 = 0;

	for (dx = 0; dx < w; ++dx) {
		unsigned dy = m->columns[dx].v[i] * h;

		if (filled)
			gdImageLine(im, x+dx+1, y+h-1, x+dx+1, y+h-1-dy,
			    color);
		else if (dx > 0)
			gdImageLine(im, x+dx, y+h-1-dy0, x+dx+1, y+h-1-dy,
			    color);
		dy0 = dy;
	}
}

void
scale_unit(double *m, char *k, int bytes)
{
	*k = ' ';
	if (bytes)
		*m *= 8.0;
	if (*m >= 1000) {
		*m /= bytes ? 1024 : 1000;
		*k = 'k';
	}
	if (*m >= 1000) {
		*m /= bytes ? 1024 : 1000;
		*k = 'm';
	}
	if (*m >= 1000) {
		*m /= bytes ? 1024 : 1000;
		*k = 'g';
	}
	if (*m >= 1000) {
		*m /= bytes ? 1024 : 1000;
		*k = 't';
	}
}

void
draw_grid(gdImagePtr im, struct matrix *matrix)
{
	const unsigned fw = gdFontSmall->w, fh = gdFontSmall->h;
	unsigned x0 = matrix->x0, y0 = matrix->y0,
	    w0 = matrix->w0, h0 = matrix->h0;
	unsigned len = matrix->end - matrix->beg;
	double spp = (double)(matrix->end - matrix->beg) / (double)matrix->w0;
	double max[2] = { 0.0, 0.0 };
	const char *unit[2] = { NULL, NULL };
	unsigned i, ii;
	char k[2];
	double dx;
	char *t;
	int black = gdImageColorAllocate(im,   0,   0,   0);
	int grey = gdImageColorAllocate(im, 220, 220, 220);

	if (matrix->graphs[0] != NULL) {
		max[0] = matrix->max.v[matrix->graphs[0]->desc_nr] / spp,
		scale_unit(&max[0], &k[0],
		    descriptor[matrix->graphs[0]->desc_nr].bytes);
		unit[0] = descriptor[matrix->graphs[0]->desc_nr].unit;
	}
	if (matrix->graphs[1] != NULL) {
		max[1] = matrix->max.v[matrix->graphs[1]->desc_nr] / spp,
		scale_unit(&max[1], &k[1],
		    descriptor[matrix->graphs[1]->desc_nr].bytes);
		unit[1] = descriptor[matrix->graphs[1]->desc_nr].unit;
	}

	/* bounding box */
	gdImageLine(im, x0, y0+h0, x0+w0, y0+h0, black);
	gdImageLine(im, x0, y0, x0, y0+h0, black);
	gdImageLine(im, x0+w0, y0, x0+w0, y0+h0, black);

	/* horizontal units */
	ii = h0 / fh;
	if (ii > 10)
		ii = 10;
	for (i = 0; i <= ii; ++i) {
		char t[10];
		unsigned y1 = y0+(i*h0)/ii;

		gdImageLine(im, x0-2, y1, x0, y1, black);
		gdImageLine(im, x0+w0, y1, x0+w0+2, y1, black);
		if (i < ii)
			gdImageLine(im, x0+1, y1, x0+w0-1, y1, grey);
		if (matrix->graphs[0] != NULL) {
			sprintf(t, "%5.1f %c", (max[0]*(ii-i))/ii, k[0]);
			gdImageString(im, gdFontSmall, fh+fw,
			    y1-fh/2, t, black);
		}
		if (matrix->graphs[1] != NULL) {
			sprintf(t, "%5.1f %c", (max[1]*(ii-i))/ii, k[1]);
			gdImageString(im, gdFontSmall, x0+w0+1*fw,
			    y1-fh/2, t, black);
		}
	}

	/* time label */
	t = "minutes";
	dx = (60.0 * (double)w0) / (double)len; // pixels per unit
	if (dx < 4*fw) {
		dx *= 60.0; t = "hours";
		if (dx < 4*fw) {
			dx *= 24.0; t = "days";
			if (dx < 4*fw) {
				dx *= 7.0; t = "weeks";
				if (dx < 4*fw) {
					dx *= 4; t = "months";
				}
			}
		}
	}
	/* time units */
	for (i = 0; (i*dx) <= w0; ++i) {
		unsigned x1 = x0+w0-(i*dx);
		char t[64];

		gdImageLine(im, x1, y0+h0, x1, y0+h0+2, black);
		if (x1 > x0 && x1 < x0+w0)
			gdImageLine(im, x1, y0+h0, x1, y0, grey);
		if (x1 < x0+w0) {
			sprintf(t, "-%u", i);
			gdImageString(im, gdFontSmall, x1-(strlen(t)*fw)/2,
			    y0+h0+5, t, black);
		} else {
			time_t tt = time(0);
			strncpy(t, asctime(localtime(&tt)), 24); t[24] = 0;
			gdImageString(im, gdFontSmall, x0+(w0-strlen(t)*fw)/2,
			    y0+h0+5+2*fh, t, black);
		}
	}
	/* hours/days/weeks/months */
	gdImageString(im, gdFontSmall, x0+w0-strlen(t)*fw, y0+h0+5, t, black);
	if (matrix->graphs[0] != NULL)
		gdImageStringUp(im, gdFontSmall, 0,
		    y0+h0-(h0-strlen(unit[0])*fw)/2, (char *)unit[0], black);
	if (matrix->graphs[1] != NULL)
		gdImageStringUp(im, gdFontSmall, x0+w0+8*fw,
		    y0+h0-(h0-strlen(unit[1])*fw)/2, (char *)unit[1], black);
}

int
add_matrix(struct matrix **matrices, const char *filename, unsigned width,
    unsigned height, unsigned beg, unsigned end)
{
	struct matrix *m;
	const unsigned fw = gdFontSmall->w, fh = gdFontSmall->h;

	m = malloc(sizeof(struct matrix));
	if (m == NULL)
		err(1, "malloc");
	m->filename = strdup(filename);
	if (m->filename == NULL)
		err(1, "strdup");
	m->beg = beg;
	m->end = end;
	m->width = width;
	m->height = height;
	m->w0 = width-fh-18*fw-2*fw;
	m->h0 = height-5*fh;
	m->x0 = fh+9*fw;
	m->y0 = fh;
	m->columns = malloc(m->w0 * sizeof(struct column));
	if (m->columns == NULL)
		err(1, "malloc");
	memset(m->columns, 0, m->w0 * sizeof(struct column));
	m->graphs[0] = m->graphs[1] = NULL;
	if (*matrices == NULL)
		m->next = NULL;
	else
		m->next = *matrices;
	*matrices = m;
	return (0);
}

int
add_graph(struct graph **graphs, const char *desc_name, const char *label,
    unsigned color, int filled)
{
	unsigned i;
	struct graph *g;

	for (i = 0; i < DESC_SIZE; ++i)
		if (!strcmp(descriptor[i].name, desc_name))
			break;
	if (i == DESC_SIZE) {
		fprintf(stderr, "invalid graph name %s\n", desc_name);
		return (1);
	}
	g = malloc(sizeof(struct graph));
	if (g == NULL)
		err(1, "malloc");
	g->desc_nr = i;
	g->label = strdup(label);
	if (g->label == NULL)
		err(1, "strdup");
	g->color = color;
	g->filled = filled;
	if (*graphs == NULL)
		g->next = NULL;
	else
		g->next = *graphs;
	*graphs = g;
	return (0);
}
