
/*
 * Copyright (c) 2005, Arnaud KLEIN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */ 
 
#include "XmlFileParser.h"
#include "StrTools.h"
#include "SysTools.h"
#include "pflogx.h"
#include <fcntl.h>
#include <unistd.h>

		
/* Constructor */
cXmlFileParser::cXmlFileParser(const string& file, const cLogEntryParser& entryParser) : m_file(file), m_entryParser(entryParser), m_vectLogEntries(), m_inputFileDesc(-1), m_xmlParser(NULL), m_xmlElementType(XMLELEMENT_UNKNOWN), m_errorString()
{
}

	
/* Parse XML file */
enum cXmlFileParser::eParseRet cXmlFileParser::Parse()
{
	enum eParseRet eRet=PARSE_NO_ERROR;

	m_vectLogEntries.clear();	
	m_xmlElementType=XMLELEMENT_UNKNOWN;

	// Open input file
	if (OpenFile())
	{
		// Initialize XML parser
		if (InitXmlParser())
		{
			// Parse file
			eRet=ParseFile();
		
			ReleaseXmlParser();
		}
		else
			eRet=PARSE_XMLPARSER_INIT_ERROR;
			
		CloseFile();
	}
	else
		eRet=PARSE_INPUTFILE_OPEN_ERROR;


	return eRet;
}


/* Parse opened input file */
enum cXmlFileParser::eParseRet cXmlFileParser::ParseFile()
{
#define BUFFER_SIZE 4096

	bool bEof=false;
	char buffer[BUFFER_SIZE];
	ssize_t nReadCount=0;
	enum eParseRet eRet=PARSE_NO_ERROR;


	while((eRet == PARSE_NO_ERROR) && !bEof)
	{
		// Read buffer from input file
		nReadCount=read(m_inputFileDesc, buffer, sizeof(buffer));

		if (nReadCount == -1)
		{
			m_errorString="Read error (" + cSysTools::GetLastSystemError() + ")";
			eRet=PARSE_INPUTFILE_READ_ERROR;
		}
		else
		{
			// EOF reached ?
			bEof=(nReadCount == 0);
			
			// Parse content of buffer
			if (XML_Parse(m_xmlParser, buffer, static_cast<int>(nReadCount), bEof) == XML_STATUS_OK)
			{
				// Ensure that input file format is valid
				if (m_xmlElementType == XMLELEMENT_ERROR)
					eRet=PARSE_INPUTFILE_INVALID;
			}
			else
			{
				// Retrieve error string
				const XML_LChar* xmlErrorString=XML_ErrorString(XML_GetErrorCode(m_xmlParser)); 
			
				// Retrieve line number
				string lineNumber;
				if (!cStrTools::IntToString(XML_GetCurrentLineNumber(m_xmlParser), lineNumber))
					lineNumber="?";
			
				m_errorString="Parse error at line " + lineNumber + " (" + string(xmlErrorString) + ")";
				eRet=PARSE_XMLPARSER_PARSE_ERROR;
			}
		}
	}
	
	return eRet;

#undef BUFFER_SIZE
}


/* Open input file */
bool cXmlFileParser::OpenFile()
{
	// Open file for reading
	m_inputFileDesc=open(m_file.c_str(), O_RDONLY, 0);
	
	if (m_inputFileDesc == -1)
	{
		m_errorString="Could not open input file (" + cSysTools::GetLastSystemError() + ")";
		return false;
	}
	
	return true;
}
	

/* Close input file */
bool cXmlFileParser::CloseFile()
{
	if (m_inputFileDesc == -1)
	{
		m_errorString="Input file not opened";
		return false;
	}
		
	close(m_inputFileDesc);
	m_inputFileDesc=-1;

	return true;
}


/* Initialize XML parser */
bool cXmlFileParser::InitXmlParser()
{
	if (m_xmlParser)
	{
		m_errorString="XML parser already initialized";
		return false;
	}
	
	m_xmlParser=XML_ParserCreate(XMLFILE_ENCODING);
	if (!m_xmlParser)
	{
		m_errorString="Could not initialize XML parser";
		return false;
	}

	// Set elements handler and user data	
	XML_SetElementHandler(m_xmlParser, XmlStartElementHandler, XmlEndElementHandler);
	XML_SetUserData(m_xmlParser, this);
	return true;
}


/* Release XML parser */
void cXmlFileParser::ReleaseXmlParser()
{
	if (m_xmlParser)
	{
		XML_ParserFree(m_xmlParser);
		m_xmlParser=NULL;
	}
}


/* XML parser handler: start element */
void cXmlFileParser::XmlStartElementHandler(const XML_Char *name, const XML_Char **atts)
{
	string strName(name);
	bool bElementError=false;
	
	switch(m_xmlElementType)
	{
		case XMLELEMENT_UNKNOWN:
		{
			// Unknown -> "pflogx"
			if (strName == "pflogx")
				m_xmlElementType=XMLELEMENT_PFLOGX;
			else
				bElementError=true;
		
			break;
		}	
		
		case XMLELEMENT_PFLOGX:
		{
			// "pflogx" -> "logs"
			if (strName == "logs")
				m_xmlElementType=XMLELEMENT_LOGS;
			else
				bElementError=true;

			break;
		}	

		case XMLELEMENT_LOGS:
		{
			// "logs" -> "log"
			if (strName == "log")
			{
				m_xmlElementType=XMLELEMENT_LOG;
				
				// Parse attributes of element
				XmlParseLogElement(atts);
			}
			else
				bElementError=true;

			break;
		}	
		
		case XMLELEMENT_LOG:
		{
			// "log" has no children
			bElementError=true;
			break;
		}
		
		case XMLELEMENT_ERROR:
		{
			// Error mode: do nothing
			break;
		}
	}

	
	if (bElementError)
	{
		// Retrieve line number
		string lineNumber;
		if (!cStrTools::IntToString(XML_GetCurrentLineNumber(m_xmlParser), lineNumber))
			lineNumber="?";

		// Error message
		m_errorString="Invalid token \"" + strName + "\" at line " + lineNumber;
		
		m_xmlElementType=XMLELEMENT_ERROR;	
	} 
}


/* XML parser handler: end element */
void cXmlFileParser::XmlEndElementHandler(const XML_Char *name)
{
	switch(m_xmlElementType)
	{
		case XMLELEMENT_LOG:
		{
			// "log" -> "logs"
			m_xmlElementType=XMLELEMENT_LOGS;
			break;
		}

		case XMLELEMENT_LOGS:
		{
			// "logs" -> "pflogx"
			m_xmlElementType=XMLELEMENT_PFLOGX;
			break;
		}	

		case XMLELEMENT_PFLOGX:
		{
			// "pflogx" -> Unknown
			m_xmlElementType=XMLELEMENT_UNKNOWN;
			break;
		}	

		case XMLELEMENT_UNKNOWN:
		{
			// Unknown has no parent
			m_xmlElementType=XMLELEMENT_ERROR;
			break;
		}
		
		case XMLELEMENT_ERROR:
		{
			// Error mode: do nothing
			break;
		}
	}
}


/* Parse XML element "log" */
bool cXmlFileParser::XmlParseLogElement(const XML_Char **atts)
{
	bool bHasError=false;
	const XML_Char** ptr_atts=atts;
	string time, ifName, ruleNumber, action, direction, protocol, destAddress, srcAddress, destPort, srcPort;	

	// Browse list of attributes
	while ((*ptr_atts) && !bHasError)
	{
		// Name
		string attrName=*ptr_atts;
		ptr_atts++;
	
		// Store value
		if (attrName == "date")
			time=*ptr_atts;
		else if (attrName == "if")
			ifName=*ptr_atts;
		else if (attrName == "action")
			action=*ptr_atts;
		else if (attrName == "rule")
			ruleNumber=*ptr_atts;
		else if (attrName == "direction")
			direction=*ptr_atts;
		else if (attrName == "protocol")
			protocol=*ptr_atts;
		else if (attrName == "src_adr")
			srcAddress=*ptr_atts;
		else if (attrName == "src_port")
			srcPort=*ptr_atts;
		else if (attrName == "dest_adr")
			destAddress=*ptr_atts;
		else if (attrName == "dest_port")
			destPort=*ptr_atts;
		else
		{
			// Unknown attribute
			bHasError=true;
		}
		
		ptr_atts++;	
	}

	
	// Parse strings
	if (m_entryParser.ParseStrings(time, ifName, ruleNumber, action, direction, protocol, destAddress, srcAddress, destPort, srcPort) == cLogEntryParser::PARSESTRING_NO_ERROR)
	{
		// Append entry to vector	
		m_vectLogEntries.push_back(m_entryParser.GetEntry());
	}
	else
		bHasError=true;

	return !bHasError;
}


/* XML parser static handler: start element */
void cXmlFileParser::XmlStartElementHandler(void *userData, const XML_Char *name, const XML_Char **atts)
{
	if (userData)
		reinterpret_cast<cXmlFileParser*>(userData)->XmlStartElementHandler(name, atts);
}


/* XML parser static handler: end element */
void cXmlFileParser::XmlEndElementHandler(void *userData, const XML_Char *name)
{
	if (userData)
		reinterpret_cast<cXmlFileParser*>(userData)->XmlEndElementHandler(name);
}
