
/*
 * Copyright (c) 2005, Arnaud KLEIN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */ 


#include "LogParser.h"
#include <sys/types.h>


/* Constructor */
cLogParser::cLogParser() : m_pcapError()
{
}


/* Parse specified log file
 * <filename>
 * <vectEntries> will contain entries
 * <vectErrors> will contain erroneous packets
 * If returns <false>, call GetPcapErrorMessage() method to retrieve the libpcap error message
 */
bool cLogParser::ParseLog(const string& filename, vector<cLogEntry>& vectEntries, vector<pair<int, enum eParsePacketError> >& vectErrors)
{
	char bufError[PCAP_ERRBUF_SIZE];
	int packetCounter=0;
	const u_char* pData=NULL;
	struct pcap_pkthdr pHeader;
		
	m_pcapError.erase();
	
	// Open file for reading
	pcap_t* infile=pcap_open_offline(filename.c_str(), bufError);
	
	if (!infile)
	{
		m_pcapError=bufError;
		return false;
	}
	
	// Browse packets
	while((pData=pcap_next(infile, &pHeader)) != NULL)
	{
		cLogEntry entry;
		
		// Parse packet
		enum eParsePacketError eRet=ParsePacket(pHeader, pData, entry);
		
		if (eRet == PACKET_NOERROR)
			vectEntries.push_back(entry);
		else
			vectErrors.push_back(pair<int, enum eParsePacketError>(packetCounter + 1, eRet));
		
		packetCounter++;
	}
	
	pcap_close(infile);
	return true;
}

	
/* Get error message corresponding to a value of eParsePacketError enumeration */
const string cLogParser::GetParsePacketErrorMessage(enum eParsePacketError e) const
{
	string errorMessage;

	switch(e)
	{
		case PACKET_NOERROR:
		{
			errorMessage="No error";
			break;
		}
		
		case PACKET_INVALID_LEN:
		{
			errorMessage="Invalid packet length";
			break;
		}
		
		case PACKET_INVALID_PFLOG_HDRLEN:
		{
			errorMessage="Invalid PFLOG header length";
			break;
		}
		
		case PACKET_INVALID_IPVERSION:
		{
			errorMessage="Incompatible IP protocol version";	
			break;
		}

		case PACKET_INVALID_IP_HDRLEN:
		{
			errorMessage="Invalid IP header length";
			break;
		}
		
		case PACKET_INVALID_TCP_HDRLEN:
		{
			errorMessage="Invalid TCP header length";
			break;
		}
		
		case PACKET_INVALID_UDP_HDRLEN:
		{
			errorMessage="Invalid UDP header length";
			break;
		}
		
		case PACKET_PARSER_INTERNAL_ERROR:
		{
			errorMessage="Internal error";
			break;
		}
	}

	return errorMessage;
}


/* Extract information from a packet */
enum cLogParser::eParsePacketError cLogParser::ParsePacket(const struct pcap_pkthdr& pHeader, const u_char* pData, cLogEntry& entry)
{
	// Get all headers of packet
	const struct ip* ipHeader=NULL;
	const struct pfloghdr* pfHeader=NULL;
	const struct tcphdr* tcpHeader=NULL;
	const struct udphdr* udpHeader=NULL;
	unsigned int pfHeader_len=PFLOG_HDRLEN, ipHeader_len=0;


	/* 
	 * Check pflog packet header
	 */	 

	// Check that packet can contain a pfloghdr structure
	if (pHeader.caplen < PFLOG_HDRLEN)
		return PACKET_INVALID_LEN;

	// Get it
	pfHeader=reinterpret_cast<const struct pfloghdr*>(pData);

	// Check length	
	if (pfHeader->length != PFLOG_REAL_HDRLEN)
		return PACKET_INVALID_PFLOG_HDRLEN;

	// Check address-family
	if (pfHeader->af != AF_INET)
		return PACKET_INVALID_IPVERSION;


	/* 
	 * Check IP packet header
	 */	 

	// Check if packet contains an IP header
	if (pHeader.caplen >= (pfHeader_len + sizeof(struct ip)))
	{
		// Get it
		ipHeader=reinterpret_cast<const struct ip*>(pData + pfHeader_len);
	
		// Check IP protocol version
		if (ipHeader->ip_v != IPVERSION)
			return PACKET_INVALID_IPVERSION;
	
		// Get IP header length
		ipHeader_len=ipHeader->ip_hl * 4;	
	
		if (pHeader.caplen < (pfHeader_len + ipHeader_len))
			return PACKET_INVALID_IP_HDRLEN;
	
		
		/* 
		 * Check TCP / UDP packet header
		 */				
		if (ipHeader->ip_p == IPPROTO_TCP)
		{
			// Check if packet contains a TCP header
			if (pHeader.caplen < (pfHeader_len + ipHeader_len + sizeof(struct tcphdr)))
				return PACKET_INVALID_TCP_HDRLEN;	
			
			// Get it
			tcpHeader=reinterpret_cast<const struct tcphdr*>(pData + pfHeader_len + ipHeader_len);
		}
		else if (ipHeader->ip_p == IPPROTO_UDP)
		{
			// Check if packet contains an UDP header
			if (pHeader.caplen < (pfHeader_len + ipHeader_len + sizeof(struct udphdr)))
				return PACKET_INVALID_UDP_HDRLEN;	
			
			// Get it
			udpHeader=reinterpret_cast<const struct udphdr*>(pData + pfHeader_len + ipHeader_len);
		}
	}

	// Feed entry
	entry.SetTime(pHeader.ts.tv_sec, pHeader.ts.tv_usec);
	return (FeedEntry(pfHeader, ipHeader, tcpHeader, udpHeader, entry) ? PACKET_NOERROR : PACKET_PARSER_INTERNAL_ERROR);
}


/* Feed an entry with information extracted from pflog, IP, TCP/UDP headers
 * <pfHeader> can't be NULL.
 * <ipHeader> can be NULL. In this case, neither <tcpHeader> or <udpHeader> is used.
 * <tcpHeader> and <udpHeader> can be both NULL.
 */
bool cLogParser::FeedEntry(const struct pfloghdr* pfHeader, const struct ip* ipHeader, const struct tcphdr* tcpHeader, const struct udphdr* udpHeader, cLogEntry& entry)
{

	/* 
	 * Store information from pflog packet header
	 */	 

	if (!pfHeader)
		return false;


	if (pfHeader->ifname)
		entry.SetIfName(pfHeader->ifname);

	entry.SetRuleNumber(ntohl(pfHeader->rulenr));
	entry.SetAction(pfHeader->action);
	entry.SetDirection(pfHeader->dir);
	entry.SetEntryDataFlags(entry.GetEntryDataFlags() | cLogEntry::ENTRYDATA_PFLOG);


	/* 
	 * Store information from IP packet header
	 */	 

	if (!ipHeader)
		return true;

	entry.SetProtocol(ipHeader->ip_p);
	entry.SetDestAddress(ipHeader->ip_dst);
	entry.SetSrcAddress(ipHeader->ip_src);
	entry.SetEntryDataFlags(entry.GetEntryDataFlags() | cLogEntry::ENTRYDATA_IP);
	

	/* 
	 * Store information from TCP / UDP packet header
	 */	 	

	if (ipHeader->ip_p == IPPROTO_TCP && tcpHeader)
	{
		entry.SetDestPort(ntohs(tcpHeader->th_dport));
		entry.SetSrcPort(ntohs(tcpHeader->th_sport));
		entry.SetEntryDataFlags(entry.GetEntryDataFlags() | cLogEntry::ENTRYDATA_TCP);
	}
	else if (ipHeader->ip_p == IPPROTO_UDP && udpHeader)
	{
		entry.SetDestPort(ntohs(udpHeader->uh_dport));
		entry.SetSrcPort(ntohs(udpHeader->uh_sport));
		entry.SetEntryDataFlags(entry.GetEntryDataFlags() | cLogEntry::ENTRYDATA_UDP);
	}

	return true;
}

