
/*
 * Copyright (c) 2005, Arnaud KLEIN
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */ 


#include "CmdLineParser.h"
#include "LogEntry.h"
#include "StrTools.h"
#include <unistd.h>
#include <vector>


/* Static members */
map<string, u_int8_t> cCmdLineParser::s_mapArgAction;
map<string, u_int8_t> cCmdLineParser::s_mapArgDirection;
map<string, u_int8_t> cCmdLineParser::s_mapArgProtocol;


/* Constructor */
cCmdLineParser::cCmdLineParser(int argc, char** argv) : m_argc(argc), m_argv(argv), m_options(), m_errorString()
{
}


/* Parse */
enum cCmdLineParser::eParseRet cCmdLineParser::Parse()
{
	int ch=0;
	enum eParseRet eRet=PARSE_NO_ERROR;

	// Generate options string
	string optString=GenerateOptionsString();
	
	// Disable display of errors to stderr
	opterr=0;

	while (((ch = getopt(m_argc, m_argv, optString.c_str())) != -1) && (eRet == PARSE_NO_ERROR))
	{
		switch(ch)
		{
			case 'i':
			{
				eRet=ParseArgInputFile(optarg) ? PARSE_NO_ERROR : PARSE_ERROR;
				break;
			}

			case 'o':
			{
				eRet=ParseArgOutputFile(optarg) ? PARSE_NO_ERROR : PARSE_ERROR;
				break;
			}		

#ifdef WITH_EXPAT
			case 'm':
			{
				m_options.SetMergeEvents(true);
				eRet=PARSE_NO_ERROR;
				break;
			}
#endif

			case 'p':
			{
				eRet=ParseArgProtocol(optarg) ? PARSE_NO_ERROR : PARSE_ERROR;
				break;
			}		

			case 'a':
			{
				eRet=ParseArgAction(optarg) ? PARSE_NO_ERROR : PARSE_ERROR;
				break;
			}

			case 'd':
			{
				eRet=ParseArgDirection(optarg) ? PARSE_NO_ERROR : PARSE_ERROR;
				break;
			}
			
			case 'n':
			{
				eRet=ParseArgInterface(optarg) ? PARSE_NO_ERROR : PARSE_ERROR;
				break;
			}
			
			case 't':
			{
				eRet=ParseArgOutDateFormat(optarg) ? PARSE_NO_ERROR : PARSE_ERROR;
				break;
			}

			case 'e':
			{
				m_options.SetDisplayErrorsPackets(true);
				eRet=PARSE_NO_ERROR;
				break;
			}
		
			case 'r':
			{
				m_options.SetReverseSort(true);
				eRet=PARSE_NO_ERROR;
				break;
			}	
			
			case 'h':
			{
				eRet=PARSE_HELP_DISPLAY;
				break;
			}
			
			case 'v':
			{
				eRet=PARSE_VERSION_DISPLAY;
				break;
			}
		
			case '?':
			{
				m_errorString="Unknown option -" + string(1, static_cast<char>(optopt));
				eRet=PARSE_ERROR;
				break;
			}
			
			case ':':
			{
				m_errorString="Missing argument for option -" + string(1, static_cast<char>(optopt));
				eRet=PARSE_ERROR;
				break;
			}
			
			default:
			{
				m_errorString="Unknown error";
				eRet=PARSE_ERROR;
				break;
			}
		}
	}
	
	return eRet;
}	


/** Generate options string */
string cCmdLineParser::GenerateOptionsString() const
{
	string optionsString;

#ifdef WITH_EXPAT
	optionsString=":i:o:mp:a:d:n:t:rehv";
#else
	optionsString=":i:o:p:a:d:n:t:rehv";
#endif

	return optionsString;
}


/* Parse argument for input file option */
bool cCmdLineParser::ParseArgInputFile(const string& argument)
{
	string argumentTrimmed=cStrTools::Trim(argument);
	
	if (!argumentTrimmed.empty())
	{
		m_options.SetInputFile(argumentTrimmed);
		return true;	
	}
	else
	{
		m_errorString="Empty value for option -i";
		return false;	
	}
}


/* Parse argument for output file option */
bool cCmdLineParser::ParseArgOutputFile(const string& argument)
{
	string argumentTrimmed=cStrTools::Trim(argument);
	
	if (!argumentTrimmed.empty())
	{
		m_options.SetOutputFile(argumentTrimmed);
		return true;	
	}
	else
	{
		m_errorString="Empty value for option -o";
		return false;	
	}
}


/* Parse argument for action option */
bool cCmdLineParser::ParseArgAction(const string& argument)
{
	vector<u_int8_t> vectFilter;
	
	if (ParseArgFilter(argument, s_mapArgAction, 'a', vectFilter))
	{
		m_options.SetFilterAction(vectFilter);
		return true;
	}
	else
		return false;
}


/* Parse argument for direction option */
bool cCmdLineParser::ParseArgDirection(const string& argument)
{
	vector<u_int8_t> vectFilter;
	
	if (ParseArgFilter(argument, s_mapArgDirection, 'd', vectFilter))
	{
		m_options.SetFilterDirection(vectFilter);
		return true;
	}
	else
		return false;
}


/* Parse argument for protocol option */
bool cCmdLineParser::ParseArgProtocol(const string& argument)
{
	vector<u_int8_t> vectFilter;
	
	if (ParseArgFilter(argument, s_mapArgProtocol, 'p', vectFilter))
	{
		m_options.SetFilterProtocol(vectFilter);
		return true;
	}
	else
		return false;
}


/* Parse argument for interface option */
bool cCmdLineParser::ParseArgInterface(const string& argument)
{
	bool bFilterAll=false;
	vector<string> vectTokens;
	
	// Get tokens
	cStrTools::GetTokens(argument, vectTokens);

	// Check if "all" token is defined
	for(vector<string>::const_iterator it=vectTokens.begin(); it != vectTokens.end(); it++)
	{
		if (*it == "all")
		{
			bFilterAll=true;
			break;
		}
	}

	if (bFilterAll)
		vectTokens.clear();	
		
	
	m_options.SetFilterInterface(vectTokens);
	return true;
}


/* Parse argument for output date format */
bool cCmdLineParser::ParseArgOutDateFormat(const string& argument)
{
	bool bHasError=false;

	if (argument == "s")
		m_options.SetOutDateFormat(cOptions::DATEFORMAT_SECONDS);
	else if (argument == "h")
		m_options.SetOutDateFormat(cOptions::DATEFORMAT_HUMAN); 
	else
	{
		m_errorString="Invalid value '" + argument + "' for option -t";
		bHasError=true;	
	}	
	
	return !bHasError;
}


/* Parse a filter argument */
bool cCmdLineParser::ParseArgFilter(const string& argument, const map<string, u_int8_t>& mapArguments, int optopt, vector<u_int8_t>& vectFilter)
{
	bool bHasError=false, bFilterAll=false;
	vector<string> vectTokens;
	
	// Get tokens
	cStrTools::GetTokens(argument, vectTokens);
		
	// Validates them
	for(vector<string>::const_iterator it=vectTokens.begin(); (it != vectTokens.end()) && !bHasError; it++)
	{
		const string& value=*it;
		
		if (value == "all")
		{
			bFilterAll=true;
			break;
		}
		else
		{	
			map<string, u_int8_t>::const_iterator itMapping=mapArguments.find(*it);
		
			if (itMapping != mapArguments.end())
				vectFilter.push_back((*itMapping).second);
			else
			{
				m_errorString="Invalid value '" + value + "' for option -" + string(1, static_cast<char>(optopt));
				bHasError=true;	
			}
		}		
	}

	if (!bHasError && bFilterAll)
		vectFilter.clear();	

	return !bHasError;
} 


/* Initialize object */
void cCmdLineParser::Initialize()
{
	// Initialize mapping between external and internal values for filters
	
	// Action
	s_mapArgAction["pass"]=PF_PASS;
	s_mapArgAction["drop"]=PF_DROP;

	// Direction
	s_mapArgDirection["in"]=PF_IN;
	s_mapArgDirection["out"]=PF_OUT;
	s_mapArgDirection["in-out"]=PF_INOUT;

	// Protocol
	s_mapArgProtocol["icmp"]=IPPROTO_ICMP;
	s_mapArgProtocol["ip"]=IPPROTO_IP;
	s_mapArgProtocol["raw"]=IPPROTO_RAW;
	s_mapArgProtocol["udp"]=IPPROTO_UDP;
	s_mapArgProtocol["tcp"]=IPPROTO_TCP;
}

