/*
 * Copyright (c) 2004 Laurence Tratt
 *
 * http://tratt.net/laurie/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR OR HIS EMPLOYER BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS    OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * 0.1 (15 October 2004)
 *
 * First release.
 *
 *
 * 0.2 (16 July 2005)
 * 
 * Add -f flag. [Inspired by a patch from Steve Shockley].
 *
 * Add -i flag. 
 * 
 * New behaviour for AC power. Try and lower performance to MIN_PERF, but raise
 * it maximum as soon as the threshold is reached. Also, perfd is now quick
 * raising hw.setperf, but slow lowering it when AC is on.
 */


#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>
#include <err.h>
#include <errno.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/dkstat.h>
#include <time.h>
#include <sys/ioctl.h>
#include <machine/apmvar.h>


/* This value is the threshold CPU utilization has to reach to force perfd
   to increase system performance; if CPU utilization falls below this value
   then system performance is decreased. */

#define ACTIVITY_THRESHOLD 80

/* This is the value that hw.setperf is incremented when necessary. */

#define PERF_INCREMENT 15

/* This is the value that hw.setperf is decremented when necessary. */

#define PERF_DECREMENT 15

/* Minimum performance level that sysctl.hw is lowered too unless overridden
   by the -m option. */

#define MIN_PERF 20

/* Sleep intervals are in microseconds. */

#define SLEEP_INTERVAL_INC 10000
#define SLEEP_INTERVAL_DEC 25000

/* setperf can sometimes return an incorrect figure (e.g. after a machine has
   woken from hibernation, the hardware may put the processor in high power
   mode, but setperf may still be set to 0). Every n cycles we update setperf
   regardless of what value it claims to be. */

#define FORCE_SET_PERF_UPDATE 40

/* APM check is only exceuted every n cycles. */

#define APM_CHECK_INTERVAL 25


/* APM related */

#define PATH_APM_NORMAL "/dev/apm"

extern char *__progname;

void usage(void);



void
usage(void)
{
    fprintf(stderr, "usage: %s [-df] [-m <min perf>]\n", __progname);
    exit(1);
}



int
main(int argc, char *argv[])
{
    static int cp_time_mib[] = {CTL_KERN, KERN_CPTIME};
    static int hw_perf_mib[] = {CTL_HW, HW_SETPERF};
    long cp_time[CPUSTATES];
    size_t size;
    struct apm_power_info apm_info;
    long cp_idletime_old, diff;
	const char *errstr;
    int active, ch, debug, fd, ac_fullpower, max_perf, min_perf, new_perf,
		next_apm_check, next_forced_setperf, perf;
	useconds_t sleep_interval;

	debug = 0;
	ac_fullpower = 0;
	min_perf = MIN_PERF;

    while ((ch = getopt(argc, argv, "dfm:")) != -1) {
        switch(ch) {
            case 'd':
                debug = 1;
                break;
			case 'f':
				ac_fullpower = 1;
				break;
			case 'm':
				min_perf = strtonum(optarg, 0, 100, &errstr);
				if (errstr != NULL)
					err(1, "min perf value '%s': %s", optarg, errstr);
				break;
            case '?':
            default:
                usage();
        }
    }

    if (!debug)
        daemon(0, 1);

    size = sizeof(cp_time);
    if (sysctl(cp_time_mib, 2, &cp_time, &size, NULL, 0) < 0)
        err(1, "sysctl kern.cp_time failed");
    cp_idletime_old = cp_time[CP_IDLE];

    if ((fd = open(PATH_APM_NORMAL, O_RDONLY)) == -1)
        err(1, "opening /dev/apm failed");

    max_perf = 100;
    next_apm_check = 0; /* Force APM's state to be read on the first iteration */
    next_forced_setperf = FORCE_SET_PERF_UPDATE;
	sleep_interval = SLEEP_INTERVAL_INC;
    while (1) {
        usleep(sleep_interval);

        /* Checking the APM state is an expensive operation which we only do
           every APM_CHECK_INTERVAL cycles. */

        if (next_apm_check == 0) {
            if (ioctl(fd, APM_IOC_GETPOWER, &apm_info) == -1) {
                warn("ioctl apm failed");
                continue;
            }
            next_apm_check = APM_CHECK_INTERVAL;
            if (apm_info.ac_state == APM_AC_ON)
                max_perf = 100;
            else {
                /* This is a heuristic - it means that system performance can be
                   maintained at something near maximum until battery life gets
                   to approximately 40% when the maximum obtainable performance
                   starts to drop off in order to conserve battery life. */
                max_perf = (int) (log10(((double) apm_info.battery_life) / 10.0)
                                  * 1.15 * 100) + 10;
                if (max_perf > 100)
                    max_perf = 100;
                else if (max_perf < min_perf)
                    max_perf = min_perf;
            }
        }
        else
            next_apm_check -= 1;

        size = sizeof(int);
        if (sysctl(hw_perf_mib, 2, &perf, &size, NULL, 0) < 0) {
            warn("sysctl hw.setperf failed");
            continue;
        }

        new_perf = perf;
        if (apm_info.ac_state == APM_AC_ON && ac_fullpower == 1)
            /* If the mains power is connected, we make sure the machine is
               running at full power. */
            new_perf = 100;
        else if (apm_info.ac_state == APM_AC_BACKUP ||
                 apm_info.ac_state == APM_AC_UNKNOWN)
            /* If we're running off backup power, or if we don't know what state
               the power is in, dump performance down to the minimum. */
            new_perf = min_perf;
        else if (apm_info.battery_state == APM_BATT_LOW ||
                 apm_info.battery_state == APM_BATT_CRITICAL)
            /* If the mains power isn't connected, and the battery is in a
               low or critical state then reduce the performance to zero. */
            new_perf = min_perf;
        else {
            /*  Read the processor stats. */
            size = sizeof(cp_time);
            if (sysctl(cp_time_mib, 2, &cp_time, &size, NULL, 0) < 0) {
                warn("sysctl kern.cp_time failed");
                continue;
            }

            /* active is set to the percentage of processor time spent perfoming
               any kind of activity since the last time we polled. */

            diff = (cp_time[CP_IDLE] - cp_idletime_old);
            if (diff == 0)
                active = 100;
            else
                active = 100 - (((diff * 1000 + (diff / 2l)) / diff) / 10);
            cp_idletime_old = cp_time[CP_IDLE];

            if (apm_info.ac_state == APM_AC_ON) {
				if (active > ACTIVITY_THRESHOLD)
    	            new_perf = max_perf;
        	    else if (active < ACTIVITY_THRESHOLD)
            	    new_perf = perf - 1;
			}
			else {
				if (active > ACTIVITY_THRESHOLD)
    	            new_perf = perf + PERF_INCREMENT;
        	    else if (active < ACTIVITY_THRESHOLD)
            	    new_perf = perf - PERF_DECREMENT;
			}
        }

        /* Enforce the upper and lower bounds. */

        if (new_perf < min_perf)
            new_perf = min_perf;
        else if (new_perf > max_perf)
            new_perf = max_perf;

		if (new_perf < perf)
			sleep_interval = SLEEP_INTERVAL_DEC;
		else if (new_perf >= perf)
			sleep_interval = SLEEP_INTERVAL_INC;

        /* Because we can't always trust the value given to us for setperf,
           every FORCE_SET_PERF_UPDATE cycles we forcibly update its value,
           even if we don't think it's changed. */

        next_forced_setperf--;
        if (new_perf == perf && next_forced_setperf > 0) {
            continue;
        }
        else if (next_forced_setperf == 0)
            next_forced_setperf = FORCE_SET_PERF_UPDATE;

        if (debug)
            fprintf(stderr, "setting perf to %d from %d [max %d] sleep %dms\n",
                    new_perf, perf, max_perf, sleep_interval);
        size = sizeof(int);
        if (sysctl(hw_perf_mib, 2, &perf, &size, &new_perf, sizeof(int)) < 0) {
            warn("sysctl hw.setperf failed");
            continue;
        }
    }
}
