/* peksystray
 * Copyright (c) 2003, 2005 Mohammed Sameer.
 * Copyright (c) 2005 Eric Piel.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
/* Sun OpenWindows */
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define SYSTEM_TRAY_REQUEST_DOCK    0
#define SYSTEM_TRAY_BEGIN_MESSAGE   1
#define SYSTEM_TRAY_CANCEL_MESSAGE  2

#define _NET_SYSTEM_TRAY_ORIENTATION_HORZ 0
#define _NET_SYSTEM_TRAY_ORIENTATION_VERT 1

/*
 * XEMBED messages
 */
#define XEMBED_EMBEDDED_NOTIFY		0
#define XEMBED_WINDOW_ACTIVATE		1
#define XEMBED_WINDOW_DEACTIVATE	2
#define XEMBED_REQUEST_FOCUS		3
#define XEMBED_FOCUS_IN			4
#define XEMBED_FOCUS_OUT		5
#define XEMBED_FOCUS_NEXT		6
#define XEMBED_FOCUS_PREV		7
/* 8-9 were used for XEMBED_GRAB_KEY/XEMBED_UNGRAB_KEY */
#define XEMBED_MODALITY_ON		10
#define XEMBED_MODALITY_OFF		11
#define XEMBED_REGISTER_ACCELERATOR	12
#define XEMBED_UNREGISTER_ACCELERATOR	13
#define XEMBED_ACTIVATE_ACCELERATOR	14


/* Flags for _XEMBED_INFO */
#define XEMBED_MAPPED                   (1 << 0)

#ifdef DEBUG
#define dbg_printf(fmt,arg...) fprintf(stderr,fmt,##arg)
#else
#define dbg_printf(fmt,arg...) do { } while (0)
#endif

#define MAX_ICONS 16

Atom selection_atom, opcode_atom, data_atom;

int no_tray_icon = 0;		/* No, of embedded icons */

Window icon[MAX_ICONS];
/* for multiple */
Window dock[MAX_ICONS];
Window icons[MAX_ICONS];

int vertical = 0;
int square = 0;
int multiple = 0;

int icon_size = 16;
int border_width = 0;
int dock_width = 64; // with WindowMaker it might be possible to retrieve it dynamically
int dock_height;

Display *display = NULL;
char *display_name = NULL;

Window Root;
Window dockWindow;
Window iconWindow;

void
quit (char *err)
{
  fprintf (stderr, err);
  exit (1);
}

void
display_help ()
{
  fprintf (stdout, ""
	   "%s - version %s\n"
	   "Copyright 2003-2005, Mohammed Sameer <msameer@foolab.org>\n"
	   "Copyright 2005, Eric Piel <eric.piel (at) tremplin-utc.net>\n"
	   "\n"
	   "Usage: %s [OPTIONS]\n"
	   "\n"
	   "Options:\n"
	   " --help\t Display this help.\n"
	   " --version\t Display version number and exit.\n"
	   " --display DISPLAY\t The X display to connect to.\n"
	   " --icon-size SIZE\t Icon size. Default is 16.\n"
	   " --vertical\t Vertical layout. Default is horizontal.\n"
	   " --square\t Square shape, for use with NeXT-like enviroments\n"
	   " --border SIZE\t Border width. Default is 0.\n"
" --multiple\t Open each icon in a separate window. Default is no.\n"
	   "\n"
	   "", PACKAGE, VERSION, PACKAGE);
  exit (0);
}

void
display_version ()
{
  fprintf (stdout, "%s\n", VERSION);
  exit (0);
}

void
parse_cmd_line (int argc, char *argv[])
{
  int x;
  if (argc == 1)
    {
      return;
    }
  for (x = 1; x < argc; x++)
    {
      dbg_printf ("%s\n", argv[x]);
      if (!strcmp (argv[x], "--help"))
	{
	  display_help ();
	}
      else if (!strcmp (argv[x], "--version"))
	{
	  display_version ();
	}
      else if (!strcmp (argv[x], "--vertical"))
	{
	  vertical = 1;
	}
      else if (!strcmp (argv[x], "--square"))
	{
	  square = 1;
	}
      else if (!strcmp (argv[x], "--multiple"))
	{
	  multiple = 1;
	}
      else if (!strcmp (argv[x], "--display"))
	{
	  if (x + 1 == argc)
	    {
	      quit ("--display requires an argument");
	    }
	  else
	    {
	      display_name = argv[x + 1];
	      ++x;
	      continue;
	    }
	}
      else if (!strcmp (argv[x], "--icon-size"))
	{
	  if (x + 1 == argc)
	    {
	      quit ("--icon-size requires an argument");
	    }
	  else
	    {
	      icon_size = atoi (argv[x + 1]);
	      if (icon_size == 0)
		{
		  quit (" the icon size must be at least 1 pixel!");
		}
	      ++x;
	      continue;
	    }
	}
      else if (!strcmp (argv[x], "--border"))
	{
	  if (x + 1 == argc)
	    {
	      quit ("--border requires an argument");
	    }
	  else
	    {
	      border_width = atoi (argv[x + 1]);
	      ++x;
	      continue;
	    }
	}
      else
	{
	  fprintf (stderr, "Unknown command line argument: %s\n", argv[x]);
	}
    }
}

Window
add_one_icon (Window embed, int icon_nb)
{
	int x, y, icons_per_row;
	dock[icon_nb] = embed;

	/* compute location of the icon */
	if (vertical)
		icons_per_row = dock_height / icon_size;
	else
		icons_per_row = dock_width / icon_size;

	/* put at least one icon per row */
	if (icons_per_row == 0)
		icons_per_row = 1;
	x = (icon_nb % icons_per_row) * icon_size;
	y = (icon_nb / icons_per_row) * icon_size;

	if (vertical) {
		int tmp = x;
		x = y;
		y = tmp;
	}

	x += border_width;
	y += border_width;

	XReparentWindow (display, embed, iconWindow, x, y);
	return iconWindow;
}

Window
add_one_icon_multiple (Window embed, int icon_nb)
{
	XClassHint *class_hint;
	XWMHints *hints;
	class_hint = XAllocClassHint ();

	dock[icon_nb] =
	XCreateSimpleWindow (display, DefaultRootWindow (display), 0, 0,
			     icon_size, icon_size, border_width, border_width, 0);
	icons[icon_nb] =
	XCreateSimpleWindow (display, dock[icon_nb], 0, 0,
			     icon_size, icon_size, border_width, border_width, 0);

	if (class_hint == NULL)
		quit ("Failed to allocate class hint");

	class_hint->res_class = PACKAGE;
	class_hint->res_name = PACKAGE;

	XSetClassHint (display, dock[icon_nb], class_hint);
	XFree (class_hint);
	hints = XAllocWMHints ();
	if (hints == NULL)
		quit ("Failed to allocate hints");

	hints->flags = StateHint | WindowGroupHint | IconWindowHint;
	hints->initial_state = WithdrawnState;
	hints->icon_window = icons[icon_nb];
	hints->window_group = dock[icon_nb];

	XSetWMHints (display, dock[icon_nb], hints);
	XFree (hints);
	XSelectInput (display, dock[icon_nb], SubstructureNotifyMask);
	XSetWindowBackgroundPixmap (display, icons[icon_nb], ParentRelative);
	XMapRaised (display, dock[icon_nb]);
	XFlush (display);

	XReparentWindow (display, embed, icons[icon_nb], border_width, border_width);
	return icons[icon_nb];
}

void
add_tray_icon (Window embed)
{
  XEvent xevent;
  Window this_icon_win;
  int icon_nb;

  dbg_printf ("%s: %i\n", __FUNCTION__, (int) embed);

  /* Find first usable icon slot */
  for (icon_nb = 0; icon_nb < MAX_ICONS; icon_nb++) {
	  if (icon[icon_nb] == 0)
		  break;
  }
  /* too many icons asked, that should never happen here */
  if (icon_nb == MAX_ICONS)
    {
      return;
    }

  XSelectInput (display, embed, StructureNotifyMask | PropertyChangeMask);
  XWithdrawWindow (display, embed, 0);
  
  if (multiple)
     this_icon_win = add_one_icon_multiple(embed, icon_nb);
  else
     this_icon_win = add_one_icon(embed, icon_nb);

  XSync (display, False);
  XMapRaised (display, embed);
  xevent.xclient.window = embed;
  xevent.xclient.type = ClientMessage;
  xevent.xclient.message_type = XInternAtom (display, "_XEMBED", False);
  xevent.xclient.format = 32;
  xevent.xclient.data.l[0] = CurrentTime;
  xevent.xclient.data.l[1] = XEMBED_EMBEDDED_NOTIFY;
  xevent.xclient.data.l[2] = 0;
  xevent.xclient.data.l[3] = this_icon_win;
  xevent.xclient.data.l[4] = 0;
  XSendEvent (display, Root, False, NoEventMask, &xevent);

  icon[icon_nb] = embed;
  no_tray_icon++;
}

void
rm_tray_icon (Window win)
{
  int i;

  dbg_printf ("%s: %i\n", __FUNCTION__, (int) win);

  for (i = 0; i < MAX_ICONS; i++) {
     if (icon[i] == win) {
         if (multiple) {
             XDestroyWindow (display, dock[i]);
             XDestroyWindow (display, icons[i]);
             dock[i] = 0;
             icons[i] = 0;
         }
         icon[i] = 0;
         no_tray_icon--;
         break;
     }
  }
}

void
handle_event (XEvent ev)
{
  XClientMessageEvent *event = (XClientMessageEvent *) & ev;
  XWindowAttributes tray_icon_attr;
  long opcode = event->data.l[1];
  Window win = event->data.l[2];

  dbg_printf ("%s\n", __FUNCTION__);

  switch (opcode)
    {
    case SYSTEM_TRAY_REQUEST_DOCK:
      {
	if (no_tray_icon == MAX_ICONS)
	  {
	    //	    XReparentWindow (display, win, DefaultRootWindow(display), 0, 0);
	    dbg_printf ("full\n");
	    return;
	  }

	XGetWindowAttributes (display, win, &tray_icon_attr);
	dbg_printf ("Width: %i\t Height:%i\tBorder: %i\n",
		 tray_icon_attr.width, tray_icon_attr.height,
		 tray_icon_attr.border_width);

	XResizeWindow (display, win, icon_size, icon_size);

	add_tray_icon (win);
	return;
      }
    case SYSTEM_TRAY_BEGIN_MESSAGE:
      {
	// for ballon messages, could be nice to handle them too
	return;
      }
    case SYSTEM_TRAY_CANCEL_MESSAGE:
      {
	return;
      }
    default:
      break;
    }
}

void
handle_message_data (XEvent ev)
{
  dbg_printf ("%s\n", __FUNCTION__);
}

void
eventLoop (void)
{
  XEvent ev;

  while (1) {
    XNextEvent (display, &ev);
    switch (ev.type) {
    case ConfigureNotify:
      {
	XWindowAttributes attrib;
	dbg_printf ("ConfigureNotify\n");
	if (ev.xany.window == dockWindow) 
	  break;
	/* Force the size specified by the user */
	XGetWindowAttributes (display, ev.xproperty.window, &attrib);
	if (attrib.width != icon_size) 
	  XResizeWindow (display, ev.xproperty.window, icon_size, icon_size);
	break;
      }
    case DestroyNotify:
      {
	XDestroyWindowEvent *xev = (XDestroyWindowEvent *) &ev;
	dbg_printf ("DestroyNotify\n");
	if (xev->window == dockWindow)
	  return;
	else
	  rm_tray_icon (xev->window);
	break;
      }
    case ReparentNotify:
      dbg_printf ("ReparentNotify\n");
      break;
    case UnmapNotify:
      {
	XUnmapEvent *xev = (XUnmapEvent *) &ev;
	dbg_printf ("UnmapNotify\n");
	if (xev->window != dockWindow)
	  rm_tray_icon (xev->window);
	break;
      }
    case ClientMessage:
      if (ev.xclient.message_type == opcode_atom)
	handle_event (ev);
      else if (ev.xclient.message_type == data_atom)
	handle_message_data (ev);
      break;
    case SelectionClear:
      dbg_printf ("SelectionClear\n");
      if (XGetSelectionOwner (display, selection_atom) == dockWindow)
	XSetSelectionOwner (display, selection_atom, None, CurrentTime);
      return;
    default:
      //dbg_printf ("message=%i\n", ev.type);
      break;
    }
  }
}

int
main (int argc, char *argv[])
{
  XClassHint *classHint;
  XWMHints *hints;

  parse_cmd_line (argc, argv);

  if ((display = XOpenDisplay (display_name)) == NULL)
      quit ("Cannot open display\n");

  if (square)
    {
	dock_height = dock_width;
    }
  else if (vertical)
    {
      dock_height = dock_width;
      dock_width = icon_size;
	// _NET_SYSTEM_TRAY_ORIENTATION_VERT should also be put but I don't know how to!
	// I guess I'm the only creature out there using this, So if it's causing problems
	// Please ping me. MSameer.
    }
  else {
    dock_height = icon_size;
  }

  dockWindow = XCreateSimpleWindow (display, DefaultRootWindow (display), 0, 0,
			     dock_width, dock_height, border_width, border_width, 0);
  iconWindow = XCreateSimpleWindow (display, dockWindow, 0, 0,
			     dock_width, dock_height, border_width, border_width, 0);

  if ((classHint = XAllocClassHint ()) == NULL)
    quit ("Failed to allocate class hint\n");

  classHint->res_class = PACKAGE;
  classHint->res_name = PACKAGE;
  XSetClassHint (display, dockWindow, classHint);
  XFree (classHint);

  if ((hints = XAllocWMHints ()) == NULL)
      quit ("Failed to allocate hints\n");

  hints->flags = StateHint | WindowGroupHint | IconWindowHint;
  hints->icon_window = iconWindow;
  hints->initial_state = WithdrawnState;
  hints->window_group = dockWindow;

  XSetWMHints (display, dockWindow, hints);
  XFree (hints);

/* Set the background */
  XSetWindowBackgroundPixmap (display, iconWindow, ParentRelative);

  selection_atom = XInternAtom (display, "_NET_SYSTEM_TRAY_S0", False);
  XSetSelectionOwner (display, selection_atom, dockWindow, CurrentTime);

  if (XGetSelectionOwner (display, selection_atom) == dockWindow)
    {
      XClientMessageEvent xev;
      int scr = DefaultScreen (display);
      Root = RootWindow (display, scr);
      xev.type = ClientMessage;
      xev.window = Root;
      xev.message_type = XInternAtom (display, "MANAGER", False);

      xev.format = 32;
      xev.data.l[0] = CurrentTime;
      xev.data.l[1] = selection_atom;
      xev.data.l[2] = dockWindow;
      xev.data.l[3] = 0;	/* manager specific data */
      xev.data.l[4] = 0;	/* manager specific data */

      XSendEvent (display, Root, False, StructureNotifyMask, (XEvent *) & xev);

      opcode_atom = XInternAtom (display, "_NET_SYSTEM_TRAY_OPCODE", False);
      data_atom = XInternAtom (display, "_NET_SYSTEM_TRAY_MESSAGE_DATA", False);

    }

  XSelectInput (display, dockWindow, SubstructureNotifyMask);
  if (!multiple)
    {
      XMapRaised (display, dockWindow);
    }
  XFlush (display);
  eventLoop ();
  XCloseDisplay (display);

  return 0;
}
