/* conff.c - Maintain configuration information
   Copyright (C) 2000, 2001 Thomas Moestl

This file is part of the pdnsd package.

pdnsd is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

pdnsd is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with pdsnd; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include "ipvers.h"
#include "conff.h"
#include "consts.h"
#include "helpers.h"
#include "conf-parse.h"

#if !defined(lint) && !defined(NO_RCSIDS)
static char rcsid[]="$Id: conff.c,v 1.26 2001/07/02 18:55:27 tmm Exp $";
#endif

#ifdef ENABLE_IPV4
globparm_t global={2048,CACHEDIR,53,{{INADDR_ANY}},0,604800,120,900,C_AUTH,C_AUTH,"",1,0,0600,"/var/lib/pcmcia/scheme",
		   20,30,TCP_TIMEOUT,PAR_QUERIES,1,0,65535};
#else
globparm_t global={2048,CACHEDIR,53,{IN6ADDR_ANY_INIT},0,604800,120,900,C_AUTH,C_AUTH,"",1,0,0600,"/var/lib/pcmcia/scheme",
		   20,30,TCP_TIMEOUT,PAR_QUERIES,1};
		   20,30,TCP_TIMEOUT,PAR_QUERIES,1,0,65535};
#endif
servparm_t server;
#ifdef ENABLE_IPV4
servparm_t serv_presets={53,C_NONE,120,900,600,"","","","","","",0,0,1,1,0,C_INCLUDED,NULL,0,{{INADDR_ANY}},{{INADDR_ANY}}};
#else
servparm_t serv_presets={53,C_NONE,120,900,600,"","","","","","",0,0,1,1,0,C_INCLUDED,NULL,0,{IN6ADDR_ANY_INIT},{IN6ADDR_ANY_INIT}};
#endif

darray servers=NULL;

void lex_set_io(FILE *in, FILE *out); /* defined in conf.l */
int  yyparse (void);                  /* from yacc/bison output */

/* Initialize a servparm_t */
void set_serv_presets(servparm_t *server)
{
	*server=serv_presets;
#if defined(ENABLE_IPV4) && defined(ENABLE_IPV6)
	/* In this case we have to set IN6ADDR_ANY_INIT */
	if (run_ipv6) {
		server->a.ipv6=server->ping_a.ipv6=in6addr_any;
	}
#endif
}

/*
 * Add a server (with parameters contained in serv) into the internal server list
 * (in the pointer servers)
 */
void add_server(servparm_t serv)
{
	if (!servers) {
		if ((servers=DA_CREATE(servparm_t))==NULL) {
			fprintf(stderr,"Error: out of memory.\n");
			exit(1);
		}
	}

	if (!(servers=da_grow(servers,1))) {
		fprintf(stderr,"Error: out of memory.\n");
		exit(1);
	}
	*DA_LAST(servers,servparm_t)=serv;
}

char *slist_add(servparm_t *sp, char *nm, int tp)
{
	slist_t *sl;
					
	if (sp->alist==NULL) {
		if ((sp->alist=DA_CREATE(slist_t))==NULL) {
			return "out of memory!";
		}
	}
	if ((sp->alist=da_grow(sp->alist,1))==NULL) {
		return "out of memory!";
	}
	sl=DA_LAST(sp->alist,slist_t);
	sl->rule=tp;
	if (strlen(nm)>255) {
		return "include/exclude: name too long.";
	}
	if (!strncp(sl->domain, nm, sizeof(sl->domain)))
		return "include/exclude: name too long!";
	if (sl->domain[strlen(sl->domain)-1]!='.') {
		return "domain name must end in dot for include=/exclude=.";
	}
	return NULL;
}

/*
 * Read a configuration file and save the result. This uses the parser generated by yacc/bison
 * from conf.y, which in turn uses the lexer generated lex/flex from conf.l
 */
void read_config_file(char *nm)
{
	FILE *in;
	struct stat sb;
	if (nm==NULL) {
		nm=CONFDIR"/pdnsd.conf";
	} 
	if (stat(nm,&sb)!=0) {
		fprintf(stderr,"Error: Could not stat conf file %s: %s\n",nm,strerror(errno));
		exit(3);
	}
	if (sb.st_uid!=getuid()) {
		fprintf(stderr,"Error: You must own the config file.\n");
		exit(3);
	}
	if ((sb.st_mode&(S_IWGRP|S_IWOTH))) {
		fprintf(stderr,"Error: Bad config file permissions: the file must be only writeable by the user.\n");
		exit(3);
	}
	if (!(in=fopen(nm,"r"))) {
		fprintf(stderr,"Error: Could not open conf file %s: %s\n",nm,strerror(errno));
		exit(3);
	}
	lex_set_io(in,stdout);
	if (yyparse())
		exit(3);
	fclose(in);
}

/* Report the currenct configuration into the file (for the status fifo, see status.c) */
void report_conf_stat(int f)
{
	char buf[ADDRSTR_MAXLEN];
	int i,j;
	servparm_t *st;
	slist_t *sl;
	
	fsprintf(f,"\nConfiguration:\n==============\nGlobal:\n-------\n");
	fsprintf(f,"\tCache size: %li kB\n",global.perm_cache);
	fsprintf(f,"\tServer directory: %s\n",global.cache_dir);
	fsprintf(f,"\tScheme file (for Linux pcmcia support): %s\n",global.scheme_file);
	fsprintf(f,"\tServer port: %i\n",global.port);
	fsprintf(f,"\tServer ip (0.0.0.0=any available one): %s\n",pdnsd_a2str(&global.a,buf,ADDRSTR_MAXLEN));
	fsprintf(f,"\tIgnore cache when link is down: %i\n",global.lndown_kluge);
	fsprintf(f,"\tMaximum ttl: %li\n",(long)global.max_ttl);
	fsprintf(f,"\tMinimum ttl: %li\n",(long)global.min_ttl);
	fsprintf(f,"\tNegative ttl: %li\n",(long)global.neg_ttl);
	fsprintf(f,"\tNegative RRS policy: %s\n",global.neg_rrs_pol==C_ON?"on":(global.neg_rrs_pol==C_OFF?"off":"auth"));
	fsprintf(f,"\tNegative domain policy: %s\n",global.neg_domain_pol==C_ON?"on":(global.neg_domain_pol==C_OFF?"off":"auth"));
	fsprintf(f,"\tRun as: %s\n",global.run_as);
	fsprintf(f,"\tStrict run as: %i\n",global.strict_suid);
	fsprintf(f,"\tParanoid mode (cache pollution prevention): %i\n",global.paranoid);
	fsprintf(f,"\tControl socket permissions (mode): %o\n",global.ctl_perms);
	fsprintf(f,"\tMaximum parallel queries served: %i\n",global.proc_limit);
	fsprintf(f,"\tMaximum queries queued for serving: %i\n",global.procq_limit);
	fsprintf(f,"\tMaximum parallel queries done: %i\n",global.par_queries);
	fsprintf(f,"\tRandomize records in answer: %i\n",global.rnd_recs);
	fsprintf(f,"\tQuery port start: %i\n",global.query_port_start);
	fsprintf(f,"\tQuery port end: %i\n",global.query_port_end);
	for(i=0;i<da_nel(servers);i++) {
		st=DA_INDEX(servers,i,servparm_t);
		fsprintf(f,"Server %i:\n------\n",i);
		fsprintf(f,"\tlabel: %s\n",st->label);
		fsprintf(f,"\tip: %s\n",pdnsd_a2str(&st->a,buf,ADDRSTR_MAXLEN));
		fsprintf(f,"\tport: %hu\n",st->port);
		fsprintf(f,"\tuptest: %i\n",st->uptest);
		fsprintf(f,"\ttimeout: %li\n",(long)st->timeout);
		fsprintf(f,"\tuptest interval: %li\n",(long)st->interval);
		fsprintf(f,"\tping timeout: %li\n",(long)st->ping_timeout);
		fsprintf(f,"\tping ip: %s\n",pdnsd_a2str(&st->ping_a,buf,ADDRSTR_MAXLEN));
		fsprintf(f,"\tinterface: %s\n",st->interface);
		fsprintf(f,"\tdevice (for special Linux ppp device support): %s\n",st->device);
		fsprintf(f,"\tuptest command: %s\n",st->uptest_cmd);
		fsprintf(f,"\tuptest user: %s\n",st->uptest_usr[0]!='\0'?st->uptest_usr:"(process owner)");
		if (st->scheme[0]!='\0')
			fsprintf(f,"\tscheme: %s\n", st->scheme);
		fsprintf(f,"\tforce cache purging: %i\n",st->purge_cache);
		fsprintf(f,"\tserver is cached: %i\n",!st->nocache);
		fsprintf(f,"\tlean query: %i\n",st->lean_query);
		fsprintf(f,"\tUse only proxy?: %i\n",st->is_proxy);
		fsprintf(f,"\tDefault policy: %s\n",st->policy==C_INCLUDED?"included":"excluded");
		fsprintf(f,"\tPolicies:\n");
		if (st->alist==NULL) {
			fsprintf(f,"\t\t(none)\n");
		} else {
			for (j=0;j<da_nel(st->alist);j++) {
				sl=DA_INDEX(st->alist,j,slist_t);
				fsprintf(f,"\t\t%s: %s\n",sl->rule==C_INCLUDED?"include":"exclude",sl->domain);
			}
		}
		fsprintf(f,"\tserver assumed available: %i\n",st->is_up);
	}
}




