/*---------------------------------------------------------------------------*
 |              PDFlib - A library for generating PDF on the fly             |
 +---------------------------------------------------------------------------+
 | Copyright (c) 1997-2001 PDFlib GmbH and Thomas Merz. All rights reserved. |
 +---------------------------------------------------------------------------+
 |    This software is NOT in the public domain.  It can be used under two   |
 |    substantially different licensing terms:                               |
 |                                                                           |
 |    The commercial license is available for a fee, and allows you to       |
 |    - ship a commercial product based on PDFlib                            |
 |    - implement commercial Web services with PDFlib                        |
 |    - distribute (free or commercial) software when the source code is     |
 |      not made available                                                   |
 |    Details can be found in the file PDFlib-license.pdf.                   |
 |                                                                           |
 |    The "Aladdin Free Public License" doesn't require any license fee,     |
 |    and allows you to                                                      |
 |    - develop and distribute PDFlib-based software for which the complete  |
 |      source code is made available                                        |
 |    - redistribute PDFlib non-commercially under certain conditions        |
 |    - redistribute PDFlib on digital media for a fee if the complete       |
 |      contents of the media are freely redistributable                     |
 |    Details can be found in the file aladdin-license.pdf.                  |
 |                                                                           |
 |    These conditions extend to ports to other programming languages.       |
 |    PDFlib is distributed with no warranty of any kind. Commercial users,  |
 |    however, will receive warranty and support statements in writing.      |
 *---------------------------------------------------------------------------*/

/* $Id: p_stream.c,v 1.38.10.2 2001/05/17 16:17:08 rjs Exp $
 *
 * Stream handling routines for PDFlib 
 *
 */

#include <ctype.h>
#include <math.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include "p_intern.h"

#ifdef HAVE_LIBZ
#include "zlib.h"
#endif

struct pdf_stream_s {
    pdf_byte	*basepos;		/* start of this chunk */
    pdf_byte	*curpos;		/* next write position */
    pdf_byte	*maxpos;		/* maximum position of chunk */
    size_t	base_offset;		/* base offset of this chunk */
    pdf_bool	compressing;		/* in compression state */
#ifdef HAVE_LIBZ
    z_stream	z;			/* zlib compression stream */
#endif
};

#ifdef HAVE_LIBZ
/* wrapper for p->malloc for use in zlib */
static voidpf
pdf_zlib_alloc(voidpf p, uInt items, uInt size)
{
    return (voidpf) ((PDF *)p)->malloc((PDF *)p, items * size, "zlib");
}

#endif	/* HAVE_LIBZ */

/*
 * The caller must use the contents of the returned buffer before
 * calling the next PDFlib function.
 */

PDFLIB_API const char * PDFLIB_CALL
PDF_get_buffer(PDF *p, long *size)
{
    static const char fn[] = "PDF_get_buffer";

    PDF_TRACE(("%s\t(pdf[%p], [%p]);", fn, (void *) p, (void *) size));

    if (PDF_SANITY_CHECK_FAILED(p)) {
	*size = (long) 0;
	PDF_TRACE((" [NULL, size=0]\n"));
	return ((const char *) NULL);
    }

    if (p->writeproc)
	pdf_error(p, PDF_RuntimeError,
	    "Don't use PDF_get_buffer() when writing to file");

    PDF_CHECK_SCOPE(p, fn, pdf_state_object | pdf_state_document);

    *size = (long) (p->stream->curpos - p->stream->basepos);

    p->stream->base_offset +=
	(size_t) (p->stream->curpos - p->stream->basepos);
    p->stream->curpos = p->stream->basepos;

    PDF_TRACE((" [%p, size=%ld]\n", (void *) (p->stream->basepos), *size));

    return (const char *) p->stream->basepos;
}

pdf_bool
pdf_buffer_not_empty(PDF *p)
{
    return(!p->writeproc && p->stream->curpos != p->stream->basepos);
}

void
pdf_boot_stream(PDF *p)
{
    p->stream = (pdf_stream *) p->malloc(p, sizeof(pdf_stream),
		"pdf_boot_stream");

    /* curpos must be initialized here so that the check for empty
     * buffer in PDF_delete() also works in the degenerate case of
     * no output produced.
     */
    p->stream->basepos = p->stream->curpos = NULL;
}

void
pdf_init_stream(PDF *p)
{
    /*
     * This may be left over from the previous run. We deliberately
     * don't reuse the previous buffer in order to avoid potentially
     * unwanted growth of the allocated buffer due to a single large
     * document in a longer series of documents.
     */
    if (p->stream->basepos)
	p->free(p, (void *) p->stream->basepos);

    p->stream->basepos		= (pdf_byte *)p->malloc(p, STREAM_CHUNKSIZE,
				    "pdf_open_stream");
    p->stream->curpos		= p->stream->basepos;
    p->stream->maxpos		= p->stream->basepos + STREAM_CHUNKSIZE;
    p->stream->base_offset	= 0L;
    p->stream->compressing	= pdf_false;

#ifdef HAVE_LIBZ
    p->stream->z.zalloc	= (alloc_func) pdf_zlib_alloc;
    p->stream->z.zfree	= (free_func) p->free;
    p->stream->z.opaque	= (voidpf) p;
#endif
}

static void
pdf_check_stream(PDF *p, size_t len)
{
    size_t max;
    int cur;
    
    if (p->stream->curpos + len <= p->stream->maxpos)
	return;

    if (p->flush & PDF_FLUSH_HEAVY) {
	pdf_flush_stream(p);

	if (p->stream->curpos + len <= p->stream->maxpos)
	    return;
    }

    max = (size_t) (2 * (p->stream->maxpos - p->stream->basepos));
    cur = p->stream->curpos - p->stream->basepos;

    p->stream->basepos = (pdf_byte *)
	p->realloc(p, (void *) p->stream->basepos, max, "pdf_check_stream");
    p->stream->maxpos = p->stream->basepos + max;
    p->stream->curpos = p->stream->basepos + cur;

    pdf_check_stream(p, len);
}

void
pdf_flush_stream(PDF *p)
{
    size_t size;

    if (!p->writeproc || p->stream->compressing)
	return;

    size = (size_t) (p->stream->curpos - p->stream->basepos);

    if (p->writeproc(p, (void *) p->stream->basepos, size) != size) {
	p->free(p, p->stream->basepos);
	p->stream->basepos = NULL;
	pdf_error(p, PDF_IOError, "Couldn't write output (disk full?)");
    }

    p->stream->base_offset += (size_t) (p->stream->curpos - p->stream->basepos);
    p->stream->curpos = p->stream->basepos;
}

void
pdf_close_stream(PDF *p)
{
    /* this may happen in rare cases */
    if (!p->stream->basepos)
	return;

    if (p->stream->basepos) {
	p->free(p, (void *) p->stream->basepos);
	p->stream->basepos = NULL;
    }

    p->free(p, (void *) p->stream);
}

long
pdf_tell(PDF *p)
{
    return(p->stream->base_offset + p->stream->curpos - p->stream->basepos);
}

int
pdf_compresslevel(PDF *p)
{
    return p->compresslevel;
}

void
pdf_compress_init(PDF *p)
{
    if (!p->compresslevel) {
	p->stream->compressing = pdf_false;
	return;
    }

#ifdef HAVE_LIBZ
    if (deflateInit(&p->stream->z, p->compresslevel) != Z_OK)
	pdf_error(p, PDF_SystemError, "Compression error (deflateInit)");

    p->stream->z.avail_in = 0;
#endif /* HAVE_LIBZ */

    p->stream->compressing = pdf_true;
}

void
pdf_compress_end(PDF *p)
{
    int status;

    /* this may happen during cleanup triggered by an exception handler */
    if (!p->stream->compressing)
	return;

    if (!p->compresslevel) {
	p->stream->compressing = pdf_false;
	return;
    }

#ifdef HAVE_LIBZ
    /* Finish the stream */
    do {
	pdf_check_stream(p, 128);
	p->stream->z.next_out = (Bytef *) p->stream->curpos;
	p->stream->z.avail_out =
	    (uInt) (p->stream->maxpos - p->stream->curpos);

	status = deflate(&(p->stream->z), Z_FINISH);
	p->stream->curpos = p->stream->z.next_out;

	if (status != Z_STREAM_END && status != Z_OK)
	    pdf_error(p, PDF_SystemError, "Compression error (Z_FINISH)");

    } while (status != Z_STREAM_END);

    if (deflateEnd(&p->stream->z) != Z_OK)
	pdf_error(p, PDF_SystemError, "Compression error (deflateEnd)");

    p->stream->compressing = pdf_false;
#endif /* HAVE_LIBZ */
}

/* **************************** output functions ************************ */

/*
 * Write binary data to the output without any modification,
 * and apply compression if we are currently in compression mode.
 */
void
pdf_write(PDF *p, const void *data, size_t size)
{
#ifdef HAVE_LIBZ
    if (p->stream->compressing) {
	p->stream->z.avail_in	= (uInt) size;
	p->stream->z.next_in	= (Bytef *) data;
	p->stream->z.avail_out	= 0;

	while (p->stream->z.avail_in > 0) {
	    if (p->stream->z.avail_out == 0) {
		/* estimate output buffer size */
		pdf_check_stream(p, (size_t) (p->stream->z.avail_in/4 + 16));
		p->stream->z.next_out	= (Bytef *) p->stream->curpos;
		p->stream->z.avail_out	= 
			(uInt) (p->stream->maxpos - p->stream->curpos);
	    }

	    if (deflate(&(p->stream->z), Z_NO_FLUSH) != Z_OK)
		pdf_error(p, PDF_SystemError, "Compression error (Z_NO_FLUSH)");

	   p->stream->curpos = p->stream->z.next_out;
	}
    }else {
#endif /* HAVE_LIBZ */

	pdf_check_stream(p, size);
	memcpy(p->stream->curpos, data, size);
	p->stream->curpos += size;

#ifdef HAVE_LIBZ
    }
#endif /* HAVE_LIBZ */
}

/*
 * Write a string to the output, converting from native encoding to ASCII
 * if necessary.
 */

void
pdf_puts(PDF *p, const char *s)
{
#ifdef PDFLIB_EBCDIC

    char buffer[PDF_LINEBUFLEN];

    strcpy(buffer, s);
    pdf_make_ascii(buffer);
    pdf_write(p, (void *) buffer, strlen(buffer));

#else

    pdf_write(p, (void *) s, strlen(s));

#endif	/* PDFLIB_EBCDIC */
}

/* Write a character to the output without any modification. */

void
pdf_putc(PDF *p, char c)
{
    pdf_write(p, (void *) &c, (size_t) 1);
}

/*
 * Write a formatted string to the output, converting from native
 * encoding to ASCII if necessary.
 */

static const char digits[] = "0123456789ABCDEF";

static char *
pdf_ltoa(char *buf, long n, int width, char pad, int base)
{
    char	aux[20];
    int		k, i = sizeof aux;
    char *	dest = buf;
    unsigned long ul = (unsigned long) n;
    ldiv_t qr;

    if (n == 0)
    {
	if (width == 0)
	    width = 1;

	for (k = 0; k < width; ++k)
	    *(dest++) = '0';

	return dest;
    }

    if (n < 0 && base == 10)
    {
	ul = (unsigned long) -ul;	/* safe against overflow,
					    while "n = -n" isn't! */
	*(dest++) = '-';
    }

    n = (long) (ul / base);
    aux[--i] = digits[ul % base];

    while (0 < n)
    {
	qr = ldiv(n, base);

	n = qr.quot;
	aux[--i] = digits[qr.rem];
    }

    width -= (int) (sizeof aux) - i;
    for (k = 0; k < width; ++k)
	*(dest++) = pad;

    memcpy(dest, &aux[i], sizeof aux - i);
    return dest + sizeof aux - i;
} /* pdf_ltoa */

static char *
pdf_ftoa(PDF *p, char *buf, double x)
{
    static const long pow10[] = { 1, 10, 100, 1000, 10000, 100000, 1000000 };

    char *	dest = buf;
    double	integ, fract;
    long	f;

    if (fabs(x) < PDF_SMALLREAL)
    {
	*dest = '0';
	return dest + 1;
    }

    if (x < 0)
    {
	x = -x;
	*(dest++) = '-';
    }

    if (x >= PDF_BIGREAL)
    {
	if (x > PDF_BIGINT)
	    pdf_error(p, PDF_ValueError,
		"floating point value too large in pdf_ftoa");

	return pdf_ltoa(dest, (long) floor(x + 0.5), 0, ' ', 10);
    }

    fract = modf(x, &integ);
    f = (long) floor(fract * pow10[p->floatdigits] + 0.5);

    if (f == pow10[p->floatdigits])
    {
	integ += 1.0;
	f = 0;
    }

    dest = pdf_ltoa(dest, (long) integ, 0, ' ', 10);

    if (f != 0)
    {
	ldiv_t	qr;
	char *	aux;
	int	i = p->floatdigits;

	*(dest++) = '.';

	do	/* avoid trailing zeros */
	{
	    qr = ldiv(f, 10);
	    f = qr.quot;
	    --i;
	} while (qr.rem == 0);

	aux = dest + i + 1;
	dest[i--] = digits[qr.rem];

	for (; 0 <= i; --i)
	{
	    qr = ldiv(f, 10);
	    f = qr.quot;
	    dest[i] = digits[qr.rem];
	}

	return aux;
    }

    return dest;
} /* pdf_ftoa */

int
pdf_vsprintf(PDF *p, char *buf, const char *format, va_list args)
{
    char *dest = buf;

    for (/* */ ; /* */ ; /* */)
    {
	int	width = 0;
	char	pad = ' ';

	/* as long as there is no '%', just print.
	*/
	while (*format != 0 && *format != '%')
	    *(dest++) = *(format++);

	if (*format == 0)
	{
	    *dest = 0;
	    return dest - buf;
	}

	if (*(++format) == '0')
	{
	    pad = '0';
	    ++format;
	}

	while (isdigit(*format))
	    width = 10*width + *(format++) - '0';

	switch (*format)
	{
	    case 'X':
		dest = pdf_ltoa(dest, va_arg(args, int), width, pad, 16);
		break;

	    case 'c':
		*(dest++) = va_arg(args, int);
		break;

	    case 'd':
		dest = pdf_ltoa(dest, va_arg(args, int), width, pad, 10);
		break;

	    case 'f':
		dest = pdf_ftoa(p, dest, va_arg(args, double));
		break;

	    case 'l':
	    {
		long n = va_arg(args, long);

		switch (*(++format))
		{
		    case 'X':
			dest = pdf_ltoa(dest, n, width, pad, 16);
			break;

		    case 'd':
			dest = pdf_ltoa(dest, n, width, pad, 10);
			break;

		    default:
			pdf_error(p, PDF_SystemError,
			    "unknown format 'l%c' (%02X) in pdf_vsprintf",
				isprint(*format) ? *format : '?', *format);
		}

		break;
	    }

	    case 's':
	    {
		char *	str = va_arg(args, char *);
		size_t	len = strlen(str);

		memcpy(dest, str, len);
		dest += len;
		break;
	    }

	    case '%':
		*(dest++) = '%';
		break;

	    default:
		pdf_error(p, PDF_SystemError,
		    "unknown format specification '%c' (%02X) in pdf_vsprintf",
			    isprint(*format) ? *format : '?', *format);
	} /* switch */

	++format;
    } /* loop */
} /* pdf_vsprintf */


void
pdf_printf(PDF *p, const char *fmt, ...)
{
    char	buf[PDF_LINEBUFLEN];	/* formatting buffer */
    va_list ap;

    va_start(ap, fmt);

    pdf_vsprintf(p, buf, fmt, ap);
    pdf_puts(p, buf);

    va_end(ap);
}

#ifdef PDFLIB_EBCDIC

/*
 * Platforms other than AS/400 and OS/390 will have to decide
 * on their own which codepage to use. This code is deliberately
 * set up such that a linker error will result if no code page
 * has been explicitly selected.
 */

#if PDF_CODEPAGE == 273

/* EBCDIC code page 273 for OS/400 */

static unsigned const char ebcdic2ascii_table[256] = "\
\000\001\002\003\234\011\206\177\227\215\216\013\014\015\016\017\
\020\021\022\023\235\015\010\207\030\031\222\217\034\035\036\037\
\200\201\202\203\204\012\027\033\210\211\212\213\214\005\006\007\
\220\221\026\223\224\225\226\004\230\231\232\233\024\025\236\032\
\040\240\342\173\340\341\343\345\347\361\304\056\074\050\053\041\
\046\351\352\353\350\355\356\357\354\176\334\044\052\051\073\136\
\055\057\302\133\300\301\303\305\307\321\366\054\045\137\076\077\
\370\311\312\313\310\315\316\317\314\140\072\043\247\047\075\042\
\330\141\142\143\144\145\146\147\150\151\253\273\360\375\376\261\
\260\152\153\154\155\156\157\160\161\162\252\272\346\270\306\244\
\265\337\163\164\165\166\167\170\171\172\241\277\104\335\336\256\
\242\243\245\267\251\100\266\274\275\276\254\174\077\250\264\327\
\344\101\102\103\104\105\106\107\110\111\255\364\246\362\363\365\
\374\112\113\114\115\116\117\120\121\122\271\373\175\371\372\377\
\326\367\123\124\125\126\127\130\131\132\262\324\134\322\323\325\
\060\061\062\063\064\065\066\067\070\071\263\333\135\331\332\237\
";

/* EBCDIC code page 819->273 for OS/400 */

static unsigned const char ascii2ebcdic_table[256] = "\
\000\001\002\003\067\055\056\057\026\005\045\013\014\015\016\017\
\020\021\022\023\074\075\062\046\030\031\077\047\034\035\036\037\
\100\117\177\173\133\154\120\175\115\135\134\116\153\140\113\141\
\360\361\362\363\364\365\366\367\370\371\172\136\114\176\156\157\
\265\301\302\303\304\305\306\307\310\311\321\322\323\324\325\326\
\327\330\331\342\343\344\345\346\347\350\351\143\354\374\137\155\
\171\201\202\203\204\205\206\207\210\211\221\222\223\224\225\226\
\227\230\231\242\243\244\245\246\247\250\251\103\273\334\131\007\
\040\041\042\043\044\025\006\027\050\051\052\053\054\011\012\033\
\060\061\032\063\064\065\066\010\070\071\072\073\004\024\076\377\
\101\252\260\261\237\262\314\174\275\264\232\212\272\312\257\157\
\220\217\352\372\276\240\266\263\235\332\233\213\267\270\271\253\
\144\145\142\146\112\147\236\150\164\161\162\163\170\165\166\167\
\157\151\355\356\353\357\340\277\200\375\376\373\132\255\256\241\
\104\105\102\106\300\107\234\110\124\121\122\123\130\125\126\127\
\214\111\315\316\313\317\152\341\160\335\336\333\320\215\216\337\
";

#endif /* PDF_CODEPAGE == 273 */

#if PDF_CODEPAGE == 37

/* EBCDIC code page 37->819 for OS/400 */

static unsigned const char ebcdic2ascii_table[256] = "\
\000\001\002\003\234\011\206\177\227\215\216\013\014\015\016\017\
\020\021\022\023\235\040\010\207\030\031\222\217\034\035\036\037\
\200\201\202\203\204\012\027\033\210\211\212\213\214\005\006\007\
\220\221\026\223\224\225\226\004\230\231\232\233\024\025\236\032\
\040\240\342\344\340\341\343\345\347\361\133\056\074\050\053\174\
\046\351\352\353\350\355\356\357\354\337\135\044\052\051\073\254\
\055\057\302\304\300\301\303\305\307\321\246\054\045\137\076\077\
\370\311\312\313\310\315\316\317\314\140\072\043\100\047\075\042\
\330\141\142\143\144\145\146\147\150\151\253\273\360\375\376\261\
\260\152\153\154\155\156\157\160\161\162\252\272\346\270\306\244\
\265\176\163\164\165\166\167\170\171\172\241\277\320\335\336\256\
\136\243\245\267\251\247\266\274\275\276\133\135\257\250\264\327\
\173\101\102\103\104\105\106\107\110\111\255\364\366\362\363\365\
\175\112\113\114\115\116\117\120\121\122\271\373\374\371\372\377\
\134\367\123\124\125\126\127\130\131\132\262\324\326\322\323\325\
\060\061\062\063\064\065\066\067\070\071\263\333\334\331\332\237\
";

/* EBCDIC code page 819->037 for OS/400 */

static unsigned const char ascii2ebcdic_table[256] = "\
\000\001\002\003\067\055\056\057\026\005\045\013\014\015\016\017\
\025\021\022\023\074\075\062\046\030\031\077\047\034\035\036\037\
\100\201\177\173\133\154\120\175\115\135\134\116\153\140\113\141\
\360\361\362\363\364\365\366\367\370\371\172\136\114\176\156\157\
\174\301\302\303\304\305\306\307\310\311\321\322\323\324\325\326\
\327\330\331\342\343\344\345\346\347\350\351\272\340\273\260\155\
\171\201\202\203\204\205\206\207\210\211\221\222\223\224\225\226\
\227\230\231\242\243\244\245\246\247\250\251\300\117\320\241\007\
\040\041\042\043\044\006\006\027\050\051\052\053\054\011\012\033\
\060\061\032\063\064\065\066\010\070\071\072\073\004\024\076\377\
\101\252\000\261\237\262\152\265\275\264\232\212\137\312\257\274\
\220\217\352\372\276\240\266\263\235\332\233\213\267\270\271\253\
\144\145\142\146\143\147\236\150\164\161\162\163\170\165\166\167\
\254\151\355\356\353\357\354\277\200\375\376\373\374\255\256\131\
\104\105\102\106\103\107\234\110\124\121\122\123\130\125\126\127\
\214\111\315\316\313\317\314\341\160\335\336\333\334\215\216\337\
";

#endif /* PDF_CODEPAGE == 037 */

#if PDF_CODEPAGE == 819

/* EBCDIC code page 819 for OS/400 */

static unsigned const char ebcdic2ascii_table[256] = "\
\000\001\002\003\234\011\206\177\227\215\216\013\014\015\016\017\
\020\021\022\023\235\020\010\207\030\031\222\217\034\035\036\037\
\200\201\202\203\204\012\027\033\210\211\212\213\214\005\006\007\
\220\221\026\223\224\225\226\004\230\231\232\233\024\025\236\032\
\040\240\342\344\340\341\343\345\347\361\133\056\074\050\053\174\
\046\351\352\353\350\355\356\357\354\337\135\044\052\051\073\254\
\055\057\302\304\300\301\303\305\307\321\246\054\045\137\076\077\
\370\311\312\313\310\315\316\317\314\140\072\043\100\047\075\042\
\330\141\142\143\144\145\146\147\150\151\253\273\360\375\376\261\
\260\152\153\154\155\156\157\160\161\162\252\272\346\270\306\244\
\265\176\163\164\165\166\167\170\171\172\241\277\320\335\336\256\
\136\243\245\267\251\247\266\274\275\276\133\135\257\250\264\327\
\173\101\102\103\104\105\106\107\110\111\255\364\366\362\363\365\
\175\112\113\114\115\116\117\120\121\122\271\373\374\371\372\377\
\134\367\123\124\125\126\127\130\131\132\262\324\326\322\323\325\
\060\061\062\063\064\065\066\067\070\071\263\333\334\331\332\237\
";

/* EBCDIC code page 819 for OS/400 */

/* TODO: validate this table */
static unsigned const char ascii2ebcdic_table[256] = "\
\000\001\002\003\067\055\056\057\026\005\045\013\014\015\016\017\
\025\021\022\023\074\075\062\046\030\031\077\047\034\035\036\037\
\100\201\177\173\133\154\120\175\115\135\134\116\153\140\113\141\
\360\361\362\363\364\365\366\367\370\371\172\136\114\176\156\157\
\174\301\302\303\304\305\306\307\310\311\321\322\323\324\325\326\
\327\330\331\342\343\344\345\346\347\350\351\272\340\273\260\155\
\171\201\202\203\204\205\206\207\210\211\221\222\223\224\225\226\
\227\230\231\242\243\244\245\246\247\250\251\300\117\320\241\007\
\040\041\042\043\044\006\006\027\050\051\052\053\054\011\012\033\
\060\061\032\063\064\065\066\010\070\071\072\073\004\024\076\377\
\101\252\000\261\237\262\152\265\275\264\232\212\137\312\257\274\
\220\217\352\372\276\240\266\263\235\332\233\213\267\270\271\253\
\144\145\142\146\143\147\236\150\164\161\162\163\170\165\166\167\
\254\151\355\356\353\357\354\277\200\375\376\373\374\255\256\131\
\104\105\102\106\103\107\234\110\124\121\122\123\130\125\126\127\
\214\111\315\316\313\317\314\341\160\335\336\333\334\215\216\337\
";

#endif /* PDF_CODEPAGE == 819 */

#if PDF_CODEPAGE == 1047

/* EBCDIC code page 1047, Unix system services for OS/390 (Open Edition) */

static unsigned const char ebcdic2ascii_table[256] = "\
\000\001\002\003\234\011\206\177\227\215\216\013\014\015\016\017\
\020\021\022\023\235\012\010\207\030\031\222\217\034\035\036\037\
\200\201\202\203\204\205\027\033\210\211\212\213\214\005\006\007\
\220\221\026\223\224\225\226\004\230\231\232\233\024\025\236\032\
\040\240\342\344\340\341\343\345\347\361\242\056\074\050\053\174\
\046\351\352\353\350\355\356\357\354\337\041\044\052\051\073\136\
\055\057\302\304\300\301\303\305\307\321\246\054\045\137\076\077\
\370\311\312\313\310\315\316\317\314\140\072\043\100\047\075\042\
\330\141\142\143\144\145\146\147\150\151\253\273\360\375\376\261\
\260\152\153\154\155\156\157\160\161\162\252\272\346\270\306\244\
\265\176\163\164\165\166\167\170\171\172\241\277\320\133\336\256\
\254\243\245\267\251\247\266\274\275\276\335\250\257\135\264\327\
\173\101\102\103\104\105\106\107\110\111\255\364\366\362\363\365\
\175\112\113\114\115\116\117\120\121\122\271\373\374\371\372\377\
\134\367\123\124\125\126\127\130\131\132\262\324\326\322\323\325\
\060\061\062\063\064\065\066\067\070\071\263\333\334\331\332\237\
";

/* EBCDIC code page 1047, Unix system services for OS/390 (Open Edition) */

static unsigned const char ascii2ebcdic_table[256] = "\
\000\001\002\003\067\055\056\057\026\005\025\013\014\015\016\017\
\020\021\022\023\074\075\062\046\030\031\077\047\034\035\036\037\
\100\132\177\173\133\154\120\175\115\135\134\116\153\140\113\141\
\360\361\362\363\364\365\366\367\370\371\172\136\114\176\156\157\
\174\301\302\303\304\305\306\307\310\311\321\322\323\324\325\326\
\327\330\331\342\343\344\345\346\347\350\351\255\340\275\137\155\
\171\201\202\203\204\205\206\207\210\211\221\222\223\224\225\226\
\227\230\231\242\243\244\245\246\247\250\251\300\117\320\241\007\
\040\041\042\043\044\045\006\027\050\051\052\053\054\011\012\033\
\060\061\032\063\064\065\066\010\070\071\072\073\004\024\076\377\
\101\252\112\261\237\262\152\265\273\264\232\212\260\312\257\274\
\220\217\352\372\276\240\266\263\235\332\233\213\267\270\271\253\
\144\145\142\146\143\147\236\150\164\161\162\163\170\165\166\167\
\254\151\355\356\353\357\354\277\200\375\376\373\374\272\256\131\
\104\105\102\106\103\107\234\110\124\121\122\123\130\125\126\127\
\214\111\315\316\313\317\314\341\160\335\336\333\334\215\216\337\
";

#endif /* PDF_CODEPAGE == 1047 */

/* The EBCDIC to ASCII in-place string conversion routine */

void
pdf_ebcdic2ascii(char *s)
{
    for (/* */; *s; s++)
	*s = (char) ebcdic2ascii_table[(unsigned char) (*s)];
}

/* The EBCDIC to ASCII in-place buffer conversion routine */

void
pdf_ebcdic2ascii_len(char *s, size_t len)
{
    char *end = s+len ;

    for (/* */; s < end; s++)
	*s = (char) ebcdic2ascii_table[(unsigned char) (*s)];
}

/* The ASCII to EBCDIC in-place buffer conversion routine */

void
pdf_ascii2ebcdic(char *s)
{
    for (/* */; *s; s++)
	*s = (char) ascii2ebcdic_table[(unsigned char) (*s)];
}

#endif /* PDFLIB_EBCDIC */
