static char rcsid[] = "$Id: PctestIpv4.cc,v 1.21 2000/03/22 00:14:34 bmah Exp $";
//
// $Id: PctestIpv4.cc,v 1.21 2000/03/22 00:14:34 bmah Exp $
//
// PctestIpv4.cc
// Bruce A. Mah <bmah@ca.sandia.gov>
//
// This work was first produced by an employee of Sandia National
// Laboratories under a contract with the U.S. Department of Energy.
// Sandia National Laboratories dedicates whatever right, title or
// interest it may have in this software to the public. Although no
// license from Sandia is needed to copy and use this software,
// copying and using the software might infringe the rights of
// others. This software is provided as-is. SANDIA DISCLAIMS ANY
// WARRANTY OF ANY KIND, EXPRESS OR IMPLIED.
//
// Class of IPv4 tests
//

#include <sys/types.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>

#include "pc.h"
#include "PctestIpv4.h"

//
// PctestIpv4::SetTargetName
//
// Input:  target hostname (target)
//
// Output:  success code (negative if error)
//
// Set target name and do protocol-dependent resolving to get a 
// network address.  In case of an error, we're responsible for
// printing some error message.
//
int PctestIpv4::SetTargetName(char *t)
{

    int len;			// temporary buffer length
    struct hostent *host;	// resolver host entry
    int rc;			// return code for syscalls

    // Attempt to resolve, if possible
    host = gethostbyname(t);

    // Resolver failed
    if (host == NULL) {

	// Some systems don't have herror (non-BSD?), so for those,
	// we'll cobble together an error message.
#ifdef HAVE_HERROR
	herror(t);
#else
	fprintf(stderr, "%s: Host not found\n", t);
#endif /* HAVE_HERROR */

	memset((void *) &targetAddress, 0, sizeof(struct in_addr));
	len = strlen(t);
	targetName = (char *) calloc(len + 1, sizeof(char));
	strncpy(targetName, t, len + 1);

	return -1;
    }

    IF_DEBUG(3, fprintf(stderr, "h_name = %s\n", host->h_name));
    IF_DEBUG(3, fprintf(stderr, "h_length = %d\n", host->h_length));
    IF_DEBUG(3, fprintf(stderr, "h_addr_list[0] = %x\n", *((int *)(host->h_addr_list[0]))));
	     
    // Get IP address
    memcpy(&targetAddress, host->h_addr_list[0], host->h_length);

    memset((void *) &targetSocketAddress, 0, sizeof(struct sockaddr_in));
    // Note:  Only BSD4.3Reno and later have sin_len in struct
    // sockaddr_in, so we need to test for it.
#ifdef HAVE_SOCKADDR_SA_LEN
    targetSocketAddress.sin_len = sizeof(struct sockaddr_in);
#endif /* HAVE_SOCKADDR_SA_LEN */
    targetSocketAddress.sin_family = AF_INET;
    targetSocketAddress.sin_port = 0; // set on each test
    memcpy(&targetSocketAddress.sin_addr, host->h_addr_list[0], host->h_length);

    // Make a copy of the canonical hostname
    len = strlen(host->h_name);
    targetName = (char *) calloc(len + 1, sizeof(char));
    if (targetName == NULL) {
	fprintf(stderr, "Couldn't allocate memory for hostname.\n");
	return -1;
    }
    strncpy(targetName, host->h_name, len + 1);

    // Now get the two sockets we need, one to send UDP datagrams
    // and another to receive ICMP packets.  The UDP socket is
    // easy, but we need to use a raw socket for ICMP.
    socketOut = GetSocketOut();
    if (socketOut < 0) {
	return socketOut;
    }

    socketIn = GetSocketIn();
    if (socketIn < 0) {
	return socketIn;
    }

    return 0;

}

//
// GetSocketIn
//
// Input:  None
//
// Output:  In return value, returns socket number.
//
// Get input socket of an appropriate type.  We'll make this virtual 
// shortly.
//
int PctestIpv4::GetSocketIn() {
    struct protoent *icmpProto = getprotobyname("icmp"); 
				// be really anal-retentive
    if (icmpProto == NULL) {
	fprintf(stderr, "Couldn't find ICMP protocol, using 1\n");
	proto = 1;		// instance variable of PctestIpv4
    }
    else {
	proto = icmpProto->p_proto;
    }
    socketIn = socket(PF_INET, SOCK_RAW, proto);
    if (socketIn < 0) {
	perror("socket");
	return socketIn;
    }

    return socketIn;
}

//
// PctestIpv4::GetPrintableAddress
//
// Input:  None
//
// Output:  Pointer to ASCII representation of address (in return
// value)
//
char *PctestIpv4::GetPrintableAddress()
{
    return (GetPrintableAddress(&targetAddress));
}

//
// PctestIpv4::GetPrintableAddress
//
// Input:  Pointer to address structure
//
// Output:  Pointer to ASCII representation of address (in return
// value)
//
char *PctestIpv4::GetPrintableAddress(void *a)
{
    return (inet_ntoa(*((struct in_addr *) a)));
}

//
// PctestIpv4::GetName
//
// Input:  Pointer to address structure
//
// Output:  Pointer to ASCII representation of name (in return
// value)
//
char *PctestIpv4::GetName(void *a)
{
    struct hostent *host;
    host = gethostbyaddr((char *) a, sizeof(struct in_addr), AF_INET);

    if (host) {
	return (host->h_name);
    }
    else {
	return (GetPrintableAddress(a));
    }

}

//
// PctestIpv4::GetMinSize
//
// Input:  None
//
// Output:  Minimum packet size possible for this protocol (in return
// value).
//
int PctestIpv4::GetMinSize() 
{
    return (sizeof(ip) + sizeof(udphdr) + 4);
}

//
// PctestIpv4::GetAction
//
// Input:  a test record
//
// Output:  action code
//
// Figure out the meaning of a particular combination of ICMPv4 type
// and code values.
//
PctestActionType PctestIpv4::GetAction(TestRecord &tr) 
{
    if (tr.icmpType == ICMP_TIMXCEED) {
	return PctestActionValid;
    }
    else if ((tr.icmpType == ICMP_UNREACH) &&
	     (tr.icmpCode == ICMP_UNREACH_PORT)) {
	return PctestActionValidLasthop;
    }
    else if ((tr.icmpType == ICMP_UNREACH) &&
	     (tr.icmpCode == ICMP_UNREACH_FILTER_PROHIB)) {
	return PctestActionFiltered;
    }
    else {
	return PctestActionAbort;
    }
}
