/* $Id: sort.c,v 1.18 2005/02/16 09:51:15 igor Exp $ */

/*
 * Copyright (c) 2004 Igor Boehm <igor@bytelabs.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/queue.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "macros.h"
#include "fail.h"
#include "pkgobject.h"
#include "sort.h"

/* PROTOTYPES */
static void     pb_qsort_two_fields(PkgObject *[], int, int);
static int      cmp_category(const void *, const void *);
static int      cmp_distrib_name(const void *, const void *);

/**
 * This function receives which field of PkgBase should be sorted
 * and if it should be done asc. or desc. and a pointer to the list
 * itself.
 */
Boolean
pb_pkglist_sort(const size_t len)
{
	PkgObject      *array[len], *np;
	size_t          i;

	i = 0;
	for (np = pkgobject_get_list_head(); (np) && (i < len);
	     np = pkgobject_get_list_next(np)) {
		array[i++] = np;
	}
	if (i != len)
		pb_error(1, "Something nasty happened during the sort process.");
	pb_qsort_two_fields(array, 0, (len - 1));
	pkgobject_list_insert_head(array[0]);
	for (i = 1; i < len; i++)
		pkgobject_list_insert_after(array[i - 1], array[i]);
	return TRUE;
}

/**
 * Two level sorting. This function is used for sorting the
 * categories first, afterwards the distrib names are sorted
 * in each category.
 */
static void
pb_qsort_two_fields(PkgObject * array[], int left, int right)
{
	int             len, cmpres;

	cmpres = 0;
	len = right;
	left = right = 0;
	/* SORT ACCORDING TO CATEGORY */
	qsort(array, len, sizeof(PkgObject *), cmp_category);
	/* SORT ACCORDING TO DISTRIB NAME */
	while (right < len) {
  		if (!strcmp(pkgobject_get_category(array[right]),
			    pkgobject_get_category(array[right + 1]))) {
			right++;
		} else {
			qsort(&array[left], (right - left), sizeof(PkgObject *),
			      cmp_distrib_name);
			left = ++right;
		}
	}
	qsort(&array[left], (len - left) + 1, sizeof(PkgObject *), cmp_distrib_name);
}

static int
cmp_category(const void *first, const void *second)
{
	char *tfirst, *tsecond;
	
	tfirst  = pkgobject_get_category(*(PkgObject **) first); 
  	tsecond = pkgobject_get_category(*(PkgObject **) second);
	return (IS_NULL(first) || IS_NULL(second)) ? (0) : strcmp(tfirst, tsecond);
}

static int
cmp_distrib_name(const void *first, const void *second)
{
	char *tfirst, *tsecond;

	tfirst  = pkgobject_get_distrib_name(*(PkgObject **) first); 
  	tsecond = pkgobject_get_distrib_name(*(PkgObject **) second);
	return (IS_NULL(first) || IS_NULL(second)) ? (0) : strcmp(tfirst, tsecond);
}
