/* $Id: search.c,v 1.5 2004/09/29 03:53:58 igor Exp $ */

/*
 * Copyright (c) 2004 Igor Boehm <igor@bytelabs.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "macros.h"
#include "pbutil.h"
#include "fail.h"
#include "pkgobject.h"
#include "interface.h"
#include "search.h"

/* PROTOTYPES */
static Boolean  find_list_elem(PkgQuery *, PkgObject *, PkgObject **);
static Boolean  find_tree_path(GtkTreeModel *, PkgObject *, GtkTreePath *, GtkTreePath **);

/**
 * Find first occurence of query and set **path pointer accordingly.
 */
Boolean
find_first(GtkTreeModel * model, PkgQuery * this, GtkTreePath ** path)
{
	PkgObject      *found;
	GtkTreePath    *start;

	if (find_list_elem(this, pkgobject_get_list_head(), &found)) {
		start = gtk_tree_path_new_first();
		find_tree_path(model, found, start, path);
		gtk_tree_path_free(start);
		return TRUE;
	} else {
		return FALSE;
	}
	return TRUE;
}

/**
 * Find next occurence of query and set **new path pointer accordingly.
 */
Boolean
find_next(GtkTreeModel * model, PkgQuery * this, GtkTreePath * old, GtkTreePath ** new)
{
	GtkTreeIter     iter;
	PkgObject      *np, *found;
	Boolean         retval;
	unsigned int    compare;

	compare = 0;
	retval = FALSE;
	gtk_tree_model_get_iter(model, &iter, old);
	gtk_tree_model_get(model, &iter, INT_COL_ID, &compare, -1);
	/* Find start of search for the list */
	for (np = pkgobject_get_list_head(); (np); np = pkgobject_get_list_next(np)) {
		if (compare == pkgobject_get_id(np)) {
			np = pkgobject_get_list_next(np);
			break;
		}
	}
	if (np && find_list_elem(this, np, &found)) {
		if (gtk_tree_model_iter_has_child(model, &iter)) {
			/* move to next child */
			gtk_tree_path_down(old);
			retval = find_tree_path(model, found, old, new);
		} else {
			if (gtk_tree_model_iter_next(model, &iter)) {
				/* move to next */
				gtk_tree_path_next(old);
				retval = find_tree_path(model, found, old, new);
			} else {
				/* move to NEXT parent */
				gtk_tree_path_up(old);
				gtk_tree_model_get_iter(model, &iter, old);
				if (gtk_tree_model_iter_next(model, &iter)) {
					gtk_tree_path_next(old);
					retval = find_tree_path(model, found, old, new);
				} else {
					/* FINITO */
					retval = FALSE;
				}
			}
		}
	}
	return retval;
}

/**
 * Helper function which does the actual tree traversal and
 * finds the path according to the search result in the list.
 */
static Boolean
find_tree_path(GtkTreeModel * model, PkgObject * this, GtkTreePath * start, GtkTreePath ** path)
{
	GtkTreeIter     iter;
	unsigned int    compare;
	Boolean         done, found;

	done = FALSE;
	found = FALSE;
	*path = gtk_tree_path_copy(start);
	gtk_tree_model_get_iter(model, &iter, start);
	while (!done) {
		/* CAN WE FIND IT ? */
		gtk_tree_model_get(model, &iter, INT_COL_ID, &compare, -1);
		if (compare == pkgobject_get_id(this)) {
			done = TRUE;
			found = TRUE;
		}
		/* NOT FOUND */
		if (!done && !gtk_tree_model_iter_has_child(model, &iter)) {
			/* THERE IS NO CHILD */
			if (gtk_tree_model_iter_next(model, &iter)) {
				/* THERE IS A NEXT ONE */
				gtk_tree_path_next(*path);
			} else {
				/* MOVE TO PARENT */
				gtk_tree_path_up(*path);
				gtk_tree_model_get_iter(model, &iter, *path);
				/* MOVE TO NEXT PARENT */
				if (gtk_tree_model_iter_next(model, &iter)) {
					/* THER IS A NEXT PARENT */
					gtk_tree_path_next(*path);
				} else {
					/* WE ARE AT THE END - FINITO */
					done = TRUE;
					found = FALSE;
				}
			}
		} else if (!done) {
			/* THERE IS A CHILD */
			gtk_tree_path_down(*path);
			gtk_tree_model_get_iter(model, &iter, *path);
		}
	}
	return found;
}


/**
 * Iterate through the list of PkgBase elements and compare PkgObject fields
 * with PkgQuery fields depending on the fields entered in the PkgQuery.
 */
static Boolean
find_list_elem(PkgQuery * query, PkgObject * startpos, PkgObject ** found)
{
	PkgObject      *np;
	Boolean         retval;
	size_t          n_entries;

	retval = FALSE;
	*found = NULL;
	if (pkgquery_get_qsall(query)) {
		for (np = startpos; (np) && (!retval); np = pkgobject_get_list_next(np)) {
			if (pkgobject_execute_query(np, query)) {
				*found = np;
				retval = TRUE;
			}
		}
	} else {
		/* SEARCH FOR VARIOUS STUFF */
		n_entries = pkgquery_get_number_of_entries(query);
		if (n_entries)
			for (np = startpos; (np) && (!retval); np = pkgobject_get_list_next(np)) {
				if (pkgobject_execute_query(np, query) == n_entries) {
					*found = np;
					retval = TRUE;
				}
			}
	}
	return retval;
}
