/* $Id: pkgobject.c,v 1.9 2005/03/13 18:36:36 igor Exp $ */

/*
 * Copyright (c) 2004 Igor Boehm <igor@bytelabs.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fail.h"
#include "macros.h"
#include "pbutil.h"
#include "scanner.h"
#include "pkgobject.h"

enum {
	PROP_0,
	PROP_ID,
	PROP_INSTALLED,
	PROP_DISTRIBNAME,
	PROP_PORTPATH,
	PROP_INSTALLPREFIX,
	PROP_COMMENT,
	PROP_DESCFILE,
	PROP_MAINTAINER,
	PROP_CATEGORY,
	PROP_BUILDDEPS,
#ifdef __OpenBSD__
	PROP_LIBDEPS,
	PROP_FORARCH,
	PROP_PACKAGECDROM,
	PROP_PACKAGEFTP,
	PROP_DISTFILESCDROM,
	PROP_DISTFILESFTP,
	PROP_FLAVOUR,
	PROP_MULTIPACKAGE,
#else
	PROP_WWW,
#endif
	PROP_RUNDEPS
};


/* PROTOTYPES */
static void     pkgobject_class_init(PkgObjectClass *);
static void     pkgobject_init(PkgObject *);
static void     pkgobject_finalize(GObject *);
static void     pkgobject_set_property(GObject *, guint, const GValue *, GParamSpec *);
static void     pkgobject_get_property(GObject *, guint, GValue *, GParamSpec *);


/* VARIABLES */
static gpointer parent_class = NULL;
/* HOW MANY PACKAGES ARE INSTALLED */
static size_t   installed = 0;
static size_t   list_size = 0;
LIST_HEAD(listhead, _PkgObject) list;	/* list holding all elements */


/**
 * Register Type.
 */
GType
pkgobject_get_type(void)
{
	static GType    pkgobject_type = 0;

	if (!pkgobject_type) {
		static const GTypeInfo pkgobject_info =
		{
			sizeof(PkgObjectClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			(GClassInitFunc) pkgobject_class_init,	/* class initialiser */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof(PkgObject),
			16,	/* n_preallocs */
			(GInstanceInitFunc) pkgobject_init,	/* instance initialiser */
			NULL
		};

		pkgobject_type = g_type_register_static(G_TYPE_OBJECT, "PkgObject",
							&pkgobject_info, 0);
	}
	return pkgobject_type;
}


/**
 * Class initialiser.
 */
static void
pkgobject_class_init(PkgObjectClass * klass)
{
	GObjectClass   *gobject_class;

	gobject_class = G_OBJECT_CLASS(klass);
	parent_class = g_type_class_peek_parent(klass);

	/* override base class methods */
	gobject_class->set_property = pkgobject_set_property;
	gobject_class->get_property = pkgobject_get_property;

	gobject_class->finalize = pkgobject_finalize;

	/* Set up properties */
	g_object_class_install_property(gobject_class,
					PROP_ID,
					g_param_spec_uint("ID",
							  "ID",
							  "Unique id.",
							  0,
							  UINT_MAX,
							  0,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_INSTALLED,
					g_param_spec_boolean("install_status",
							     "Install-Status",
			   "Indicates wheather the port is installed or not.",
							     FALSE,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_DISTRIBNAME,
					g_param_spec_string("distrib_name",
							    "Distribname",
					 "The distribution name of the port.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_PORTPATH,
					g_param_spec_string("port_path",
							    "Portpath",
				"The directory in which the port is located.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_INSTALLPREFIX,
					g_param_spec_string("install_prefix",
							    "Installprefix",
					     "The installprefix of the port.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_COMMENT,
					g_param_spec_string("comment",
							    "Comment",
					 "Short comment describing the port.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_DESCFILE,
					g_param_spec_string("descr_file",
							    "Descfile",
	"Location of the file containing the extended description for a port.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_MAINTAINER,
					g_param_spec_string("maintainer",
							    "Maintainer",
						"The maintainer of the port.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_CATEGORY,
					g_param_spec_string("category",
							    "Category",
				    "The category in which the port resides.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_BUILDDEPS,
					g_param_spec_string("bdep",
							    "Build-depends",
							    "The dependencies which have to be satisfied in order to build the port.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_RUNDEPS,
					g_param_spec_string("rdep",
							    "Rund-depends",
							    "The dependencies which have to be satisfied in order to run the port.",
							    NULL,
							  G_PARAM_READWRITE));
#ifdef __OpenBSD__
	g_object_class_install_property(gobject_class,
					PROP_LIBDEPS,
					g_param_spec_string("ldep",
							    "Library-depends",
							    "The librarby dependencies which have to be satisfied in order to run the port.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_FORARCH,
					g_param_spec_string("forarch",
							    "Forarch",
		   "Specification of architectures on which this port works.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_PACKAGECDROM,
					g_param_spec_string("cdrom",
							    "Package-cdrom",
							    "NONE.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_PACKAGEFTP,
					g_param_spec_string("ftp",
							    "Package-ftp",
							    "NONE.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_DISTFILESFTP,
					g_param_spec_string("distftp",
							    "Dist-ftp",
							    "NONE.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_DISTFILESCDROM,
					g_param_spec_string("distcdrom",
							    "Dist-cdrom",
							    "NONE.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_FLAVOUR,
					g_param_spec_string("flav",
							    "Flavor",
				 "Specific options provided by a given port.",
							    NULL,
							  G_PARAM_READWRITE));
	g_object_class_install_property(gobject_class,
					PROP_MULTIPACKAGE,
					g_param_spec_string("multi",
							    "Multipackage",
							    "This mechanism is used when a larger package is broken down into several smaller components referred to as subpackages.",
							    NULL,
							  G_PARAM_READWRITE));
#else
	g_object_class_install_property(gobject_class,
					PROP_WWW,
					g_param_spec_string("www",
							    "WWW",
						  "The homepage of the port.",
							    NULL,
							  G_PARAM_READWRITE));
#endif
}

/**
 * Instance initialiser.
 */
static void
pkgobject_init(PkgObject * obj)
{
	obj->next.le_next = NULL;
}

/**
 * Finalize instance.
 */
static void
pkgobject_finalize(GObject * obj)
{
	PkgObject      *pkg;

	pkg = PKGOBJECT(obj);
	FREE(pkg->distrib_name);
	FREE(pkg->port_path);
	FREE(pkg->install_prefix);
	FREE(pkg->comment);
	FREE(pkg->descr_file);
	FREE(pkg->maintainer);
	FREE(pkg->category);
	FREE(pkg->bdep);
	FREE(pkg->rdep);
#ifdef __OpenBSD__
	FREE(pkg->ldep);
	FREE(pkg->forarch);
	FREE(pkg->cdrom);
	FREE(pkg->ftp);
	FREE(pkg->distcdrom);
	FREE(pkg->distftp);
	FREE(pkg->flav);
	FREE(pkg->multi);
#else
	FREE(pkg->www);
#endif
	/* DONT FORGET TO CALL PARENTS FINALIZE */
	G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
pkgobject_set_property(GObject * obj, guint prop_id, const GValue * value, GParamSpec * pspec)
{
	PkgObject      *pkg;

	pkg = PKGOBJECT(obj);
	switch (prop_id) {
	case PROP_ID:
		pkgobject_set_id(pkg, (unsigned int) g_value_get_uint(value));
		break;
	case PROP_INSTALLED:
		pkgobject_set_install_status(pkg, (unsigned int) g_value_get_boolean(value));
		break;
	case PROP_DISTRIBNAME:
		pkgobject_set_distrib_name(pkg, g_value_get_string(value));
		break;
	case PROP_PORTPATH:
		pkgobject_set_port_path(pkg, g_value_get_string(value));
		break;
	case PROP_INSTALLPREFIX:
		pkgobject_set_install_prefix(pkg, g_value_get_string(value));
		break;
	case PROP_COMMENT:
		pkgobject_set_comment(pkg, g_value_get_string(value));
		break;
	case PROP_DESCFILE:
		pkgobject_set_descr_path(pkg, g_value_get_string(value));
		break;
	case PROP_MAINTAINER:
		pkgobject_set_maintainer(pkg, g_value_get_string(value));
		break;
	case PROP_CATEGORY:
		pkgobject_set_category(pkg, g_value_get_string(value));
		break;
	case PROP_BUILDDEPS:
		pkgobject_set_build_depends(pkg, g_value_get_string(value));
		break;
	case PROP_RUNDEPS:
		pkgobject_set_run_depends(pkg, g_value_get_string(value));
		break;
#ifdef __OpenBSD__
	case PROP_LIBDEPS:
		pkgobject_set_library_depends(pkg, g_value_get_string(value));
		break;
	case PROP_FORARCH:
		pkgobject_set_forarch(pkg, g_value_get_string(value));
		break;
	case PROP_PACKAGECDROM:
		pkgobject_set_cdrom(pkg, g_value_get_string(value));
		break;
	case PROP_PACKAGEFTP:
		pkgobject_set_ftp(pkg, g_value_get_string(value));
		break;
	case PROP_DISTFILESCDROM:
		pkgobject_set_distfiles_cdrom(pkg, g_value_get_string(value));
		break;
	case PROP_DISTFILESFTP:
		pkgobject_set_distfiles_ftp(pkg, g_value_get_string(value));
		break;
	case PROP_FLAVOUR:
		pkgobject_set_flavour(pkg, g_value_get_string(value));
		break;
	case PROP_MULTIPACKAGE:
		pkgobject_set_multipackage(pkg, g_value_get_string(value));
		break;
#else
	case PROP_WWW:
		pkgobject_set_homepage(pkg, g_value_get_string(value));
		break;
#endif
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, prop_id, pspec);
		break;
	}
}

static void
pkgobject_get_property(GObject * obj, guint prop_id, GValue * value, GParamSpec * pspec)
{
	PkgObject      *pkg;

	pkg = PKGOBJECT(obj);
	switch (prop_id) {
	case PROP_ID:
		g_value_set_uint(value, pkg->ID);
		break;
	case PROP_INSTALLED:
		g_value_set_boolean(value, (int) pkg->install_status);
		break;
	case PROP_DISTRIBNAME:
		g_value_set_string(value, pkg->distrib_name);
		break;
	case PROP_PORTPATH:
		g_value_set_string(value, pkg->port_path);
		break;
	case PROP_INSTALLPREFIX:
		g_value_set_string(value, pkg->install_prefix);
		break;
	case PROP_COMMENT:
		g_value_set_string(value, pkg->comment);
		break;
	case PROP_DESCFILE:
		g_value_set_string(value, pkg->descr_file);
		break;
	case PROP_MAINTAINER:
		g_value_set_string(value, pkg->maintainer);
		break;
	case PROP_CATEGORY:
		g_value_set_string(value, pkg->category);
		break;
	case PROP_BUILDDEPS:
		g_value_set_string(value, pkg->bdep);
		break;
	case PROP_RUNDEPS:
		g_value_set_string(value, pkg->rdep);
		break;
#ifdef __OpenBSD__
	case PROP_LIBDEPS:
		g_value_set_string(value, pkg->ldep);
		break;
	case PROP_FORARCH:
		g_value_set_string(value, pkg->forarch);
		break;
	case PROP_PACKAGECDROM:
		g_value_set_string(value, pkg->cdrom);
		break;
	case PROP_PACKAGEFTP:
		g_value_set_string(value, pkg->ftp);
		break;
	case PROP_DISTFILESCDROM:
		g_value_set_string(value, pkg->distcdrom);
		break;
	case PROP_DISTFILESFTP:
		g_value_set_string(value, pkg->distftp);
		break;
	case PROP_FLAVOUR:
		g_value_set_string(value, pkg->flav);
		break;
	case PROP_MULTIPACKAGE:
		g_value_set_string(value, pkg->multi);
		break;
#else
	case PROP_WWW:
		g_value_set_string(value, pkg->www);
		break;
#endif
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, prop_id, pspec);
		break;
	}
}

/**
 * Clone a PkgObject -> make an exact copy of it.
 */
PkgObject      *
pkgobject_clone(PkgObject * obj)
{
	PkgObject      *new;

	new = g_object_new(TYPE_PKGOBJECT, NULL);
	if (pkgobject_get_id(obj))
		g_object_set(new, "ID", pkgobject_get_id(obj), NULL);
	g_object_set(new, "install-status", pkgobject_get_install_status(obj), NULL);
	if (pkgobject_get_distrib_name(obj))
		g_object_set(new, "distrib-name", pkgobject_get_distrib_name(obj), NULL);
	if (pkgobject_get_port_path(obj))
		g_object_set(new, "port-path", pkgobject_get_port_path(obj), NULL);
	if (pkgobject_get_install_prefix(obj))
		g_object_set(new, "install-prefix", pkgobject_get_install_prefix(obj), NULL);
	if (pkgobject_get_comment(obj))
		g_object_set(new, "comment", pkgobject_get_comment(obj), NULL);
	if (pkgobject_get_descr_path(obj))
		g_object_get(new, "descr-file", pkgobject_get_descr_path(obj), NULL);
	if (pkgobject_get_maintainer(obj))
		g_object_set(new, "maintainer", pkgobject_get_maintainer(obj), NULL);
	if (pkgobject_get_category(obj))
		g_object_set(new, "category", pkgobject_get_category(obj), NULL);
	if (pkgobject_get_build_depends(obj))
		g_object_set(new, "bdep", pkgobject_get_build_depends(obj), NULL);
	if (pkgobject_get_run_depends(obj))
		g_object_set(new, "rdep", pkgobject_get_run_depends(obj), NULL);
#ifdef __OpenBSD__
	if (pkgobject_get_library_depends(obj))
		g_object_set(new, "ldep", pkgobject_get_library_depends(obj), NULL);
	if (pkgobject_get_forarch(obj))
		g_object_set(new, "forarch", pkgobject_get_forarch(obj), NULL);
	if (pkgobject_get_cdrom(obj))
		g_object_set(new, "cdrom", pkgobject_get_cdrom(obj), NULL);
	if (pkgobject_get_ftp(obj))
		g_object_set(new, "ftp", pkgobject_get_ftp(obj), NULL);
	if (pkgobject_get_distfiles_cdrom(obj))
		g_object_set(new, "distcdrom", pkgobject_get_distfiles_cdrom(obj), NULL);
	if (pkgobject_get_distfiles_ftp(obj))
		g_object_set(new, "distftp", pkgobject_get_distfiles_ftp(obj), NULL);
	if (pkgobject_get_flavour(obj))
		g_object_set(new, "flav", pkgobject_get_flavour(obj), NULL);
	if (pkgobject_get_multipackage(obj))
		g_object_set(new, "multi", pkgobject_get_multipackage(obj), NULL);
#else
	if (pkgobject_get_homepage(obj))
		g_object_set(new, "www", pkgobject_get_homepage(obj), NULL);
#endif
	return new;
}


Boolean
pkgobject_get_install_status(PkgObject * obj)
{
	return obj->install_status;
}

unsigned int
pkgobject_get_id(PkgObject * obj)
{
	return obj->ID;
}

Boolean
pkgobject_get_descr_content(PkgObject * obj, char *str, size_t siz)
{
	FILE           *fileptr;
	size_t          pos;
	Boolean         retval;
	int             c;
#ifdef __OpenBSD__
	char            fullpath[_POSIX_PATH_MAX] = PORTS_DIR;
#else
	char            fullpath[_POSIX_PATH_MAX] = "";
#endif

	pos = 0;
	retval = FALSE;
	STRLCAT(fullpath, pkgobject_get_descr_path(obj), sizeof(fullpath));
	if (!(fileptr = fopen(fullpath, "r"))) {
		pb_warning("%s: %s", strerror(errno), fullpath);
		retval = FALSE;
	} else {
		while (((c = getc(fileptr)) != EOF) && (pos < siz)) {
			if (isascii(c)) {
				*str++ = c;
				pos++;
			}
		}
		/* don't forget to terminate with '\0' */
		*str = '\0';
		fclose(fileptr);
		retval = TRUE;
	}
	return retval;
}

Boolean
pkgobject_get_plist_content(PkgObject * obj, char *str, size_t siz)
{
	FILE           *fileptr;
	size_t          pos;
	Boolean         retval;
	int             c;
	char            fullpath[_POSIX_PATH_MAX];

	pos = 0;
	retval = FALSE;
	snprintf(fullpath, sizeof(fullpath),
#ifdef __OpenBSD__
		 "%s%s%s",
		 PORTS_DIR,
#else
		 "%s%s",
#endif
		 pkgobject_get_port_path(obj),
		 PLISTFILE);
#ifdef __OpenBSD__
	if (pkgobject_get_multipackage(obj))
		STRLCAT(fullpath, pkgobject_get_multipackage(obj), sizeof(fullpath));
#endif
	if (!(fileptr = fopen(fullpath, "r"))) {
#ifdef __OpenBSD__
		/*
		 * XXX: In FreeBSD PLIST stuff may also be defined in the
		 * Makefile instead, so if we don't find the PLIST File we
		 * don't panic -> this still needs to be implemented.
		 */
		pb_warning("%s: %s", strerror(errno), fullpath);
#endif				/* __OpenBSD__ */
		retval = FALSE;
	} else {
		while (((c = getc(fileptr)) != EOF) && (pos < siz)) {
			if (c != '%') {
				if (isascii(c)) {
					*str++ = c;
					pos++;
				}
			} else {
				/* skip lines starting with '%' */
				for (c = getc(fileptr);; c = getc(fileptr))
					if (c == '\n' || c == EOF)
						break;
			}
		}
		/* don't forget to terminate with '\0' */
		*str = '\0';
		fclose(fileptr);
		retval = TRUE;
	}
	return retval;
}

char           *
pkgobject_get_distrib_name(PkgObject * obj)
{
	return obj->distrib_name;
}

char           *
pkgobject_get_port_path(PkgObject * obj)
{
	return obj->port_path;
}

char           *
pkgobject_get_install_prefix(PkgObject * obj)
{
	return obj->install_prefix;
}

char           *
pkgobject_get_comment(PkgObject * obj)
{
	return obj->comment;
}

char           *
pkgobject_get_maintainer(PkgObject * obj)
{
	return obj->maintainer;
}

char           *
pkgobject_get_category(PkgObject * obj)
{
	return obj->category;
}

char           *
pkgobject_get_build_depends(PkgObject * obj)
{
	return obj->bdep;
}

char           *
pkgobject_get_run_depends(PkgObject * obj)
{
	return obj->rdep;
}

char           *
pkgobject_get_descr_path(PkgObject * obj)
{
	return obj->descr_file;
}

void
pkgobject_set_install_status(PkgObject * obj, Boolean stat)
{
	obj->install_status = stat;
}

void
pkgobject_set_id(PkgObject * obj, unsigned int id)
{
	obj->ID = id;
}

void
pkgobject_set_distrib_name(PkgObject * obj, const char *str)
{
	FREE(obj->distrib_name);
	obj->distrib_name = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_port_path(PkgObject * obj, const char *str)
{
	FREE(obj->port_path);
	obj->port_path = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_install_prefix(PkgObject * obj, const char *str)
{
	FREE(obj->install_prefix);
	obj->install_prefix = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_comment(PkgObject * obj, const char *str)
{
	FREE(obj->comment);
	obj->comment = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_maintainer(PkgObject * obj, const char *str)
{
	FREE(obj->maintainer);
	obj->maintainer = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_category(PkgObject * obj, const char *str)
{
	FREE(obj->category);
	obj->category = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_build_depends(PkgObject * obj, const char *str)
{
	FREE(obj->bdep);
	obj->bdep = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_run_depends(PkgObject * obj, const char *str)
{
	FREE(obj->rdep);
	obj->rdep = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_descr_path(PkgObject * obj, const char *str)
{
	FREE(obj->descr_file);
	obj->descr_file = (str) ? strdup(str) : NULL;
}



/* OpenBSD specific getter|setter functions */
#ifdef __OpenBSD__

char           *
pkgobject_get_library_depends(PkgObject * obj)
{
	return obj->ldep;
}

char           *
pkgobject_get_forarch(PkgObject * obj)
{
	return obj->forarch;
}

char           *
pkgobject_get_cdrom(PkgObject * obj)
{
	return obj->cdrom;
}

char           *
pkgobject_get_ftp(PkgObject * obj)
{
	return obj->ftp;
}

char           *
pkgobject_get_distfiles_cdrom(PkgObject * obj)
{
	return obj->distcdrom;
}

char           *
pkgobject_get_distfiles_ftp(PkgObject * obj)
{
	return obj->distftp;
}

char           *
pkgobject_get_flavour(PkgObject * obj)
{
	return obj->flav;
}

char           *
pkgobject_get_multipackage(PkgObject * obj)
{
	return obj->multi;
}

void
pkgobject_set_library_depends(PkgObject * obj, const char *str)
{
	FREE(obj->ldep);
	obj->ldep = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_forarch(PkgObject * obj, const char *str)
{
	FREE(obj->forarch);
	obj->forarch = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_cdrom(PkgObject * obj, const char *str)
{
	FREE(obj->cdrom);
	obj->cdrom = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_ftp(PkgObject * obj, const char *str)
{
	FREE(obj->ftp);
	obj->ftp = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_distfiles_cdrom(PkgObject * obj, const char *str)
{
	FREE(obj->distcdrom);
	obj->distcdrom = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_distfiles_ftp(PkgObject * obj, const char *str)
{
	FREE(obj->distftp);
	obj->distftp = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_flavour(PkgObject * obj, const char *str)
{
	FREE(obj->flav);
	obj->flav = (str) ? strdup(str) : NULL;
}

void
pkgobject_set_multipackage(PkgObject * obj, const char *str)
{
	FREE(obj->multi);
	obj->multi = (str) ? strdup(str) : NULL;
}

#else				/* __OpenBSD__ */

char           *
pkgobject_get_homepage(PkgObject * obj)
{
	return obj->www;
}

void
pkgobject_set_homepage(PkgObject * obj, const char *str)
{
	FREE(obj->www);
	obj->www = (str) ? strdup(str) : NULL;
}

#endif

void
pkgobject_print(PkgObject * obj)
{
	pb_message("---- PkgObject element START ----");
	pb_message(" ID: %d", pkgobject_get_id(obj));
	pb_message(" INSTALLED: %s", (pkgobject_get_install_status(obj) ? "YES" : "NO"));
	pb_message(" NAME: %s", pkgobject_get_distrib_name(obj));
	pb_message(" DIRECTORY: %s", pkgobject_get_port_path(obj));
	pb_message(" INSTALLPREFIX: %s", pkgobject_get_install_prefix(obj));
	pb_message(" COMMENT: %s", pkgobject_get_comment(obj));
	pb_message(" DESCFILE: %s", pkgobject_get_descr_path(obj));
	pb_message(" MAINTAINER: %s", pkgobject_get_maintainer(obj));
	pb_message(" CATEGORY: %s", pkgobject_get_category(obj));
	pb_message(" R-DEPS: %s", pkgobject_get_run_depends(obj));
	pb_message(" B-DEPS: %s", pkgobject_get_build_depends(obj));
#ifdef __OpenBSD__
	pb_message(" L-DEPS: %s", pkgobject_get_library_depends(obj));
	pb_message(" FORARCH: %s", pkgobject_get_forarch(obj));
	pb_message(" PACKAGECDROM: %s", pkgobject_get_cdrom(obj));
	pb_message(" PACKAGEFTP: %s", pkgobject_get_ftp(obj));
	pb_message(" DISTFILESCDROM: %s", pkgobject_get_distfiles_cdrom(obj));
	pb_message(" DISTFILESFTP: %s", pkgobject_get_distfiles_ftp(obj));
	pb_message(" FLAVOR: %s", pkgobject_get_flavour(obj));
	pb_message(" MULTIPACKAGE: %s", pkgobject_get_multipackage(obj));
#else
	pb_message(" HTTP-URL: %s", pkgobject_get_homepage(obj));
#endif
	pb_message("----- PkgObject element END -----");

}

Boolean
pkgobject_parse_package_list(void)
{
	return parse_index(INDEXFILE);
}

size_t
pkgobject_get_number_installed_packages(void)
{
	return installed;
}

void
pkgobject_increment_installed_packages(void)
{
	installed++;
}

void
pkgobject_decrement_installed_packages(void)
{
	installed--;
}

Boolean
pkgobject_is_list_empty(void)
{
	return LIST_EMPTY(&list);
}

PkgObject      *
pkgobject_get_list_head(void)
{
	return LIST_FIRST(&list);
}

void
pkgobject_destroy_list(void)
{
	PkgObject      *np;

	for (np = LIST_FIRST(&list); (!np); np = LIST_NEXT(np, next))
		g_object_unref(np);
	list_size = 0;
}

void
pkgobject_create_list(size_t len)
{
	PkgObject      *obj;
	size_t          i;

	list_size = len;
	LIST_INIT(&list);
	for (i = 0; i < len; i++) {
		obj = g_object_new(TYPE_PKGOBJECT, NULL);
		LIST_INSERT_HEAD(&list, obj, next);
	}

}

size_t
pkgobject_get_list_size(void)
{
	return list_size;
}

PkgObject      *
pkgobject_get_list_next(PkgObject * obj)
{
	return LIST_NEXT(obj, next);
}

void
pkgobject_list_insert_head(PkgObject * obj)
{
	LIST_INSERT_HEAD(&list, obj, next);
	obj->next.le_next = NULL;
}

void
pkgobject_list_insert_after(PkgObject * obj1, PkgObject * obj2)
{
	LIST_INSERT_AFTER(obj1, obj2, next);
	obj2->next.le_next = NULL;
}

/**
 * This function checks if a package is installed.
 */
Boolean
pkgobject_check_install_status(PkgObject * elem)
{
	struct dirent  *dp;
	DIR            *dirptr;
	Boolean         found;

	found = FALSE;
	if (!(dirptr = opendir(DBDIR))) {
		pb_warning("%s: %s", strerror(errno), DBDIR);
		return FALSE;
	}
	while ((dp = readdir(dirptr)) && !found) {
		if (strcmp(dp->d_name, ".") == 0 ||
		    strcmp(dp->d_name, "..") == 0) {
			continue;
		}
		if (!strcmp(pkgobject_get_distrib_name(elem), dp->d_name))
			found = TRUE;
	}
	closedir(dirptr);
	return found;
}

/**
 * If nothing matches 0 is returned -> else the number of
 * matches is returned.
 */
size_t
pkgobject_execute_query(PkgObject * obj, PkgQuery * query)
{
	Boolean         q_case;
	size_t          q_result;
	char            buf[_POSIX_SSIZE_MAX];
	char           *sall;

	q_result = 0;
	q_case = pkgquery_is_case_sensitive(query);
	sall = pkgquery_get_qsall(query);
	if (sall) {
		/* SEARCH ALL */
		if (pb_compare(pkgobject_get_distrib_name(obj),
			       sall, q_case))
			q_result++;
		if (pb_compare(pkgobject_get_comment(obj),
			       sall, q_case))
			q_result++;
		if (pkgquery_get_qedesc(query)) {
			/* SOMEONE WANTS TO SEARCH IN DESCR FILES */
			pkgobject_get_descr_content(obj, buf, sizeof(buf));
			if (pb_compare(buf, sall, q_case))
				q_result++;
		}
		if (pb_compare(pkgobject_get_run_depends(obj),
			       sall, q_case))
			q_result++;
		if (pb_compare(pkgobject_get_build_depends(obj),
			       sall, q_case))
			q_result++;
#ifdef __OpenBSD__
		if (pb_compare(pkgobject_get_library_depends(obj),
			       sall, q_case))
			q_result++;
#endif				/* __OpenBSD__ */

	} else {
		if (pb_compare(pkgobject_get_distrib_name(obj),
			       pkgquery_get_qname(query), q_case))
			q_result++;
		if (pb_compare(pkgobject_get_comment(obj),
			       pkgquery_get_qcomment(query), q_case))
			q_result++;
		if (pkgquery_get_qedesc(query)) {
			/* SOMEONE WANTS TO SEARCH IN DESCR FILES */
			pkgobject_get_descr_content(obj, buf, sizeof(buf));
			if (pb_compare(buf, pkgquery_get_qedesc(query), q_case))
				q_result++;
		}
		if (pb_compare(pkgobject_get_run_depends(obj),
			       pkgquery_get_qrdep(query), q_case))
			q_result++;
		if (pb_compare(pkgobject_get_build_depends(obj),
			       pkgquery_get_qbdep(query), q_case))
			q_result++;
#ifdef __OpenBSD__
		if (pb_compare(pkgobject_get_library_depends(obj),
			       pkgquery_get_qldep(query), q_case))
			q_result++;
#endif				/* __OpenBSD__ */
	}
	return q_result;
}
