/* $Id: gram.y,v 1.18 2004/10/20 07:18:50 igor Exp $ */

/*
 * Copyright (c) 2004 Igor Boehm <igor@bytelabs.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

%{

#include <sys/types.h>

#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "fail.h"
#include "macros.h"
#include "pkgobject.h"
#include "y.tab.h"
#include "scanner.h"
#include "sort.h"

/* PROTOTYPES */
static void     nextcontent(void);
static void     nextentry(void);
static void     init(char *);
static size_t   linecount(char *);
static Boolean  assign(char *);
int             yyerror(const char *,...);
int             yyparse(void);
int             yylex(void);

/* in here we store all the info per Package */
static int      curpos = 0;
static int      lineno = 0;
static size_t   id = 0;
PkgObject      *elem;
char           *infile;



%}

/* our tokens which we have defined in scan.l */
%token SIMPLEWORD DELIM EOL EF

%%

/*
 *  GRAMMER for OpenBSD
 *
 *  SYNTAX:
 *           distribution-name|port-path|installation-prefix|comment| \
 *           description-file|maintainer|categories|lib-deps|build-deps| \
 *           run-deps|for-arch|package-cdrom|package-ftp|distfiles-cdrom| \
 *           distfiles-ftp \n
 *
 *  CAVEATS: Most definitelty we need to take special care of Multipackages
 *           and or Multiflavors! The next thing we need to take care of are
 *           packages which fall into more than one category - in such a case
 *           we will grab the FIRST category and IGNORE the rest!
 *---------------------------------------------------------------------
 *
 * GRAMMER for FreeBSD
 *
 * SYNTAX:
 * 			 distribution-name|port-path|installation-prefix|comment| \
 *  		 description-file|maintainer|categories|build deps|run deps \
 *  		 www site \n
 *
 * CAVEATS:  I don't have a FreeBSD System.
 */


entries:
	|
	entries entry
	;

entry:
	content EOL
	{
	  nextentry();		/* we start over again */
	}
	|
	content DELIM entry
	;

content:
	SIMPLEWORD
	{
	  assign((char *) $1);
	  nextcontent();
	}
	|
	/* EMPTY */
	{
	  nextcontent();
	}
	;


%%

/**
 * Count the number of lines of our INDEX file so that we know
 * how much space we are going to need to our package array.
 */
static size_t
linecount(char *file)
{
	char            buf[MAXBSIZE];
	char           *c;
	int             len, fd;
	size_t          linect;

	linect = 0;
	if ((fd = open(file, O_RDONLY, 0)) < 0) {
		pb_warning(strerror(errno));
	} else {
		while ((len = read(fd, buf, MAXBSIZE)) > 0) {
			for (c = buf; len--; ++c)
				if (*c == '\n')
					++linect;
		}
		if (len == -1)
			pb_warning(strerror(errno));
	}
	if (close(fd))
		pb_warning(strerror(errno));
	if (pb_get_verbose_level())
		pb_message("%d Ports available.", linect);
	return linect;
}


/*
 * Function which opens the ports INDEX file and calls yyparse()
 * to begin the parsing process.
 */
Boolean
parse_index(char *filename)
{
	extern FILE    *yyin;

	if (IS_NULL((yyin = fopen(filename, "r")))) {
		pb_error(1, "%s: %s", strerror(errno), filename);
		return FALSE;
	}
	init(filename);
	infile = strdup(filename);
	yyparse();
	FREE(infile);
	fclose(yyin);
	/* SORT IT ASCENDING AND BASED ON CATEGORY */
	pb_pkglist_sort((pkgobject_get_list_size() - 1));
	return TRUE;
}

int
yyerror(const char *fmt,...)
{
	va_list         ap;
	char           *nfmt;

	va_start(ap, fmt);
	if (asprintf(&nfmt, "%s:%d: %s", infile, lineno, fmt) == -1)
		pb_error(1, "yyerror asprintf");
	va_end(ap);
	free(nfmt);
	return (0);
}


/**
 * This function SHOULD ONLY BE CALLED ONCE. It initialises the LIST
 * and allocates the first PkgBase structure.
 */
static void
init(char *file)
{
	pkgobject_create_list(linecount(file));
	curpos = 0;		/* just to be sure we reset the counter */
	elem = pkgobject_get_list_head();
}

/*
 * Reset the CONTENT position curpos to 1 again, add the old ENTRY to the
 * list and ALLOCATE more memory for a new ENTRY.
 */
static void
nextentry(void)
{
	/* reset the curpos counter */
	curpos = 0;
	lineno++;
	g_object_set(elem,
		     "install-status", pkgobject_check_install_status(elem),
		     NULL);
	if (pkgobject_get_install_status(elem))
		pkgobject_increment_installed_packages();
	g_object_set(elem,
		     "ID", id++,
		     NULL);
	if (pb_get_verbose_level() > 1)
		pkgobject_print(elem);
	elem = pkgobject_get_list_next(elem);
}

/**
 * XXX: this is not really nice but more may come in here
 *      in the future and that's why this function exists.
 */
static void
nextcontent(void)
{
	curpos++;
}

/**
 * Depending on where the parsing actually is located
 * we assign the right values to our PkgObject.
 */
static Boolean
assign(char *str)
{
	char           *ptr, *nextptr;
	Boolean         retval;

	retval = TRUE;
	if ((curpos == P_BUILDDEPS)
	    | (curpos == P_RUNDEPS)
#ifdef __OpenBSD__
	    | (curpos == P_LIBDEPS)
#endif				/* __OpenBSD__ */
		) {
		ptr = str;
		/* REPLACE (' ' or \t) WITH \n */
		while (!IS_NULL((nextptr = strchr(ptr, ' '))) ||
		       !IS_NULL((nextptr = strchr(ptr, '\t')))) {
			*nextptr++ = '\n';
			ptr = nextptr;
		}
	}
	/*
	 * XXX: BE CAREFUL WITH MULTIPLE CATEGORIES for right now we only take
	 * the first category a package falls into -> but there may be more
	 * than one For right now we also only take major categories, so if we
	 * find something like x11/xfce we will only grab the x11 part.
	 */
	if (curpos == P_CATEGORY) {
		/* deal with multiple categories */
		if (!IS_NULL((ptr = strchr(str, ' '))) ||
		    !IS_NULL((ptr = strchr(str, '\t'))))
			*ptr++ = '\0';
		if (!IS_NULL((ptr = strchr(str, '/'))))
			*ptr++ = '\0';
	}
#ifdef __OpenBSD__
	/*
	 * XXX: Handle MULTIPACKAGES and FLAVOURS. Both, MULTIPACKAGES and
	 * FLAVOURS, are separated by commas where the only difference between
	 * them is that MULTIPACKAGES are preceded by a dash '-' and FLAVOURS
	 * aren't.
	 */
	if ((curpos == P_PORTPATH) && !IS_NULL((ptr = strchr(str, ',')))) {
		*ptr++ = '\0';
		nextptr = ptr;
		while (!IS_NULL(nextptr)) {
			ptr = strsep(&nextptr, ",");
			if (!IS_NULL(strchr(ptr, '-'))) {
				/* MULTIPACKAGE */
				g_object_set(elem, "multi", ptr, NULL);
			} else {
				/* FLAVOUR      */
				g_object_set(elem, "flav", ptr, NULL);
			}

		}
	}
#endif				/* __OpenBSD__ */

	switch (curpos) {
	case P_DISTRIBNAME:
		g_object_set(elem, "distrib-name", str, NULL);
		break;
	case P_PORTPATH:
		g_object_set(elem, "port-path", str, NULL);
		break;
	case P_INSTALLPREFIX:
		g_object_set(elem, "install-prefix", str, NULL);
		break;
	case P_COMMENT:
		g_object_set(elem, "comment", str, NULL);
		break;
	case P_DESCFILE:
		g_object_set(elem, "descr-file", str, NULL);
		break;
	case P_MAINTAINER:
		g_object_set(elem, "maintainer", str, NULL);
		break;
	case P_CATEGORY:
		g_object_set(elem, "category", str, NULL);
		break;
	case P_BUILDDEPS:
		g_object_set(elem, "bdep", str, NULL);
		break;
	case P_RUNDEPS:
		g_object_set(elem, "rdep", str, NULL);
		break;
#ifdef __OpenBSD__
	case P_LIBDEPS:
		g_object_set(elem, "ldep", str, NULL);
		break;
	case P_FORARCH:
		g_object_set(elem, "forarch", str, NULL);
		break;
	case P_PACKAGECDROM:
		g_object_set(elem, "cdrom", str, NULL);
		break;
	case P_PACKAGEFTP:
		g_object_set(elem, "ftp", str, NULL);
		break;
	case P_DISTFILESCDROM:
		g_object_set(elem, "distcdrom", str, NULL);
		break;
	case P_DISTFILESFTP:
		g_object_set(elem, "distftp", str, NULL);
		break;
	case P_FLAVOUR:
		/* THIS HAS ALREADY BEEN TAKEN CARE OF */
		break;
	case P_MULTIPACKAGE:
		/* THIS HAS ALREADY BEEN TAKEN CARE OF  */
		break;
#else
	case P_WWW:
		g_object_set(elem, "www", str, NULL);
		break;
#endif
	default:
		retval = FALSE;
		yyerror("Invalid position in parsing process.");
	}
	return retval;
}
