/* $Id: callbacks.c,v 1.28 2004/11/13 12:42:10 igor Exp $ */

/*
 * Copyright (c) 2004 Igor Boehm <igor@bytelabs.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF MIND, USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "fail.h"
#include "macros.h"
#include "pbutil.h"
#include "pkgobject.h"
#include "pkgcmd.h"
#include "gtkutil.h"
#include "search.h"
#include "interface.h"
#include "callbacks.h"

/* STRUCT PASSED TO THREADED FUNCTIONS */
struct _ThreadData {
	PBWindow       *pb;
	GtkWidget      *pbarwin;
	GtkTextView    *view;
	GtkButton      *bclose;
	GtkButton      *bcancel;
	InstallParams  *param;
	Boolean         canceled;
};


/* PROTOTYPES */
static void     search_clear_entry(GtkWidget *, gpointer);
static void     search_exec_query(GtkWidget *, gpointer);
static Boolean  search_get_query(PBSearchWindow *);

static void     pb_thread_data_init(struct _ThreadData **);
static void     pb_thread_data_destroy(struct _ThreadData **);
static Boolean  pb_update_ui_entries(PkgObject *, PBWindow *);
static void     kill_async_cmd(gpointer data);
static gpointer exec_async_cmd(gpointer data);
static gboolean exec_async_cmd_done(gpointer data);

static Boolean  treeview_get_selection(PBWindow *, PkgObject **);

/**
 * Callback function for the delete_event.
 */
int
delete_event(GtkWidget *source, GdkEvent event, gpointer data)
{
	return FALSE;
}

/**
 * Callback function for quitting the app.
 */
void
end_event(GtkWidget *source, gpointer data)
{
	gtk_main_quit();
}

/**
 * Callback function for expanding all Nodes in the treeview.
 */
void
treeview_expand(GtkWidget *source, gpointer data)
{
	GtkTreeView    *view;
	GtkTreePath    *path;
	PBWindow       *pb;
	GList          *listptr;

	pb = (PBWindow *) data;
	/* EXTRACT THE TREEVIEW FROM PARENT SCROLLEDWINDOW */
	listptr = g_list_first(gtk_container_get_children(GTK_CONTAINER(
							  pb->scr_win_tree)));
	view = GTK_TREE_VIEW((listptr)->data);
	gtk_tree_view_expand_all(view);
	/* SELECT FIRST ENTRY */
	path = gtk_tree_path_new_first();
	gtk_tree_view_set_cursor(view, path, NULL, FALSE);
	FREE_PATH(path);
	g_list_free(listptr);
}

/**
 * Callback function for collapsing all Nodes in the treeview.
 */
void
treeview_collapse(GtkWidget *source, gpointer data)
{
	GtkTreeView    *view;
	GtkTreePath    *path;
	PBWindow       *pb;
	GList          *listptr;

	pb = (PBWindow *) data;
	/* EXTRACT THE TREEVIEW FROM PARENT SCROLLEDWINDOW */
	listptr = g_list_first(gtk_container_get_children(GTK_CONTAINER(
							  pb->scr_win_tree)));
	view = GTK_TREE_VIEW(listptr->data);
	gtk_tree_view_collapse_all(view);
	/* SELECT FIRST ENTRY */
	path = gtk_tree_path_new_first();
	gtk_tree_view_set_cursor(view, path, NULL, FALSE);
	FREE_PATH(path);
	g_list_free(listptr);
}

/**
 * Callback function which creates a window where one can edit
 * the PKG_PATH.
 */
void
path_create_widget(GtkWidget *source, gpointer data)
{
	PBWindow       *pb;
	GtkWidget      *path_dialog;
	int             res;

	pb = (PBWindow *) data;
	path_dialog = pb_create_path_dialog(pb->main_window);
	res = gtk_dialog_run(GTK_DIALOG(path_dialog));
	if ((res == GTK_RESPONSE_REJECT) || (res == GTK_RESPONSE_OK))
		gtk_widget_destroy(path_dialog);
}

/**
 * Callback function which creates the search window if not
 * already openend.
 */
void
search_create_widget(GtkWidget *source, gpointer data)
{
	PBWindow       *pb;
	PBSearchWindow *pbs;

	pb = (PBWindow *) data;
	pbs = pb->search_window;
	if (!pb->search_active) {
		pb->search_active = TRUE;
		pbs->search_window = pb_create_search_window(pb->main_window,
							     &pbs->entry,
							     &pbs->tog_case,
							     &pbs->b_search,
							     &pbs->b_clear,
							     &pbs->b_close,
							     &pbs->omenu,
							     "Ports Search");
		/* XXX: DONT FORGET TO FREE THIS IN search_destroy_widget */
		pbs->cur_query = g_object_new(TYPE_PKGQUERY, NULL);
		pbs->old_path = NULL;
		/* CONNECT BUTTON SIGNALS */
		g_signal_connect(G_OBJECT(pbs->b_close), "clicked",
				 G_CALLBACK(search_destroy_widget), pb);
		g_signal_connect(G_OBJECT(pbs->b_clear), "clicked",
				 G_CALLBACK(search_clear_entry),
				 pb->search_window);
		g_signal_connect(G_OBJECT(pbs->b_search), "clicked",
				 G_CALLBACK(search_exec_query), pb);
		gtk_widget_show_all(pbs->search_window);
	}
}
/**
 * Callback function which destroys the search window if it
 * is already opened.
 */
void
search_destroy_widget(GtkWidget *source, gpointer data)
{
	PBWindow       *pb;
	PBSearchWindow *pbs;

	pb = (PBWindow *) data;
	pbs = pb->search_window;
	if (pb->search_active) {
		pb->search_active = FALSE;
		g_object_unref(pbs->cur_query);
		FREE_PATH(pbs->old_path);
		gtk_widget_destroy(pbs->search_window);
	}
}

/**
 * Search callback function.
 */
static void
search_exec_query(GtkWidget *widget, gpointer data)
{
	GtkTreeView    *view;
	GtkTreeModel   *model;
	GtkTreePath    *newpath;
	PBWindow       *pb;
	PBSearchWindow *pbs;
	GList          *listptr;
	GdkCursor      *cursor;

	pb = (PBWindow *) data;
	pbs = pb->search_window;
	listptr = g_list_first(gtk_container_get_children(GTK_CONTAINER(
							  pb->scr_win_tree)));
	view = GTK_TREE_VIEW(listptr->data);
	model = GTK_TREE_MODEL(gtk_tree_view_get_model(view));
	/* SHOW THE USER THAT SOMETHING IS GOING ON  */
	cursor = gdk_cursor_new(GDK_WATCH);
	gdk_window_set_cursor((pbs->search_window)->window, cursor);
	gdk_flush();
	if (search_get_query(pbs) && !pkgquery_is_empty(pbs->cur_query)) {
		/* FIND FIRST */
		FREE_PATH(pbs->old_path);
		if (find_first(model, pbs->cur_query, &(pbs->old_path))) {
			gtk_tree_view_expand_to_path(view, pbs->old_path);
			gtk_tree_view_scroll_to_cell(view, pbs->old_path,
						     NULL, TRUE, 0.5, 0.5);
			gtk_tree_view_set_cursor(view, pbs->old_path, NULL,
						 FALSE);
		}
	} else {
		if (!pkgquery_is_empty(pbs->cur_query) && (pbs->old_path)) {
			/* FIND NEXT */
			if (find_next(model, pbs->cur_query, pbs->old_path,
				      &newpath)) {
				FREE_PATH(pbs->old_path);
				pbs->old_path = newpath;
				gtk_tree_view_expand_to_path(view, newpath);
				gtk_tree_view_scroll_to_cell(view, newpath,
							NULL, TRUE, 0.5, 0.5);
				gtk_tree_view_set_cursor(view, newpath,
							 NULL, FALSE);
			} else {
				/* NOTHING TO DO */
				FREE_PATH(pbs->old_path);
			}
		}
	}
	gdk_window_set_cursor((pbs->search_window)->window, NULL);
	gdk_cursor_unref(cursor);
	g_list_free(listptr);
}

/**
 * Extracts query from combobox and returns TRUE if
 * it has extracted a new query.
 */
static Boolean
search_get_query(PBSearchWindow * pbs)
{
	Boolean         retval;
	char           *oquery;
	const char     *sphrase;

	retval = FALSE;
	sphrase = NULL;
	sphrase = gtk_entry_get_text(pbs->entry);
	if (sphrase && strlen(sphrase)) {
		/* CASE SENSITIVE ? */
		g_object_set(pbs->cur_query, "case-sens",
			     gtk_toggle_button_get_active(
				     GTK_TOGGLE_BUTTON(pbs->tog_case)), NULL);
		switch (gtk_option_menu_get_history(pbs->omenu)) {
		case MENU_SELECT_SALL:
			oquery = pkgquery_get_qsall(pbs->cur_query);
			/* oquery is either NULL or different than before */
			if ((!oquery) || (strcmp(sphrase, oquery)))
				retval = TRUE;
			pkgquery_clear(pbs->cur_query);
			g_object_set(pbs->cur_query, "q-sall", sphrase, NULL);
			break;
		case MENU_SELECT_PKGNAME:
			oquery = pkgquery_get_qname(pbs->cur_query);
			if ((!oquery) || (strcmp(sphrase, oquery)))
				retval = TRUE;
			pkgquery_clear(pbs->cur_query);
			g_object_set(pbs->cur_query, "q-name", sphrase, NULL);
			break;
		case MENU_SELECT_COMMENT:
			oquery = pkgquery_get_qcomment(pbs->cur_query);
			if ((!oquery) || (strcmp(sphrase, oquery)))
				retval = TRUE;
			pkgquery_clear(pbs->cur_query);
			g_object_set(pbs->cur_query, "q-comment", sphrase, NULL);
			break;
		case MENU_SELECT_DESC:
			oquery = pkgquery_get_qedesc(pbs->cur_query);
			if ((!oquery) || (strcmp(sphrase, oquery)))
				retval = TRUE;
			pkgquery_clear(pbs->cur_query);
			g_object_set(pbs->cur_query, "q-edesc", sphrase, NULL);
			break;
		case MENU_SELECT_RDEP:
			oquery = pkgquery_get_qrdep(pbs->cur_query);
			if ((!oquery) || (strcmp(sphrase, oquery)))
				retval = TRUE;
			pkgquery_clear(pbs->cur_query);
			g_object_set(pbs->cur_query, "q-rdep", sphrase, NULL);
			break;
#ifdef __OpenBSD__
		case MENU_SELECT_LDEP:
			oquery = pkgquery_get_qldep(pbs->cur_query);
			if ((!oquery) || (strcmp(sphrase, oquery)))
				retval = TRUE;
			pkgquery_clear(pbs->cur_query);
			g_object_set(pbs->cur_query, "q-ldep", sphrase, NULL);
			break;
#endif				/* __OpenBSD__ */
		case MENU_SELECT_BDEP:
			oquery = pkgquery_get_qbdep(pbs->cur_query);
			if ((!oquery) || (strcmp(sphrase, oquery)))
				retval = TRUE;
			pkgquery_clear(pbs->cur_query);
			g_object_set(pbs->cur_query, "q-bdep", sphrase, NULL);
			break;
		default:
			pb_warning("Selected search criterion does not exist");
		}
		return retval;
	}
	return retval;
}

/**
 * Callback function clearing out the GtkEntry.
 */
static void
search_clear_entry(GtkWidget *source, gpointer data)
{
	PBSearchWindow *pbs;

	pbs = (PBSearchWindow *) data;
	gtk_entry_set_text(pbs->entry, "");
	/* clear PkgQuery */
	pkgquery_clear(pbs->cur_query);
	FREE_PATH(pbs->old_path);
}

/**
 * Callback function which always selects the next element
 * in the treeview.
 */
void
treeview_next_selection(GtkWidget *source, gpointer data)
{
	GtkTreeSelection *selection;
	GtkTreeIter     iter, iterchild, mother;
	GtkTreeModel   *model;
	GtkTreePath    *path;
	GtkTreeView    *view;
	PBWindow       *pb;
	GList          *listptr;

	pb = (PBWindow *) data;
	/* EXTRACT THE TREEVIEW FROM PARENT SCROLLEDWINDOW */
	listptr = g_list_first(gtk_container_get_children(GTK_CONTAINER(
							  pb->scr_win_tree)));
	if (listptr && GTK_IS_TREE_VIEW(listptr->data)) {
		view = GTK_TREE_VIEW(listptr->data);
		selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
		model = gtk_tree_view_get_model(view);
		/* DID WE SELECT ANYTHING ? */
		if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
			/* SOMETHING WAS SELECTED -> LOOK IF IT HAS A CHILD */
			if (gtk_tree_model_iter_has_child(model, &iter)) {
				/* EXPAND IT FIRST */
				path = gtk_tree_model_get_path(model, &iter);
				gtk_tree_view_expand_row(view, path, TRUE);
				gtk_tree_path_down(path);
				gtk_tree_view_set_cursor(view, path, NULL, FALSE);
				gtk_tree_view_scroll_to_cell(view, path, NULL,
							     TRUE, 0.5, 0.5);
				FREE_PATH(path);
			} else {
				/* WE ARE AT THE END */
				if (!gtk_tree_model_iter_next(model, &iter)) {
					/*
					 * WE EITHER NEED TO GO UP TO THE PARENT OR
					 * WE ARE AT THE END AND START FROM THE TOP
					 */
					if (gtk_tree_model_iter_parent(model, &mother, &iter)) {
						path = gtk_tree_model_get_path(model, &mother);
						gtk_tree_view_collapse_row(view, path);
						FREE_PATH(path);
						iter = mother;
						if (!gtk_tree_model_iter_next(model, &iter))
							/*
							 * WE ARE AT THE END AND NEED
							 * TO START FROM THE TOP
							 */
							gtk_tree_model_get_iter_first(model, &iter);
					}
				}
				path = gtk_tree_model_get_path(model, &iter);
				gtk_tree_view_scroll_to_cell(view, path, NULL,
							     TRUE, 0.5, 0.5);
				gtk_tree_view_set_cursor(view, path, NULL, FALSE);
				FREE_PATH(path);
			}
		} else {
			/* IF NOTHING WAS SELECTED THEN SELECT SOMETHING */
			path = gtk_tree_path_new_first();
			gtk_tree_view_set_cursor(view, path, NULL, FALSE);
			FREE_PATH(path);
		}
	}
	g_list_free(listptr);
}

/**
 * Callback function which always selects the previous element
 * in the treeview.
 */
void
treeview_previous_selection(GtkWidget *source, gpointer data)
{
	GtkTreeSelection *selection;
	GtkTreeIter     iter, test;
	GtkTreeModel   *model;
	GtkTreePath    *path, *pathcmp1, *pathcmp2;
	GtkTreeView    *view;
	PBWindow       *pb;
	gchar          *ptr;
	GList          *listptr;

	pb = (PBWindow *) data;
	/* EXTRACT THE TREEVIEW FROM PARENT SCROLLEDWINDOW */
	listptr = g_list_first(gtk_container_get_children(GTK_CONTAINER(
							  pb->scr_win_tree)));
	if (listptr && GTK_IS_TREE_VIEW(listptr->data)) {
		view = GTK_TREE_VIEW(listptr->data);
		selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
		model = gtk_tree_view_get_model(view);
		/* DID WE SELECT ANYTHING ? */
		if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
			gtk_tree_model_get_iter_first(model, &test);
			/* ARE WE AT THE BEGINNING */
			pathcmp1 = gtk_tree_model_get_path(model, &iter);
			pathcmp2 = gtk_tree_model_get_path(model, &test);
			if (gtk_tree_path_compare(pathcmp1, pathcmp2) == 0) {
				/* TRAVERSE TO THE LAST ELEMENT */
				while (gtk_tree_model_iter_next(model, &iter)) {
					/* DO NOTHING */
				}
			} else {
				/* FIND THE PREVIOUS ELEMENT */
				path = gtk_tree_model_get_path(model, &iter);
				if (gtk_tree_path_prev(path)) {
					/* DO NOTHING */
				} else {
					gtk_tree_path_up(path);
					gtk_tree_view_collapse_row(view, path);
				}
				ptr = gtk_tree_path_to_string(path);
				gtk_tree_model_get_iter_from_string(model, &iter, ptr);
				g_free(ptr);
				FREE_PATH(path);
			}
			path = gtk_tree_model_get_path(model, &iter);
			gtk_tree_view_scroll_to_cell(view, path, NULL, TRUE, 0.5, 0.5);
			gtk_tree_view_set_cursor(view, path, NULL, FALSE);
			FREE_PATH(pathcmp1);
			FREE_PATH(pathcmp2);
			FREE_PATH(path);
		} else {
			/* IF NOTHING WAS SELECTED THEN SELECT SOMETHING */
			path = gtk_tree_path_new_first();
			gtk_tree_view_set_cursor(view, path, NULL, FALSE);
			FREE_PATH(path);
		}
	}
	g_list_free(listptr);
}

/**
 * Callback function for selection changed event in our tree view
 */
void
treeview_changed_selection(GtkTreeSelection * selected, gpointer data)
{
	PkgObject      *pkg;
	PBWindow       *pb;

	pb = (PBWindow *) data;
	if (treeview_get_selection(pb, &pkg)) {
		pb_update_ui_entries(pkg, pb);
	}
}

/**
 * Callback function which refreshes the treeview by recreating it.
 */
void
treeview_refresh_event(GtkWidget *source, gpointer data)
{
	GtkTreeSelection *selection;
	PBWindow       *pb;
	GtkTreeIter     iter;
	GtkTreeModel   *model;
	GtkTreeView    *view;
	GtkTreeStore   *store;
	PkgObject      *np;
	GList          *listptr;
	GtkTreePath    *path;

	path = NULL;
	pb = (PBWindow *) data;
	listptr = g_list_first(gtk_container_get_children(GTK_CONTAINER(
							  pb->scr_win_tree)));
	if (listptr && GTK_IS_TREE_VIEW(listptr->data)) {
		view = GTK_TREE_VIEW(listptr->data);
		selection = gtk_tree_view_get_selection(view);
		store = GTK_TREE_STORE(gtk_tree_view_get_model(view));
		model = gtk_tree_view_get_model(view);
		if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
			/* SOMETHING WAS SELECTED - REMEMBER IT */
			path = gtk_tree_model_get_path(model, &iter);
		}
		/*
		 * THE INSTALL STATUS IS THE ONLY REAL THING THAT CAN CHANGE
		 * SO UPDATED IT
		 */
		for (np = pkgobject_get_list_head(); (np); np = pkgobject_get_list_next(np)) {
			pkgobject_set_install_status(np, pkgobject_check_install_status(np));
		}
		/* !!! CLEAR THIS ONE OUT TO AVOID MEMORY LEAKS  !!! */
		gtk_tree_store_clear(store);
		g_object_unref(store);
		gtk_container_remove(GTK_CONTAINER(pb->scr_win_tree),
				     GTK_WIDGET(view));
		create_treeview(pb);
		pb_set_statusbar_text(pb->statusbar, "Ports Statistics: %d"
				      "Ports available, %d Ports installed",
				      pkgobject_get_list_size(),
				   pkgobject_get_number_installed_packages());
		g_list_free(listptr);
		if (path) {
			listptr = g_list_first(gtk_container_get_children(GTK_CONTAINER(
							  pb->scr_win_tree)));
			if (listptr && GTK_IS_TREE_VIEW(listptr->data)) {
				view = GTK_TREE_VIEW(listptr->data);
				gtk_tree_view_expand_to_path(view, path);
				gtk_tree_view_set_cursor(view, path, NULL, FALSE);
				gtk_tree_view_scroll_to_cell(view, path, NULL, TRUE, 0.5, 0.5);
			}
			FREE_PATH(path);
			g_list_free(listptr);
		}
	}
}

/**
 * Set's all the relevant Textbuffer, Labels etc. with
 * information from ptr. Basically this get's called after
 * each selection-changed event.
 */
static Boolean
pb_update_ui_entries(PkgObject * ptr, PBWindow * pb)
{
	static char     descr[_POSIX_SSIZE_MAX];
	static char     plist[_POSIX_SSIZE_MAX];

	if (ptr) {
		/* DISTRIBNAME */
		gtk_entry_set_text(GTK_ENTRY(pb->e_name), pkgobject_get_distrib_name(ptr));
		/* DIRECTORY */
		gtk_entry_set_text(GTK_ENTRY(pb->e_dir), pkgobject_get_port_path(ptr));
		/* CATEGORY */
		gtk_entry_set_text(GTK_ENTRY(pb->e_cat), pkgobject_get_category(ptr));
		/* SHORT DESC */
		gtk_entry_set_text(GTK_ENTRY(pb->e_sdesc), pkgobject_get_comment(ptr));
#ifdef __OpenBSD__
		/* FLAVOUR */
		if (pkgobject_get_flavour(ptr)) {
			gtk_entry_set_text(GTK_ENTRY(pb->e_flav), pkgobject_get_flavour(ptr));
		} else {
			gtk_entry_set_text(GTK_ENTRY(pb->e_flav), "");
		}
		/* MULTIPACKAGE */
		if (pkgobject_get_multipackage(ptr)) {
			gtk_entry_set_text(GTK_ENTRY(pb->e_multi), pkgobject_get_multipackage(ptr));
		} else {
			gtk_entry_set_text(GTK_ENTRY(pb->e_multi), "");
		}
#else
		/* WWW */
		if (pkgobject_get_homepage(ptr)) {
			gtk_entry_set_text(GTK_ENTRY(pb->e_www), pkgobject_get_homepage(ptr));
		} else {
			gtk_entry_set_text(GTK_ENTRY(pb->e_www), "");
		}
#endif
		/* DESCR */
		if (pkgobject_get_descr_content(ptr, descr, sizeof(descr))) {
			gtk_text_buffer_set_text(pb->txtb_ldesc, descr, -1);
		} else {
			gtk_text_buffer_set_text(pb->txtb_ldesc, "", -1);
		}
		/* PLIST */
		if (pkgobject_get_plist_content(ptr, plist, sizeof(plist))) {
			gtk_text_buffer_set_text(pb->txtb_plist, plist, -1);
		} else {
			gtk_text_buffer_set_text(pb->txtb_plist, "", -1);
		}
#ifdef __OpenBSD__
		/* LIBDEPS */
		if (pkgobject_get_library_depends(ptr)) {
			gtk_text_buffer_set_text(pb->txtb_ldep, pkgobject_get_library_depends(ptr), -1);
		} else {
			gtk_text_buffer_set_text(pb->txtb_ldep, "", -1);
		}
#endif				/* __OpenBSD__ */
		/* RUNDEPS */
		if (pkgobject_get_run_depends(ptr)) {
			gtk_text_buffer_set_text(pb->txtb_rdep, pkgobject_get_run_depends(ptr), -1);
		} else {
			gtk_text_buffer_set_text(pb->txtb_rdep, "", -1);
		}
		/* BUILDDEPS */
		if (pkgobject_get_build_depends(ptr)) {
			gtk_text_buffer_set_text(pb->txtb_bdep, pkgobject_get_build_depends(ptr), -1);
		} else {
			gtk_text_buffer_set_text(pb->txtb_bdep, "", -1);
		}
		/* SET STATE OF INSTALL|DELETE BUTTONS */
		if (pkgobject_get_install_status(ptr)) {
			gtk_widget_set_sensitive(pb->b_iport, FALSE);
			gtk_widget_set_sensitive(pb->b_ipkg, FALSE);
			gtk_widget_set_sensitive(pb->b_dpkg, TRUE);
			gtk_widget_set_sensitive(pb->b_dpkgf, TRUE);
		} else {
			gtk_widget_set_sensitive(pb->b_iport, TRUE);
			gtk_widget_set_sensitive(pb->b_ipkg, TRUE);
			gtk_widget_set_sensitive(pb->b_dpkg, FALSE);
			gtk_widget_set_sensitive(pb->b_dpkgf, FALSE);
		}
		/* SET THE STATUS BAR TO THE CURRENT PACKAGE */
		pb_set_statusbar_text(pb->statusbar, "Portname: %s Category: %s",
				      pkgobject_get_distrib_name(ptr),
				      pkgobject_get_category(ptr));

		return TRUE;
	} else
		return FALSE;
}

/**
 * Finds the selected PkgBase Element in the list and returns a pointer to it.
 */
static Boolean
treeview_get_selection(PBWindow * pb, PkgObject ** pkg)
{
	GtkTreeSelection *selection;
	GList          *listptr;
	GtkTreeView    *view;
	GtkTreeModel   *model;
	GtkTreeIter     iter;
	PkgObject      *np;
	unsigned int    pkg_id;

	pkg_id = 0;
	/* EXTRACT THE TREEVIEW FROM PARENT SCROLLEDWINDOW */
	listptr = g_list_first(gtk_container_get_children(GTK_CONTAINER(pb->scr_win_tree)));
	view = GTK_TREE_VIEW(listptr->data);
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
	g_list_free(listptr);
	if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
		gtk_tree_model_get(model, &iter, INT_COL_ID, &pkg_id, -1);
		for (np = pkgobject_get_list_head(); (np); np = pkgobject_get_list_next(np)) {
			if (pkg_id == pkgobject_get_id(np)) {
				*pkg = np;
				return TRUE;
			}
		}

	}
	return FALSE;
}

static void
pb_thread_data_init(struct _ThreadData ** data)
{
	(*data) = (struct _ThreadData *) calloc(1, sizeof(struct _ThreadData));
	(*data)->param = (InstallParams *) calloc(1, sizeof(InstallParams));
	(*data)->pb = NULL;
	(*data)->pbarwin = NULL;
	(*data)->view = NULL;
	(*data)->pbarwin = NULL;
	(*data)->bclose = NULL;
	(*data)->bcancel = NULL;
	((*data)->param)->err = NULL;
	(*data)->canceled = FALSE;
}

static void
pb_thread_data_destroy(struct _ThreadData ** data)
{
	if (((*data)->param)->err)
		g_error_free(((*data)->param)->err);
	FREE((*data)->param);
	FREE((*data));
}

/**
 * Callback function which calls the appropriate install function for packages.
 */
void
port_install_function(GtkWidget *widget, gpointer data)
{
	struct _ThreadData *tparam;
	GtkWidget      *pwd_dialog;
	GtkProgressBar *prog;
	char            title[256], labeltxt[256];
	unsigned int    sid_1, sid_2;
	int             res;

	pwd_dialog = NULL;
	pb_thread_data_init(&tparam);
	tparam->pb = (PBWindow *) data;
	if (treeview_get_selection(tparam->pb, &(tparam->param)->pkg)) {
		if (!pb_root_pwd_is_valid()) {
		  	/* 
			 * THE PASSWORD WE ENTERED WAS EITHER INVALID OR WE HAVE 
			 * NOT ENTERED A PASSWORD YET AT ALL
			 */
			pwd_dialog = pb_create_pwd_dialog((tparam->pb)->main_window);
			res = gtk_dialog_run(GTK_DIALOG(pwd_dialog));
		} else {
		  	/* WE HAVE ALREADY ENTERED A VALID PASSWORD */
			res = GTK_RESPONSE_OK;
		}
		if (res == GTK_RESPONSE_OK) {
			if (pwd_dialog)
			  gtk_widget_destroy(pwd_dialog);
			/* RETURN CONTROL TO GTKMAIN */
			while (g_main_context_iteration(NULL, FALSE));
			/* Extract the install type from the button widget */
			(tparam->param)->inst_type = GPOINTER_TO_INT((gpointer) g_object_get_data(G_OBJECT(widget),
								"inst_type"));
			/* PUT TOGETHER TEXT LABELS FOR THE PROGRESS WINDOW */
			if ((tparam->param)->inst_type == PB_PKGCMD_INST_PORT) {
				snprintf(title, sizeof(title), "Installing Port %s ...",
					 pkgobject_get_distrib_name((tparam->param)->pkg));
				STRLCPY(labeltxt, "Port installation output:", sizeof(labeltxt));
			} else if ((tparam->param)->inst_type == PB_PKGCMD_INST_PKG) {
				snprintf(title, sizeof(title), "Installing Package %s ...",
					 pkgobject_get_distrib_name((tparam->param)->pkg));
				STRLCPY(labeltxt, "Package installation output:", sizeof(labeltxt));
			} else if ((tparam->param)->inst_type == PB_PKGCMD_DEL_PKG) {
				snprintf(title, sizeof(title), "Deleting Package %s ...",
					 pkgobject_get_distrib_name((tparam->param)->pkg));
				STRLCPY(labeltxt, "Uninstallation output:", sizeof(labeltxt));
			} else if ((tparam->param)->inst_type == PB_PKGCMD_DELF_PKG) {
				snprintf(title, sizeof(title), "Forcing Removal of Package %s ...",
					 pkgobject_get_distrib_name((tparam->param)->pkg));
				STRLCPY(labeltxt, "Uninstallation output:", sizeof(labeltxt));
			} else {
				STRLCPY(title, "Unknown Operation...", sizeof(title));
				STRLCPY(labeltxt, "Unknown Operation:", sizeof(labeltxt));
			}
			/* CREATE|RUN THE PROGRESS BAR */
			tparam->pbarwin = pb_create_progressbar((tparam->pb)->main_window,
								&prog,
								&tparam->view,
							      &tparam->bclose,
							     &tparam->bcancel,
								title,
								labeltxt,
								NULL);
			/* CONNECT BUTTON SIGNALS */
			g_signal_connect_swapped(tparam->bclose, "clicked",
			     G_CALLBACK(gtk_widget_destroy), tparam->pbarwin);
			g_signal_connect_swapped(tparam->bcancel, "clicked",
					  G_CALLBACK(kill_async_cmd), tparam);
			/* THE BUTTON DONE CAN NOT BE CLICKED YET */
			gtk_widget_set_sensitive(GTK_WIDGET(tparam->bclose), FALSE);

			gtk_widget_show(tparam->pbarwin);
			/* UPDATE THE PROGRESS BAR */
			sid_1 = g_timeout_add(100, pb_update_progress, prog);
			/* UPDATE THE TEXTVIEW */
			sid_2 = g_timeout_add(100, pb_update_textview, tparam->view);
			g_object_set_data(G_OBJECT(tparam->pbarwin), "sid_1", GUINT_TO_POINTER(sid_1));
			g_object_set_data(G_OBJECT(tparam->pbarwin), "sid_2", GUINT_TO_POINTER(sid_2));
			/* START OUR INSTALL|UNINSTALL THREAD */
			g_thread_create(exec_async_cmd, tparam, FALSE, NULL);
		} else {
			gtk_widget_destroy(pwd_dialog);
		}

	}
}

/**
 * Execute the installation process.
 */
static gpointer
exec_async_cmd(gpointer data)
{
	struct _ThreadData *tparam;

	tparam = (struct _ThreadData *) data;
	pkg_perform_operation(tparam->param);
	/* CALL THIS FUNCTION WHENEVER OUR THREAD IS DONE */
	g_idle_add(exec_async_cmd_done, data);
	return NULL;
}


/**
 * Clean up after the installation process and notify the
 * user of possible errors.
 */
static gboolean
exec_async_cmd_done(gpointer data)
{
	struct _ThreadData *tparam;
	GtkWidget      *dialog;

	tparam = (struct _ThreadData *) data;
	if (!tparam->canceled) {
		/* MAKE SURE WE CATCHED ALL OF THE OUTPUT */
		pb_update_textview(tparam->view);
		/* STOP PULSING THE PROGRESS BAR AND UPDATING THE TEXT VIEW */
		g_source_remove(GPOINTER_TO_UINT((gpointer) g_object_get_data(G_OBJECT(tparam->pbarwin),
								   "sid_1")));
		g_source_remove(GPOINTER_TO_UINT((gpointer) g_object_get_data(G_OBJECT(tparam->pbarwin),
								   "sid_2")));
		/* WERE THERE ANY ERRORS */
		if ((tparam->param)->err) {
			dialog = pb_create_warning_dialog((tparam->pb)->main_window,
					     ((tparam->param)->err)->message);
			gtk_dialog_run(GTK_DIALOG(dialog));
			gtk_widget_destroy(dialog);
		} else {
			/* NO ERRORS -> REFRESH TREEVIEW */
			treeview_refresh_event(NULL, tparam->pb);
		}
		/* SET THE CLOSE|CANCEL BUTTON */
		gtk_widget_set_sensitive(GTK_WIDGET(tparam->bclose), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(tparam->bcancel), FALSE);
	}
	pb_thread_data_destroy(&tparam);
	return FALSE;
}

/**
 * Function which will kill the install|uninstall process the
 * hard way!
 */
static void
kill_async_cmd(gpointer data)
{
	struct _ThreadData *tparam;

	tparam = (struct _ThreadData *) data;
	/* YES WE CANCEL THE PROCESS */
	tparam->canceled = TRUE;
	pkg_stop_operation(SIGTERM);
	/* STOP PULSING THE PROGRESS BAR AND UPDATING THE TEXT VIEW */
	g_source_remove(GPOINTER_TO_UINT((gpointer) g_object_get_data(G_OBJECT(tparam->pbarwin),
								   "sid_1")));
	g_source_remove(GPOINTER_TO_UINT((gpointer) g_object_get_data(G_OBJECT(tparam->pbarwin),
								   "sid_2")));
	/* SET THE CLOSE|CANCEL BUTTON */
	gtk_widget_set_sensitive(GTK_WIDGET(tparam->bclose), TRUE);
	gtk_widget_set_sensitive(GTK_WIDGET(tparam->bcancel), FALSE);
}
