/*
 * $Id: passwdqc.c,v 1.3 2005/01/18 00:49:42 djm Exp $
 *
 * Copyright (c) 2005 Damien Miller <djm@mindrot.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef lint
static const char rcsid[] = "$Id: passwdqc.c,v 1.3 2005/01/18 00:49:42 djm Exp $";
#endif /* not lint */

#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include <errno.h>
#include <err.h>
#include <pwd.h>
#include <getopt.h>

#include "passwdqc.h"

static void __dead
usage(void)
{
	fprintf(stderr, "passwdqc version %s\n", PASSWDQC_VER);
	fprintf(stderr, "Usage: passwdqc: [-1|2|3|4|p length] [-w words] "
	    "[-m max] [-s sub_len]\n");
	exit(1);
}

int main(int argc, char **argv)
{
	char password[1024];
	const char *reason;
	struct passwd *pw;
	size_t len;
	int disabled, ch, *intp;
	passwdqc_params_t defparams = {
		{INT_MAX, 24, 12, 8, 7},	/* min */
		256,				/* max */
		3,				/* passphrase_words */
		4,				/* match_length */
		1				/* similar_deny */
	};

	while ((ch = getopt(argc, argv, "h1:2:3:4:p:w:s:")) != -1) {
		switch (ch) {
		case '1':
			intp = &defparams.min[0];
			disabled = INT_MAX;
 parse_len:
			if (strcmp(optarg, "disabled") == 0 || 
			    strcmp(optarg, "disable") == 0)
				*intp = disabled;
			else { 
				*intp = (int)strtonum(optarg, 0, 1024, &reason);
				if (reason != NULL)
					errx(1, "Length is %s: %s", reason,
					    optarg);
			}
			break;
		case '2':
			disabled = INT_MAX;
			intp = &defparams.min[1];
			goto parse_len;
		case 'p':
			disabled = INT_MAX;
			intp = &defparams.min[2];
			goto parse_len;
		case '3':
			disabled = INT_MAX;
			intp = &defparams.min[3];
			goto parse_len;
		case '4':
			disabled = INT_MAX;
			intp = &defparams.min[4];
			goto parse_len;
		case 'w':
			disabled = 0;
			intp = &defparams.passphrase_words;
			goto parse_len;
		case 's':
			disabled = 0;
			intp = &defparams.match_length;
			goto parse_len;
		case 'h':
		default:
			usage();
		}
	}

	if (optind < argc)
		usage();

	if ((pw = getpwuid(getuid())) == NULL)
		err(1, "getpwuid");
	if (fgets(password, sizeof(password), stdin) == NULL)
		err(1, "fgets");

	/* Expect eof, otherwise password length is greater than buffer size */
	if (!feof(stdin)) {
		bzero(password, strlen(password));
		errx(1, "EOF expected");
	}

	len = strlen(password);
	if (len > 0 && password[len - 1] == '\n')
		password[--len] = '\0';
	if (len == 0)
		errx(1, "Password expected");

	if ((reason = passwdqc_check(&defparams, password, NULL, pw)) != NULL) {
		bzero(password, strlen(password));
		printf("Password invalid: %s\n", reason);
		exit(1);
	}
	bzero(password, strlen(password));

	return (0);
}
