/*
    ParaGUI - crossplatform widgetset
    Copyright (C) 2000,2001  Alexander Pipelka

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Alexander Pipelka
    pipelka@teleweb.at

    Last Update:      $Author: pipelka $
    Update Date:      $Date: 2001/05/22 09:34:41 $
    Source File:      $Source: /usr/local/CVSROOT/linux/paragui/src/Attic/pgsurfacecache.cpp,v $
    CVS/RCS Revision: $Revision: 1.1.2.10 $
    Status:           $State: Exp $
*/

#include "pgsurfacecache.h"
#include <iostream.h>
#include <map>
#include <string.h>
#include <string>

#define MY_SURFACEMAP ((pg_surfacemap_t*)my_surfacemap)

struct pg_surface_cmp {
	bool operator()(std::string o1, std::string o2) const {
		return (o1 < o2);
	}
};

typedef std::map<std::string, pg_surface_cache_t*, pg_surface_cmp> pg_surfacemap_t;
typedef pg_surfacemap_t::iterator pg_surfacemap_iter_t;

PG_SurfaceCache::PG_SurfaceCache() {
	my_surfacemap = (void*)new(pg_surfacemap_t);
}

PG_SurfaceCache::~PG_SurfaceCache() {
	Cleanup();
}

void PG_SurfaceCache::Cleanup() {

	if(my_surfacemap == NULL) {
		return;
	}

	pg_surfacemap_iter_t i = MY_SURFACEMAP->begin();
	while(i != MY_SURFACEMAP->end()) {
		pg_surface_cache_t* t = (*i).second;
		SDL_FreeSurface(t->surface);
		delete t;
		MY_SURFACEMAP->erase(i);
		i = MY_SURFACEMAP->begin();
	}
	
	MY_SURFACEMAP->clear();
	delete MY_SURFACEMAP;
	my_surfacemap = NULL;
}

void PG_SurfaceCache::CreateKey(char* key, int w, int h, PG_Gradient* gradient, SDL_Surface* background, int bkmode, Uint8 blend) {
	char colorkey[4][10];
	int i=0;
	
	for(i=0; i<4; i++) {
		sprintf(colorkey[i], "%03i%03i%03i",
			gradient->colors[i].r,
			gradient->colors[i].g,
			gradient->colors[i].b
			);
	}
	
	sprintf(key, "%i%i%s%s%s%s%012i%02i%02i",
		w, h,
		colorkey[0],
		colorkey[1],
		colorkey[2],
		colorkey[3],
		(unsigned int)background,
		bkmode,
		blend);
		
#ifdef DEBUG
	cerr << "KEY:  " << key << endl;
#endif
}

bool PG_SurfaceCache::FindSurfacePointer(SDL_Surface* surface) {
	pg_surfacemap_iter_t i = MY_SURFACEMAP->begin();
	SDL_Surface* result = NULL;
	
	while(i != MY_SURFACEMAP->end()) {
		result = (*i).second->surface;
		if(result == surface) {
			break;
		}
		i++;
	}
	
	return (result == surface);
}

pg_surface_cache_t* PG_SurfaceCache::FindByKey(char* key) {
	pg_surfacemap_iter_t i = MY_SURFACEMAP->find(key);

	if(i == MY_SURFACEMAP->end()) {
		return NULL;
	}

	return (*i).second;
}

pg_surface_cache_t* PG_SurfaceCache::FindBySurface(SDL_Surface* surface) {
	pg_surfacemap_iter_t i = MY_SURFACEMAP->begin();
	pg_surface_cache_t* result = NULL;
	
	while(i != MY_SURFACEMAP->end()) {
		result = (*i).second;
		if(result->surface == surface) {
			break;
		}
		i++;
	}

	return result;
}

SDL_Surface* PG_SurfaceCache::FindSurface(char* key) {
	pg_surfacemap_iter_t i = MY_SURFACEMAP->find(key);

#ifdef DEBUG
	cerr << "FIND: " << key << " ";
#endif
	
	if(i == MY_SURFACEMAP->end()) {
#ifdef DEBUG
		cerr << "NOT FOUND" << endl;
#endif		
		return NULL;
	}

#ifdef DEBUG
	cerr << "FOUND!" << endl;
#endif	
	
	return (*i).second->surface;
}

SDL_Surface* PG_SurfaceCache::AddSurface(char* key, SDL_Surface* surface) {
	std::string newkey = key;
	pg_surface_cache_t* t = NULL;

#ifdef DEBUG
	cerr << "ADD:  " << key << " ";
#endif

	t = FindByKey(key);
	if(t != NULL) {
#ifdef DEBUG
		cerr << "ERR DUP!" << endl;
#endif		
		SDL_FreeSurface(surface);
		t->refcount++;
		return t->surface;
	}

	pg_surface_cache_t* item = new pg_surface_cache_t;
	item->refcount = 1;
	item->surface = surface;
	
	(*MY_SURFACEMAP)[newkey] = item;
#ifdef DEBUG
	cerr << "OK" << endl;
#endif
	
	return surface;
}

void PG_SurfaceCache::DeleteSurface(SDL_Surface* surface) {

	// free unmanaged surface
	if(!FindSurfacePointer(surface)) {
		SDL_FreeSurface(surface);
		return;
	}

	// dec reference
	pg_surface_cache_t* t = FindBySurface(surface);
	t->refcount--;
	
	// no more references ?
	if(t->refcount > 0) {
		return;
	}

	// remove surface from cache
	pg_surfacemap_iter_t i = MY_SURFACEMAP->begin();
	pg_surface_cache_t* result = NULL;
	
	while(i != MY_SURFACEMAP->end()) {
		result = (*i).second;
		if(result->surface == surface) {
			break;
		}
		i++;
	}

	MY_SURFACEMAP->erase(i);
	SDL_FreeSurface(t->surface);
	delete t;
}

void PG_SurfaceCache::IncRef(char* key) {
	pg_surface_cache_t* t = FindByKey(key);

	if(t == NULL) {
		return;
	}

	t->refcount++;
}
