/*
    ParaGUI - crossplatform widgetset
    Copyright (C) 2000,2001  Alexander Pipelka
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.
 
    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.
 
    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
    Alexander Pipelka
    pipelka@teleweb.at

    Last Update:      $Author: pipelka $
    Update Date:      $Date: 2001/05/30 23:18:06 $
    Source File:      $Source: /usr/local/CVSROOT/linux/paragui/src/Attic/pgdrawobject.cpp,v $
    CVS/RCS Revision: $Revision: 1.1.2.50 $
    Status:           $State: Exp $
*/

#include <iostream>
#include "pgdrawobject.h"
#include "pgapplication.h"
#include "pgdrawline.h"
#include "rendertextrect.h"
#include "parastretch.h"

using namespace std;

PG_SurfaceCache PG_DrawObject::my_SurfaceCache;

PG_DrawObject::PG_DrawObject(const PG_Rect& rect, bool bCreateSurface) :
PG_Rect(rect),
my_srfObject(NULL),
my_rectClip(rect),
my_havesurface(bCreateSurface) {

	my_visible = false;
	my_srfScreen = PG_Application::GetScreen();

	if(my_havesurface) {
		my_srfObject = SDL_CreateRGBSurface (
					SDL_SWSURFACE,
					my_width, my_height,
					my_srfScreen->format->BitsPerPixel,
					my_srfScreen->format->Rmask,
					my_srfScreen->format->Gmask,
					my_srfScreen->format->Bmask,
					0
					);
	}

	// default border colors

	my_colorBorder[0][0].r = 255;
	my_colorBorder[0][0].g = 255;
	my_colorBorder[0][0].b = 255;

	my_colorBorder[0][1].r = 239;
	my_colorBorder[0][1].g = 239;
	my_colorBorder[0][1].b = 239;

	my_colorBorder[1][0].r = 89;
	my_colorBorder[1][0].g = 89;
	my_colorBorder[1][0].b = 89;

	my_colorBorder[1][1].r = 134;
	my_colorBorder[1][1].g = 134;
	my_colorBorder[1][1].b = 134;

	my_firstredraw = true;
	my_rectClip = *this;
	my_transparency = 0;
}

PG_DrawObject::~PG_DrawObject() {
	if(!my_havesurface && my_srfObject) {
		cerr << "?? DrawObject declared without a surface has unexpectedly born one ??" << endl;
	}
	RemoveObjectSurface();
}

void PG_DrawObject::RemoveObjectSurface() {
	if(my_srfObject) {
		SDL_FreeSurface(my_srfObject);
	}
	my_srfObject = NULL;
}

/**  */
bool PG_DrawObject::SizeWindow(int w, int h) {
	RestoreBackground();

	// create new widget drawsurface
	if(my_srfObject) {
		Uint32 depth = my_srfScreen->format->BitsPerPixel;
		Uint32 Rmask = my_srfScreen->format->Rmask;
		Uint32 Gmask = my_srfScreen->format->Gmask;
		Uint32 Bmask = my_srfScreen->format->Bmask;
		Uint32 Amask = 0; //my_srfScreen->format->Amask;

		SDL_FreeSurface(my_srfObject);

		my_srfObject = SDL_CreateRGBSurface (
						SDL_SWSURFACE,
						w, h,
						depth,
			 			Rmask,
			 			Gmask,
			 			Bmask,
			 			Amask
						);
	}

	eventSizeWindow(w, h);

	my_width = w;
	my_height = h;

	return true;
}

bool PG_DrawObject::RestoreBackground(PG_Rect* clip) {
	return true;
}

/**  */
bool PG_DrawObject::Redraw(bool update) {
	PG_Rect r(0, 0, my_width, my_height);

	if(my_srfObject != NULL) {
		eventDraw(my_srfObject, r);
	}
	else {
		r.x = my_xpos;
		r.y = my_ypos;
		eventDraw(my_srfScreen, r);
	}

	if(!my_visible)
		return false;

	if(update) {
		Update();
	}

	return true;
}

/**  */
void PG_DrawObject::eventDraw(SDL_Surface* surface, const PG_Rect& rect) {
}

void PG_DrawObject::eventBlit(SDL_Surface* srf, const PG_Rect& src, const PG_Rect& dst) {

	// Don't blit an object without a surface

	if(srf == NULL) {
		return;
	}

 	// Set alpha
 	if(my_transparency != 255) {
		/*if(PG_Application::GetGLMode()) {
			SDL_SetAlpha(srf, 0,0);
		}
		else {*/
			SDL_SetAlpha(srf, SDL_SRCALPHA, 255-my_transparency);
		//}

 		// Blit widget surface to screen
 		if(!(dst.w == 0 || dst.h == 0 || src.w == 0 || src.h == 0)) {
#ifdef DEBUG
 			cout << "SRC BLIT: x:" << src.x << " y:" << src.y << " w:" << src.w << " h:" << src.h << endl;
 			cout << "DST BLIT: x:" << dst.x << " y:" << dst.y << " w:" << dst.w << " h:" << dst.h << endl;
#endif // DEBUG
	 			PG_BlitSurface(srf, src, my_srfScreen, dst);
 		}
 	}
}

void PG_DrawObject::eventSizeWindow(int w, int h) {}

void PG_DrawObject::eventMoveWindow(int x, int y) {}

void PG_DrawObject::Blit(bool restore) {
	static PG_Rect src;
	static PG_Rect dst;

	if(!my_visible) {
		return;
	}

	if(restore) {
	 	RestoreBackground(&my_rectClip);
	 }
	
	RecalcClipRect();
	GetClipRects(src, dst);
	eventBlit(my_srfObject, src, dst);
}

/**  */
void PG_DrawObject::Update(bool doBlit) {
	static PG_Rect src;
	static PG_Rect dst;

	if(!my_visible) {
		return;
	}

	GetClipRects(src, dst);

	if(doBlit) {
		Blit();
	}

	// Update screen surface
	if(!(dst.my_width == 0 || dst.my_height == 0)) {
#ifdef DEBUG
		cout << "UPD: x:" << dst.x << " y:" << dst.y << " w:" << dst.w << " h:" << dst.h << endl;
#endif // DEBUG
		PG_UpdateRects(my_srfScreen, 1, &dst);
	}
}


void PG_DrawObject::RecalcClipRect() {
	PG_Rect pr(
			0,
			0,
			PG_Application::GetScreenWidth(),
			PG_Application::GetScreenHeight());

	PG_Rect ir = IntersectRect(pr);
	SetClipRect(ir);
}


/**  */
bool PG_DrawObject::MoveWindow(int x, int y) {

	RecalcClipRect();
	
	// Restore old background
	RestoreBackground();

	// Set new widget position

	int dx = x - my_xpos;
	int dy = y - my_ypos;
	my_xpos = x;
	my_ypos = y;
	my_rectClip.my_xpos += dx;
	my_rectClip.my_ypos += dy;

	Blit(false);

	return true;
}

/**  */
bool PG_DrawObject::SetVisible(bool visible) {

	// Attempt to make object visible
	if(visible) {
		if(my_visible) {			// Object already visible
			return false;
		} else {					// Display object
			my_visible = visible;
			if(my_firstredraw) {
				Redraw(false);
				my_firstredraw = false;
			}
		}

	}

	// Attempt to make object invisible
	if(!visible) {
		if(!my_visible) {			// Object is already invisible
			return false;
		} else {					// Hide object
			RestoreBackground();
			my_visible = visible;
		}
	}

	return true;
}

bool PG_DrawObject::IsDisplayRectValid() {
	if(my_xpos < 0 || my_ypos < 0) {
		return false;
	}

	if(my_xpos + my_width > PG_Application::GetScreenWidth()) {
		return false;
	}

	if(my_ypos + my_height > PG_Application::GetScreenHeight()) {
		return false;
	}

	return true;
}

/**  */
void PG_DrawObject::SetTransparency(Uint8 t) {
	my_transparency = t;
}

SDL_Surface* PG_DrawObject::CreateGradient(const PG_Rect& r, PG_Gradient& gradient) {
	return CreateGradient(
			r,
			gradient.colors[0],
			gradient.colors[1],
			gradient.colors[2],
			gradient.colors[3]);
}

/**  */
SDL_Surface* PG_DrawObject::CreateGradient(const PG_Rect& r, SDL_Color & ul, SDL_Color & ur, SDL_Color & dl, SDL_Color & dr) {
	SDL_Surface* screen = PG_Application::GetScreen();

	SDL_Surface *grd = SDL_CreateRGBSurface(
							SDL_SWSURFACE,
							r.my_width,
							r.my_height,
							screen->format->BitsPerPixel,
							screen->format->Rmask,
							screen->format->Gmask,
							screen->format->Bmask,
							0 //screen->format->Amask
							);

	r.my_xpos = 0;
	r.my_ypos = 0;
	DrawGradient(grd, r, ul, ur, dl, dr);

	return grd;
}

void PG_DrawObject::DrawGradient(SDL_Surface* surface, const PG_Rect& r, PG_Gradient& gradient) {
	DrawGradient(
		surface,
		r,
		gradient.colors[0],
		gradient.colors[1],
		gradient.colors[2],
		gradient.colors[3]);
}

/**  */
void PG_DrawObject::DrawGradient(SDL_Surface * surface, const PG_Rect& r, SDL_Color & ul, SDL_Color & ur, SDL_Color & dl, SDL_Color & dr) {
	double v[2][3];
	double w = r.my_width;
	double h = r.my_height;
	SDL_Color c;
	c.unused = 0;
	
	Uint32 c1 = SDL_MapRGB(surface->format, ul.r, ul.g, ul.b);
	Uint32 c2 = SDL_MapRGB(surface->format, ur.r, ur.g, ur.b);
	Uint32 c3 = SDL_MapRGB(surface->format, dl.r, dl.g, dl.b);
	Uint32 c4 = SDL_MapRGB(surface->format, dr.r, dr.g, dr.b);
	
	// solid color gradient ?
	if((c1 == c2) && (c2 == c3) && (c3 == c4)) {
		SDL_FillRect(surface, (PG_Rect*)&r, c1);
		return;
	}
	
	// color space vectors
	v[0][0] = (ur.r - ul.r) / w;
	v[0][1] = (ur.g - ul.g) / w;
	v[0][2] = (ur.b - ul.b) / w;

	v[1][0] = (dr.r - dl.r) / w;
	v[1][1] = (dr.g - dl.g) / w;
	v[1][2] = (dr.b - dl.b) / w;

	int r1, g1, b1;
	int r2, g2, b2;
	double yr, yg, yb;

	for (int x = 0; x < r.my_width; x++) {
		r1 = (int) (ul.r + v[0][0] * x);
		g1 = (int) (ul.g + v[0][1] * x);
		b1 = (int) (ul.b + v[0][2] * x);

		r2 = (int) (dl.r + v[1][0] * x);
		g2 = (int) (dl.g + v[1][1] * x);
		b2 = (int) (dl.b + v[1][2] * x);

		yr = ((double) (r2 - r1)) / h;
		yg = ((double) (g2 - g1)) / h;
		yb = ((double) (b2 - b1)) / h;

		for (int y = 0; y < r.my_height; y++) {
			c.r = (Uint8) (r1 + yr * y);
			c.g = (Uint8) (g1 + yg * y);
			c.b = (Uint8) (b1 + yb * y);

			SetPixel(r.my_xpos + x, r.my_ypos + y, c.r,c.g,c.b, surface);
		}
	}

}

SDL_Surface* PG_DrawObject::CreateThemedSurface(const PG_Rect& r, PG_Gradient* gradient, SDL_Surface* background, int bkmode, Uint8 blend) {
	SDL_Surface* cache_surface = NULL;	
	SDL_Surface* screen = PG_Application::GetScreen();
	char key[256];

	// create a key for the surface
	my_SurfaceCache.CreateKey(key, r.my_width, r.my_height,  gradient, background, bkmode, blend);
	
	// lookup the surface in cache
	cache_surface = my_SurfaceCache.FindSurface(key);
	
	// draw the cached surface :)
	if(cache_surface != NULL) {
		// increase the reference count
		my_SurfaceCache.IncRef(key);
		return cache_surface;
	}

	Uint8 bpp = screen->format->BitsPerPixel;
	Uint32 Rmask = screen->format->Rmask;
	Uint32 Gmask = screen->format->Gmask;
	Uint32 Bmask = screen->format->Bmask;
	Uint32 Amask = 0;

	if(background != NULL) {
		if(background->format->BitsPerPixel > 8) {
			bpp = background->format->BitsPerPixel;
			Rmask = background->format->Rmask;
			Gmask = background->format->Gmask;
			Bmask = background->format->Bmask;
			Amask = background->format->Amask;
		}
	}

	SDL_Surface *surface = SDL_CreateRGBSurface(
								SDL_SWSURFACE,
								r.my_width,
								r.my_height,
								bpp,
								Rmask,
								Gmask,
								Bmask,
								Amask
								);
	
	if(surface) {
		DrawThemedSurface(surface, PG_Rect(0, 0, r.my_width, r.my_height), gradient, background, bkmode, blend);
	}
	
	my_SurfaceCache.AddSurface(key, surface);
	return surface;
}

void PG_DrawObject::DeleteThemedSurface(SDL_Surface* surface) {
	if(surface == NULL) {
		return;
	}
	my_SurfaceCache.DeleteSurface(surface);
}

void PG_DrawObject::DrawThemedSurface(SDL_Surface* surface, const PG_Rect& r, PG_Gradient* gradient, SDL_Surface* background, int bkmode, Uint8 blend) {
	static PG_Rect srcrect;
	static PG_Rect dstrect;
	
	// check if we have anything to do
	if(!gradient && !background) {
		return;
	}

	// draw the gradient first
	if((background == NULL) || (background && (blend > 0))) {
		DrawGradient(surface, r, *gradient);
	}

	// tile the background image over the r in surface
	if((bkmode == BKMODE_TILE) && background) {
		dstrect = r;

		srcrect.SetRect(0, 0, background->w, background->h);

		int yc = (r.my_height / background->h) +1;
		int xc = (r.my_width / background->w) +1;

		for(int y=0; y<yc; y++) {
			for(int x=0; x<xc; x++) {

				dstrect.SetRect(
					r.my_xpos + background->w * x,
					r.my_ypos + background->h * y,
					background->w,
					background->h);

				srcrect.my_width = background->w;
				srcrect.my_height = background->h;

				if((dstrect.my_xpos + dstrect.my_width -1) > r.my_width) {
					dstrect.my_width = r.my_width - dstrect.my_xpos;
				}

				if((dstrect.my_ypos + dstrect.my_height -1) > r.my_height) {
					dstrect.my_height = r.my_height - dstrect.my_ypos;
				}

				srcrect.my_width = dstrect.my_width;
				srcrect.my_height = dstrect.my_height;

				if(blend > 0) {
					SDL_SetAlpha(background, SDL_SRCALPHA, 255-blend);
				}
				else {
					SDL_SetAlpha(background, 0, 0);
				}
				PG_BlitSurface(background, srcrect, surface, dstrect);
			}
		}
	}
	
	// stretch the background to fit the surface
	if((bkmode == BKMODE_STRETCH) && background) {
	
		// create a temp surface (same size as widget surface, but my_background format)
		SDL_Surface* temp = SDL_CreateRGBSurface(
					SDL_SWSURFACE,
					r.my_width, r.my_height,
					background->format->BitsPerPixel,
					background->format->Rmask,
					background->format->Gmask,
					background->format->Bmask,
					background->format->Amask);
					
		// stretch the my_background surface to temp (no antialiasing 'til now :( )
		RectStretch(
			background,
			0, 0,
			background->w, background->h,
			temp,
			0, 0,
			temp->w,
			temp->h,
			NULL);
			
		// set per surface alpha
		if(blend > 0) {
			SDL_SetAlpha(temp, SDL_SRCALPHA, 255-blend);
		}
		else {
			SDL_SetAlpha(temp, 0, 0);
		}
		
		// blit it
		SDL_BlitSurface(temp, NULL, surface, (PG_Rect*)&r);
		
		// free the temp surface
		SDL_FreeSurface(temp);
	}
}

void PG_DrawObject::GetTextSize(const char* text, int& width, int &height, TTF_Font* font) {

	if((text == NULL) || (text[0] == 0)) {
		width = 0;
		height = 0;
		return;
	}

	if(font == NULL) {
		font = PG_Application::GetDefaultFont();
	}

	TTF_SizeText(font, text, &width, &height);
}


void PG_DrawObject::DrawText(int x, int y, const char* text, SDL_Color c, TTF_Font* textfont) {
	SDL_Surface* surface = my_srfObject;

	if(textfont == NULL) {	// use default font
		textfont = PG_Application::GetDefaultFont();
	}

	// render text into surface
	SDL_Surface* label_srf = RenderText(textfont, text, c);

	if(label_srf == NULL) {
		return;
	}

	if(surface == NULL) {			// blit to screen
		PG_Rect src, dst;
		GetClipRects(src, dst, PG_Rect(my_xpos+x, my_ypos+y, label_srf->w, label_srf->h));
		PG_DrawObject::eventBlit(label_srf, src, dst);
	}
	else {										// blit to surface
		PG_Rect src(0, 0, label_srf->w, label_srf->h);
		PG_Rect dst(x, y, label_srf->w,label_srf->h);
		PG_BlitSurface(label_srf, src, surface, dst);
	}

	// free text surface
	SDL_FreeSurface(label_srf);
}

void PG_DrawObject::DrawText(const PG_Rect& r, const char* text, SDL_Color c, int align, TTF_Font* textfont) {

	// use default font
	if(textfont == NULL) {
		textfont = PG_Application::GetDefaultFont();
	}

	if(my_srfObject != NULL) {
		RenderTextRect(my_srfObject, textfont, r, text, c, align);
	}
	else {
		r.my_xpos += my_xpos;
		r.my_ypos += my_ypos;
		RenderTextRect(my_srfScreen, textfont, r, text, c, align);
	}
}

void PG_DrawObject::SetClipRect(PG_Rect& r) {
	my_rectClip = r;
}

void PG_DrawObject::GetClipRects(PG_Rect& src, PG_Rect& dst) {
	GetClipRects(src, dst, *this);
}

void PG_DrawObject::GetClipRects(PG_Rect& src, PG_Rect& dst, const PG_Rect& rect) {

	dst = IntersectRect(my_rectClip, rect);

	int dx = dst.my_xpos - rect.my_xpos;
	int dy = dst.my_ypos - rect.my_ypos;

	if(dx < 0) {
		dx = 0;
	}

	if(dy < 0) {
		dy = 0;
	}

	src.my_xpos = dx;
	src.my_ypos = dy;
	src.my_width = dst.my_width;
	src.my_height = dst.my_height;
}

PG_Rect PG_DrawObject::GetClipRect() {
	return my_rectClip;
}

void PG_DrawObject::SetPixel(int x, int y, Uint8 r, Uint8 g, Uint8 b) {
	static PG_Point p;

	if(my_srfObject == NULL) {
		p.x = my_xpos + x;
		p.y = my_ypos + y;
		if(my_rectClip.IsInside(p)) {
			SetPixel(p.x, p.y, r, g, b, my_srfScreen);
		}
	}
	else {
		SetPixel(x, y, r, g, b, my_srfObject);
	}
}

void PG_DrawObject::SetPixel(int x, int y, Uint8 r, Uint8 g, Uint8 b, SDL_Surface * surface) {
	static Uint8 old_r = 0;
	static Uint8 old_g = 0;
	static Uint8 old_b = 0;
	static Uint8 ri, gi, bi;
	
	static Uint32 pixel;
	static Uint8 *bits;
	static Uint8 bpp;

	static SDL_Rect rect;
	SDL_GetClipRect(surface, &rect);

	if((x < rect.x) || (y < rect.y)) {
		return;
	}

	if((x >= rect.x+rect.w) || (y >= rect.y+rect.h)) {
		return;
	}

	bpp = surface->format->BytesPerPixel;
	bits = ((Uint8 *) surface->pixels) + y * surface->pitch + x * bpp;

	if(!((old_r==r) && (old_g==g) && (old_b==b))) {
		pixel = SDL_MapRGB(surface->format, r, g, b);
		old_r = r;
		old_g = g;
		old_b = b;		
	}
	
	/* Set the pixel */
	switch (bpp) {
		case 1:
			*((Uint8 *) (bits)) = (Uint8) pixel;
			break;

		case 2:
			*((Uint16 *) (bits)) = (Uint16) pixel;
			break;

		case 3: {			/* Format/endian independent */
				ri = (pixel >> surface->format->Rshift) & 0xFF;
				gi = (pixel >> surface->format->Gshift) & 0xFF;
				bi = (pixel >> surface->format->Bshift) & 0xFF;
				*((bits) + surface->format->Rshift / 8) = ri;
				*((bits) + surface->format->Gshift / 8) = gi;
				*((bits) + surface->format->Bshift / 8) = bi;
			}
			break;

		case 4:
			*((Uint32 *) (bits)) = (Uint32) pixel;
			break;
	}
}

void PG_DrawObject::DrawHLine(int x, int y, int w, Uint8 r, Uint8 g, Uint8 b) {
	int x1 = x + w;

	for (int x0 = x; x0 < x1; x0++) {
		SetPixel(x0, y, r, g, b/*, surface*/);
	}
}

void PG_DrawObject::DrawVLine(int x, int y, int h, Uint8 r, Uint8 g, Uint8 b) {
	int y1 = y + h;

	for (int y0 = y; y0 < y1; y0++) {
		SetPixel(x, y0, r, g, b/*, surface*/);
	}
}

/**  */
void PG_DrawObject::DrawRectWH(int x, int y, int w, int h, Uint8 r, Uint8 g, Uint8 b) {

	DrawHLine(x, y, w, r, g, b);
	DrawHLine(x, y + h - 1, w, r, g, b);
	DrawVLine(x, y, h, r, g, b);
	DrawVLine(x + w - 1, y, h, r, g, b);

}

void PG_DrawObject::DrawBorder(const PG_Rect& r, int size, bool up) {
	int i0, i1;

	if(!IsVisible()) {
	    return;
	}
	
	i0 = (up) ? 0 : 1;
	i1 = (up) ? 1 : 0;

	// outer frame
	if (size >= 1) {
		DrawHLine(r.x + 0, r.y + 0, r.w, my_colorBorder[i0][0].r, my_colorBorder[i0][0].g, my_colorBorder[i0][0].b);
		DrawVLine(r.x + 0, r.y + 0, r.h - 1, my_colorBorder[i0][0].r, my_colorBorder[i0][0].g, my_colorBorder[i0][0].b);

		DrawHLine(r.x + 0, r.y + r.h - 1, r.w - 1, my_colorBorder[i1][0].r, my_colorBorder[i1][0].g, my_colorBorder[i1][0].b);
		DrawVLine(r.x + r.w - 1, r.y + 1, r.h - 1, my_colorBorder[i1][0].r, my_colorBorder[i1][0].g, my_colorBorder[i1][0].b);
	}
	// inner frame
	if (size >= 2) {
		DrawHLine(r.x + 1, r.y + 1, r.w - 1, my_colorBorder[i0][1].r, my_colorBorder[i0][1].g, my_colorBorder[i0][1].b);
		DrawVLine(r.x + 1, r.y + 1, r.h - 2, my_colorBorder[i0][1].r, my_colorBorder[i0][1].g, my_colorBorder[i0][1].b);

		DrawHLine(r.x + 1, r.y + r.h - 2, r.w - 2, my_colorBorder[i1][1].r, my_colorBorder[i1][1].g, my_colorBorder[i1][1].b);
		DrawVLine(r.x + r.w - 2, r.y + 2, r.h - 2, my_colorBorder[i1][1].r, my_colorBorder[i1][1].g, my_colorBorder[i1][1].b);
	}
}

SDL_Surface* PG_DrawObject::RenderText(const PG_Rect& rect, TTF_Font* font, const char* text, SDL_Color color, int align) {
	return NULL;
}

SDL_Surface* PG_DrawObject::RenderText(TTF_Font* font, const char* text, SDL_Color color) {
		if(text == 0) {
			return NULL;
		}

		if(text[0] == 0) {
			return NULL;
		}

		if(PG_Application::GetGLMode() || PG_Application::GetFontRendering()) {
			return TTF_RenderText_Solid(font, text, color);
		}
		else {
			return TTF_RenderText_Blended(font, text, color);
		}
}

void PG_DrawObject::DrawLine(Uint32 x0, Uint32 y0, Uint32 x1, Uint32 y1, SDL_Color color, Uint8 width, SDL_Surface* surface) {
	PG_DrawLine(surface, x0, y0, x1, y1, color, width);
}

void PG_DrawObject::DrawLine(Uint32 x0, Uint32 y0, Uint32 x1, Uint32 y1, SDL_Color color, Uint8 width) {
	SDL_Surface* surface = my_srfObject;

	if(surface == NULL) {
		surface = PG_Application::GetScreen();
		x0 += my_xpos;
		y0 += my_ypos;
		x1 += my_xpos;
		y1 += my_ypos;
	}

	PG_DrawLine(surface, x0, y0, x1, y1, color, width);
}
