/*
    ParaGUI - crossplatform widgetset
    Copyright (C) 2000,2001  Alexander Pipelka
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.
 
    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.
 
    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
    Alexander Pipelka
    pipelka@teleweb.at
 
    Last Update:      $Author: pipelka $
    Update Date:      $Date: 2001/06/28 08:38:08 $
    Source File:      $Source: /usr/local/CVSROOT/linux/paragui/src/pgapplication.cpp,v $
    CVS/RCS Revision: $Revision: 1.1.2.32 $
    Status:           $State: Exp $
*/

#include <iostream.h>
#include <string.h>

#include "pgapplication.h"
#include "pgdrawobject.h"
#include "pgwidget.h"

#ifdef HAVE_SDLIMAGE
#include "SDL_image.h"
#endif

// usually PARAGUI_THEMEDIR is defined through the configure script
// or passed to the compiler. This is just a kind of last resort.

#ifndef PARAGUI_THEMEDIR
#define PARAGUI_THEMEDIR	"./"
#endif	// PARAGUI_THEMEDIR

SDL_mutex* PG_Application::mutexScreen = NULL;
PG_Application* PG_Application::pGlobalApp = NULL;
SDL_Surface* PG_Application::screen = NULL;
TTF_Font* PG_Application::font = NULL;
std::string PG_Application::app_path = "";
PG_Theme* PG_Application::my_Theme = NULL;
bool PG_Application::bulkMode = false;
bool PG_Application::glMode = false;
bool PG_Application::emergencyQuit = false;
bool PG_Application::solidFontRendering = false;
bool PG_Application::enableBackground = true;
SDL_Surface* PG_Application::my_background = NULL;

SDL_Surface* PG_LoadImage(const char* file) {
#ifdef HAVE_SDLIMAGE
	return IMG_Load(file);
#else
	return SDL_LoadBMP(file);
#endif
}


/* We need to kludge a bit for keyboards work under windows */
/* we'll call SDL_RegisterApp right before SDL_Init()       */
/* Pete Shinners, Feb 1, 2001                               */

#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#define VC_EXTRALEAN
#include<windows.h>
extern int SDL_RegisterApp(char*, Uint32, void*);
#undef SendMessage 
#endif

void DrawTile(SDL_Surface* surface, const PG_Rect& ref, const PG_Rect& drawrect, SDL_Surface* tilemap) {
	PG_Point index1;
	PG_Point index2;
	PG_Rect oldclip;

	index1.x = (drawrect.x - ref.x) / tilemap->w;
	index1.y = (drawrect.y - ref.y) / tilemap->h;

	index2.x = ((drawrect.x - ref.x) + drawrect.w + tilemap->w - 1) / tilemap->w;
	index2.y = ((drawrect.y - ref.y) + drawrect.h + tilemap->h - 1) / tilemap->h;

	SDL_GetClipRect(surface, (SDL_Rect*)&oldclip);
	SDL_SetClipRect(surface, (SDL_Rect*)&drawrect);

	PG_Rect src(0,0, tilemap->w, tilemap->h);
	PG_Rect dst = src;

	for(int y = index1.y; y < index2.y; y++) {
		for(int x = index1.x; x < index2.x; x++) {

			dst.x = ref.x + x * tilemap->w;
			dst.y = ref.y + y * tilemap->h;

			PG_BlitSurface(tilemap, src, surface, dst);
		}
	}

	SDL_SetClipRect(surface, (SDL_Rect*)&oldclip);
}

PG_Application::PG_Application() {

	if(pGlobalApp != NULL) {
		cout << "ERROR: PG_Application Object already exists !\n";
		exit(-1);
	}


#ifdef WIN32
  SDL_RegisterApp("ParaGUI", 0, GetModuleHandle(NULL));
#endif

// -- see above

	/* Initialize the SDL library */
	if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_NOPARACHUTE) < 0) {
		cerr << "Could not initialize SDL: " << SDL_GetError() << endl;
		exit(-1);
	}

	pGlobalApp = this;
	screen = NULL;
	
	TTF_Init();

	mutexScreen = SDL_CreateMutex();
	my_background = NULL;
	my_freeBackground = false;
	my_backmode = BKMODE_TILE;

}

PG_Application::~PG_Application() {
	Shutdown();
}

/**  */
bool PG_Application::InitScreen(int w, int h, int depth, Uint32 flags) {

	if(depth == 0) {
		const SDL_VideoInfo* info = SDL_GetVideoInfo();
		if ( info->vfmt->BitsPerPixel > 8 ) {
			depth = info->vfmt->BitsPerPixel;
		}
	}

	if(SDL_VideoModeOK(w, h, depth, flags) == 0)
		return false;

	/* Initialize the display */
	PG_Application::screen = SDL_SetVideoMode(w, h, depth, flags);
	if (PG_Application::screen == NULL) {
		cerr << "Could not set video mode: " << SDL_GetError() << endl;
		return false;
	}

#ifdef DEBUG
	PrintVideoTest();
#endif // DEBUG

	SetScreen(screen);

	eventInit();

	return true;
}

/**  */
SDL_Thread* PG_Application::Run(bool threaded) {

	if(threaded) {
		SDL_Thread* thrd = SDL_CreateThread(PG_MessageObject::RunEventLoop, NULL);
		return thrd;
	}
	
	RunEventLoop(NULL);
	return NULL;
}

void PG_Application::Quit() {
	SendMessage(this, MSG_QUIT, PG_IDAPPLICATION, 0);
}

/**  */
bool PG_Application::eventKeyUp(const SDL_KeyboardEvent* key) {

	if((key->keysym.sym == SDLK_ESCAPE) && emergencyQuit) {

		Quit();
		return true;
	}

	return false;
}

bool PG_Application::eventResize(const SDL_ResizeEvent* event) {

	screen = SDL_SetVideoMode(
			event->w, event->h,
			screen->format->BitsPerPixel,
			screen->flags);

	PG_Widget::UpdateRect(PG_Rect(0,0,event->w,event->h));
	SDL_UpdateRect(screen,0,0,event->w,event->h);
	SendMessage(NULL, MSG_VIDEORESIZE, 0, 0);

	return true;
}

/**  */
SDL_Surface* PG_Application::SetScreen(SDL_Surface* surf) {
	PG_Application::screen = surf;

	glMode = (surf->flags & SDL_OPENGLBLIT);

	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
	SDL_EnableUNICODE(true);

	PG_Widget::UpdateRect(PG_Rect(0,0,screen->w,screen->h));
	SDL_UpdateRect(screen, 0,0,screen->w,screen->h);

	return PG_Application::screen;
}

bool PG_Application::LockScreen() {
	if(SDL_mutexP(mutexScreen) == -1)
		return false;

	return true;
}

/**  */
bool PG_Application::UnlockScreen() {
	return (SDL_mutexV(mutexScreen) == 0);
}

/**  */
bool PG_Application::SetBackground(const char* filename, int mode) {
	my_background = PG_LoadImage(GetRelativePath((char*)filename));

	if(my_background != NULL) {
		my_freeBackground = true;
		my_backmode = mode;
		RedrawBackground(PG_Rect(0,0,screen->w,screen->h));
		return true;
	} else {
		cout << "failed to load '" << GetRelativePath((char*)filename) << "'\n";
	}

	return true;
}

/**  */
bool PG_Application::SetBackground(SDL_Surface* surface, int mode) {
	if(surface == NULL) {
		return false;
	}

	if(my_freeBackground && my_background) {
		SDL_FreeSurface(my_background);
		my_freeBackground = false;
	}

	my_background = surface;
	my_backmode = mode;

	RedrawBackground(PG_Rect(0,0,screen->w,screen->h));
	return true;
}

/**  */
void PG_Application::RedrawBackground(const PG_Rect& rect) {
	static Uint32 color_black = SDL_MapRGBA(screen->format, 0,0,0,255);
	PG_Rect fillrect = rect;

	if(GetGLMode()) {
		SDL_FillRect(screen, (SDL_Rect*)&fillrect, color_black);
	}

	if(!enableBackground) {
		return;
	}

	PG_Application::LockScreen();

	if(!my_background) {
		SDL_FillRect(screen, (SDL_Rect*)&fillrect, color_black);
		PG_Application::UnlockScreen();
		return;
	}

	DrawTile(screen, PG_Rect(0,0,screen->w,screen->h), rect, my_background);

	PG_Application::UnlockScreen();
}

/**  */
const char* PG_Application::GetApplicationPath() {
	return app_path.c_str();
}

void PG_Application::SetApplicationPath(char* path) {
	app_path = path;
}

/**  */
const char* PG_Application::GetRelativePath(char* file) {
	static std::string buffer = "";

	if(FileExists(file)) {
		buffer = file;
	}
	else {
		buffer = GetApplicationPath();
		buffer += PG_FILE_SEPARATOR;
		buffer += file;
	}

	return buffer.c_str();
}

void PG_Application::FlipPage() {
	SDL_Flip(screen);
}

#ifdef DEBUG
// All calls to this are ifdefd out also
void PG_Application::PrintVideoTest() {
	const SDL_VideoInfo *info;
	int i;
	SDL_Rect **modes;

	info = SDL_GetVideoInfo();
	cout << "Current display: "
	<< info->vfmt->BitsPerPixel << " bits-per-pixel\n";
	if ( info->vfmt->palette == NULL ) {
		// FIXME: did I screw this up? :) -Dave
		cout << "\tRed Mask = 0x" << info->vfmt->Rmask << endl;
		cout << "\tGreen Mask = 0x" << info->vfmt->Gmask << endl;
		cout << "\tBlue Mask = 0x" << info->vfmt->Bmask << endl;
	}
	/* Print available fullscreen video modes */
	modes = SDL_ListModes(NULL, SDL_FULLSCREEN);
	if ( modes == (SDL_Rect **)0 ) {
		cout << "No available fullscreen video modes\n";
	} else
		if ( modes == (SDL_Rect **)-1 ) {
			cout << "No special fullscreen video modes\n";
		} else {
			cout << "Fullscreen video modes:\n";
			for ( i=0; modes[i]; ++i ) {
				cout << "\t" << modes[i]->w << "x" << modes[i]->h;
			}
		}
	if ( info->wm_available ) {
		cout << "A window manager is available\n";
	}
	if ( info->hw_available ) {
		cout << "Hardware surfaces are available ("
		<< info->video_mem << "K video memory)\n";
	}
	if ( info->blit_hw ) {
		cout <<
		"Copy blits between hardware surfaces are accelerated\n";
	}
	if ( info->blit_hw_CC ) {
		cout <<
		"Colorkey blits between hardware surfaces are accelerated\n";
	}
	if ( info->blit_hw_A ) {
		cout <<
		"Alpha blits between hardware surfaces are accelerated\n";
	}
	if ( info->blit_sw ) {
		cout <<
		"Copy blits from software surfaces \
		to hardware surfaces are accelerated\n";
	}
	if ( info->blit_sw_CC ) {
		cout <<
		"Colorkey blits from software surfaces \
		to hardware surfaces are accelerated\n";
	}
	if ( info->blit_sw_A ) {
		cout <<
		"Alpha blits from software surfaces \
		to hardware surfaces are accelerated\n";
	}
	if ( info->blit_fill ) {
		cout <<
		"Color fills on hardware surfaces are accelerated\n";
	}
}
#endif // DEBUG

void PG_Application::eventInit() {
}

TTF_Font* PG_Application::LoadFont(char* filename, int size) {
	return TTF_OpenFont(GetRelativePath(filename), size);
}

bool PG_Application::FileExists(const char* filename) {
	static FILE* test;

	test = fopen(filename, "r");

	if(test) {
		fclose(test);
		return true;
	}

	return false;
}

PG_Theme* PG_Application::LoadTheme(const char* xmltheme, bool asDefault, const char* searchpath) {
	PG_Theme* theme = NULL;
	
	cerr << "Locating theme '" << xmltheme << "' ..." << endl;
	
	// try to find the theme

	// MacOS does not use file path separator '/', instead ':' is used
	// There could be clever solution for this, but for a while...
	// let's assume that "data" directory must exist in working directory on MacOS.
	// Masahiro Minami<elsur@aaa.letter.co.jp>
	// 01/05/06

#ifndef macintosh
	if(searchpath != NULL) {
		cerr << searchpath << " ..." << endl;
		theme = PARAGUI_LoadTheme(searchpath, xmltheme);
	}

	if(theme == NULL) {
		cerr << "./ ..." << endl;
		theme = PARAGUI_LoadTheme(".", xmltheme);
		SetApplicationPath(".");
	}

	if(theme == NULL) {
		cerr << "./data/ ..." << endl;
		theme = PARAGUI_LoadTheme("./data", xmltheme);
		SetApplicationPath("./data");
	}

	if(theme == NULL) {
		cerr << "../data/ ..." << endl;
		theme = PARAGUI_LoadTheme("../data", xmltheme);
		SetApplicationPath("../data");
	}

	if(theme == NULL) {
		if(getenv("PARAGUIDIR") != NULL) {
			cerr << getenv("PARAGUIDIR") << " ..." << endl;
			theme = PARAGUI_LoadTheme(getenv("PARAGUIDIR"), xmltheme);
			SetApplicationPath(getenv("PARAGUIDIR"));
		}
	}

	if(theme == NULL) {
		cerr << PARAGUI_THEMEDIR << " ..." << endl;
		theme = PARAGUI_LoadTheme(PARAGUI_THEMEDIR, xmltheme);
		SetApplicationPath(PARAGUI_THEMEDIR);
	}
#else
	cerr << ":data" << endl;
	theme = PARAGUI_LoadTheme(":data", "default_mac"/*xmltheme*/);
	SetApplicationPath(":data");
#endif // macintosh

	if(theme && asDefault) {

		font = theme->FindDefaultFont();

		my_background = theme->FindSurface("Background", "Background", "background");
		my_backmode = theme->FindProperty("Background", "Background", "backmode");
	}
	else {
		cerr << "failed to load!" << endl;
	}
	
	if((my_Theme != NULL) && asDefault) {
		PARAGUI_UnloadTheme(my_Theme);
		my_Theme = NULL;
	}
	
	if(font == NULL) {
		cerr << "Unable to load default font" << endl;
		PARAGUI_UnloadTheme(theme);
		return NULL;
	}

	if(asDefault && theme) {
		my_Theme = theme;
	}

	return theme;
}

PG_Theme* PG_Application::GetTheme() {
	return my_Theme;
}

bool PG_Application::eventQuit(int id, PG_Widget* widget, unsigned long data) {
	my_quitEventLoop = true;
	return true;
}

void PG_Application::SetBulkMode(bool bulk) {
	bulkMode = bulk;
}

bool PG_Application::GetBulkMode() {
	return bulkMode;
}

void PG_Application::Shutdown() {

	// destroy still existing objects
	std::vector<PG_MessageObject*>::iterator list = objectList.begin();

	while(list != objectList.end()) {

		if(*list == this) {
			list++;
			continue;
		}

		PG_MessageObject* temp = *list;
		//RemoveObject(*list);
		delete temp;

		list = objectList.begin();
	}

	// unload theme (and free the allocated mem)
	if(my_Theme) {
		PARAGUI_UnloadTheme(my_Theme);
		font = NULL;
	}

	// empty the surfacecache
	PG_DrawObject::my_SurfaceCache.Cleanup();

	// destroy screen mutex
	SDL_DestroyMutex(mutexScreen);

	// shutdown ttf engine
	TTF_Quit();

	// shutdown SDL
	SDL_Quit();
}

void PG_Application::SetEmergencyQuit(bool esc) {
	emergencyQuit = esc;
}

void PG_Application::SetFontRendering(bool solid) {
	solidFontRendering = solid;
}

bool PG_Application::GetFontRendering() {
	return solidFontRendering;
}

bool PG_Application::GetGLMode() {
	return glMode;
}

SDL_Surface* PG_Application::GetScreen() {
	return screen;
}

TTF_Font* PG_Application::GetDefaultFont() {
	return font;
}

int PG_Application::GetScreenHeight() {
	return screen->h;
}

int PG_Application::GetScreenWidth() {
	return screen->w;
}

void PG_Application::EnableBackground(bool enable) {
	enableBackground = enable;
}

void PG_Application::SetIcon(const char *filename) {

 SDL_Surface *icon;
 Uint8       *pixels;
 Uint8       *mask;
 int          mlen, i;

 // Load the icon surface
 icon = PG_LoadImage(filename);
 if ( icon == NULL ) {
  cerr << "failed to load!" << endl;
  return;
 }

 // Check width and height
 if ( (icon->w%8) != 0 ) {
  cerr << "Icon width must be a multiple of 8!" << endl;
  SDL_FreeSurface(icon);
  return;
 }

 //Check the palette
 if ( icon->format->palette == NULL ) {
  cerr << "Icon must have a palette!" << endl;
  SDL_FreeSurface(icon);
  return;
 }

 // Set the colorkey
 SDL_SetColorKey(icon, SDL_SRCCOLORKEY, *((Uint8 *)icon->pixels));

 // Create the mask
 pixels = (Uint8 *)icon->pixels;
 mlen = icon->w*icon->h;
 mask = (Uint8 *)malloc(mlen/8);
 if ( mask == NULL ) {
  cerr << "Out of memory!" << endl;
  SDL_FreeSurface(icon);
  return;
 }
 memset(mask, 0, mlen/8);
 for ( i=0; i<mlen; ) {
  if ( pixels[i] != *pixels )
   mask[i/8] |= 0x01;
  ++i;
  if ( (i%8) != 0 )
   mask[i/8] <<= 1;
 }

 //Set icon
 if ( icon != NULL ) {
  SDL_WM_SetIcon(icon, mask);
 }

 //Clean up
 if ( mask != NULL )
  free(mask);

 return;
}

void PG_Application::SetCaption(const char *title, const char *icon) {
 SDL_WM_SetCaption(title,icon);
 return;
}

void PG_Application::GetCaption(char **title, char **icon) {
 SDL_WM_GetCaption(title,icon);
 return;
}

int PG_Application::Iconify(void) {
 return SDL_WM_IconifyWindow();
}
