/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001 Pan Development Team (pan@rebelbase.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gdk-pixbuf-loader.h>
#include <gmime/gmime.h>

#include <pan/base/argset.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-object.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/util-mime.h>
#include <pan/base/util-wrap.h>

#include <pan/articlelist.h>
#include <pan/gui.h>
#include <pan/gui-headers.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/message-window.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/task-body.h>
#include <pan/text.h>
#include <pan/util.h>

static gchar welcome[] =
{
"Pan " VERSION "\n"
"\n"
"  This is beta software.  If you find a bug, please report it.\n"
"\n"
"  http://pan.rebelbase.com/ -  Pan Homepage\n"
"  http://pan.rebelbase.com/bugs/ -  Report a Bug\n"
"  http://pan.rebelbase.com/download/ - Upgrade\n"
"\n"
"This program is free software; you can redistribute it \n"
"and/or modify it under the terms of the GNU General Public \n"
"License as published by the Free Software Foundation; \n"
"either version 2 of the License, or (at your option) any \n"
"later version. \n"
"\n"
"This program is distributed in the hope that it will be \n"
"useful, but WITHOUT ANY WARRANTY; without even the implied \n"
"warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR \n"
"PURPOSE.  See the GNU General Public License for more \n"
"details. \n"
"\n"
"The GNU Public License can be found from the menu above \n"
"in Help|About|License. \n"
};

static GtkTextBuffer * _text_buffer = NULL;
static GMimeMessage * current_mm = NULL;
static Article * current_article = NULL;
static GtkWidget * header_table = NULL;
static GtkWidget * text_vbox = NULL;
static GtkWidget * scrolled_window = NULL;

GdkColor text_fg_color;
GdkColor text_bg_color;
GdkColor text_quoted_color[3];
GdkColor text_url_color;
PanCallback * current_article_changed = NULL;

static void clear_text_buffer_nolock (GtkTextBuffer * buffer);
static void append_text_buffer_nolock (GtkTextBuffer*,
                                       const char*,
                                       gboolean mute_quotes);

/****
*****
*****    LOW-LEVEL TEXT UPDATE
*****
****/

static char*
get_url_from_location (GtkWidget * w, int x, int y)
{
	char * retval = NULL;
	gboolean clicked_on_url;
	GtkTextBuffer * text_buffer;
	static GtkTextTag * url_tag = NULL;
	GtkTextIter pos;

	/* get the buffer */
	g_return_val_if_fail (GTK_IS_TEXT_VIEW(w), FALSE);
	text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(w));

	/* did the user click on a url? */
	gtk_text_view_get_iter_at_location (GTK_TEXT_VIEW(w), &pos, x, y);
	if (url_tag == NULL) { 
		GtkTextTagTable * tag_table = gtk_text_buffer_get_tag_table (text_buffer);
		url_tag = gtk_text_tag_table_lookup (tag_table, "url");
	}
	clicked_on_url = gtk_text_iter_has_tag (&pos, url_tag);

	if (clicked_on_url)
	{
		GtkTextIter begin;
		GtkTextIter end;

		/* get the URL */
		begin = end = pos;
		gtk_text_iter_backward_to_tag_toggle (&begin, url_tag);
		gtk_text_iter_forward_to_tag_toggle (&end, url_tag);
		retval = gtk_text_iter_get_text (&begin, &end);
	}

	return retval;
}

static gboolean
text_button_pressed (GtkWidget * w, GdkEventButton * event, gpointer unused)
{
	g_return_val_if_fail (GTK_IS_TEXT_VIEW(w), FALSE);

	if (event->button==1 || event->button==2)
	{
		char * url = get_url_from_location (w, (int)event->x, (int)event->y);
		if (url != NULL)
		{
			pan_url_show (url);
			g_free (url);
		}
	}

	return FALSE;
}

static const char *
get_quote_tag (const char * utf8_line, int utf8_byte_len)
{
	const char * str = utf8_line;
	const char * line_end = utf8_line + utf8_byte_len;
	const char * retval = "quote_0";

	if (0<utf8_byte_len && is_nonempty_string(str))
	{
		int depth = 0;

		/* walk past leading spaces */
		while (str!=line_end && g_unichar_isspace(g_utf8_get_char(str)))
			str = g_utf8_next_char (str);

		/* count the number of spaces or quote characters */
		for (;;) {
			if (str == line_end)
				break;
			else if (is_quote_char((guchar)*str))
			       ++depth;
			else if (!g_unichar_isspace(g_utf8_get_char(str)))
				break;

			str = g_utf8_next_char (str);
		}

		if (!depth)
			retval = "quote_0";
		else switch (depth % 3) {
			case 1: retval = "quote_1"; break;
			case 2: retval = "quote_2"; break;
			case 0: retval = "quote_3"; break;
		}
	}

	return retval;
}

static void
clear_text_buffer_nolock (GtkTextBuffer * buffer)
{
	GtkTextIter start;
	GtkTextIter end;
	gtk_text_buffer_get_bounds (buffer, &start, &end);
	gtk_text_buffer_delete (buffer, &start, &end);
}

static void
append_text_buffer_nolock (GtkTextBuffer * text_buffer,
                           const char    * body,
		           gboolean        mute_quotes)
{
	int line_len;
	char * freeme1 = NULL;
	char * freeme2 = NULL;
	const char * line;
	const char * pch;
	const char * last_quote_begin = NULL;
	const char * quote_tag = NULL;
	const char * last_quote_tag = NULL;
	GtkTextIter mark_start;
	GtkTextIter mark_end;
	GtkTextIter start;
	GtkTextIter end;
	GtkTextMark * mark;
	debug_enter ("append_text_buffer_nolock");

	/* sanity checks */
	g_return_if_fail (text_buffer!=NULL);
	g_return_if_fail (GTK_IS_TEXT_BUFFER(text_buffer));

	/* mute the quoted text, if desired */
	if (mute_quotes)
		body = freeme1 = mute_quoted_text_in_body (body);

	body = pan_utf8ize (body, -1, &freeme2);

	/* insert the text */
	gtk_text_buffer_get_end_iter (text_buffer, &end);
	mark = gtk_text_buffer_create_mark (text_buffer, "blah", &end, TRUE);
	gtk_text_buffer_insert (text_buffer, &end, body, -1);
	gtk_text_buffer_get_iter_at_mark (text_buffer, &start, mark);
	gtk_text_buffer_delete_mark (text_buffer, mark);

	/* markup quotes */
	line = NULL;
	line_len = 0;
	pch = last_quote_begin = body;
	mark_start = start;
	while (get_next_token_run (pch, '\n', &pch, &line, &line_len)) {
		if (!line_len || (line_len==1 && *line=='\n'))
			continue;
		quote_tag = get_quote_tag (line, line_len);
		if (last_quote_tag!=NULL && strcmp (quote_tag, last_quote_tag)) {
			mark_end = mark_start;
			gtk_text_iter_forward_chars (&mark_end, g_utf8_strlen(last_quote_begin,line-1-last_quote_begin));
			gtk_text_buffer_apply_tag_by_name (text_buffer, last_quote_tag, &mark_start, &mark_end);
			mark_start = mark_end;
			gtk_text_iter_forward_chars (&mark_start, 1);
			last_quote_begin = line;
		}
		last_quote_tag = quote_tag;
	}
	if (last_quote_tag != NULL) {
		gtk_text_buffer_get_end_iter (text_buffer, &mark_end);
		gtk_text_buffer_apply_tag_by_name (text_buffer, last_quote_tag, &mark_start, &mark_end);
	}

	/* markup URLs */
	pch = body;
	while (pch!=NULL && (pch=strstr(pch,"http://"))) {
		const char * end = pch + 1;
		while (*end && !g_unichar_isspace(g_utf8_get_char(end)))
			end = g_utf8_next_char(end);
		mark_start = start;
		gtk_text_iter_forward_chars (&mark_start, g_utf8_strlen(body,pch-body));
		mark_end = mark_start;
		gtk_text_iter_forward_chars (&mark_end, g_utf8_strlen(pch,end-pch));
		gtk_text_buffer_remove_all_tags (text_buffer, &mark_start, &mark_end);
		gtk_text_buffer_apply_tag_by_name (text_buffer, "url", &mark_start, &mark_end);
		pch = *end ? end+1 : NULL;
	}

	/* cleanup */
	g_free (freeme1);
	g_free (freeme2);
	debug_exit ("append_text_buffer_nolock");
}

void
update_body_pane (GtkTextBuffer  * text_buffer,
                  const char     * body,
		  gboolean         mute_quotes)
{
	debug_enter ("update_body_pane");

	pan_lock ();
	clear_text_buffer_nolock (text_buffer);
	append_text_buffer_nolock (text_buffer, body, mute_quotes);
	pan_unlock ();

	debug_exit ("update_body_pane");
}


/**
***  Font
**/

static const gchar*
fontname (void)
{
	return text_use_fixed_font ? message_body_font_fixed
	                           : message_body_font;
}

void
text_set_font (void)
{
	pan_widget_set_font (GTK_WIDGET(Pan.text), fontname());
	text_refresh ();
}



/***
****
****   SPACE READING
****
***/

static void
sylpheed_textview_smooth_scroll_do (GtkAdjustment  * vadj,
                                    gfloat           old_value,
                                    gfloat           last_value,
                                    int              step)
{
	int i;
	int change_value;
	gboolean up;

	if (old_value < last_value) {
		change_value = last_value - old_value;
		up = FALSE;
	} else {
		change_value = old_value - last_value;
		up = TRUE;
	}

	//FIXME? gdk_key_repeat_disable ();

	for (i=step; i<=change_value; i+=step)
		gtk_adjustment_set_value (vadj, old_value+(up?-i:i));
	gtk_adjustment_set_value (vadj, last_value);

	//gdk_key_repeat_restore ();
}

void
text_read_more (void)
{
	const int arbitrary_font_height_pixels_hack = 18;
	Group * grouplist_group = grouplist_get_selected_group ();
	const Group * articlelist_group = articlelist_get_group ();
	const Article * a = articlelist_get_selected_article_nolock ();
	GtkAdjustment * v = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(scrolled_window));
	const gfloat val = CLAMP (v->value + v->page_size - arbitrary_font_height_pixels_hack,
	                          v->lower,
	                          MAX(v->upper,v->page_size)-MIN(v->upper,v->page_size));

	if (grouplist_group!=NULL && grouplist_group!=articlelist_group)
	{
		/* if user has changed group selection, change group. */
		articlelist_set_group (grouplist_group);
	}
	else if (queue_is_online() && a!=current_article)
	{
		/* if user has changed article selection, read the selection. */
		articlelist_read_selected ();
	}
	else if ((v->upper < v->page_size) || (val == v->value))
	{
		/* or if we're at the end of a page, go to the next page */
		articlelist_read_next ();
	}
	else if (Pan.text->parent==scrolled_window && text_window_smooth_scrolling)
	{
		/* or if smooth scrolling is turned on, scroll down a page */
		sylpheed_textview_smooth_scroll_do (v, v->value, val,
		                                    text_window_smooth_scrolling_speed);
	}
	else
	{
		/* or jump down a page */
		gtk_adjustment_set_value (v, val);
	}
}


/****
*****
*****   SETTING THE TEXT FROM A RAW TEXT MESSAGE
*****
****/

static int
text_set_raw_mainthread (gpointer data)
{
	GtkTextIter start;
	GtkTextIter end;
	char * text = (char*) data;
	debug_enter ("text_set_raw_mainthread");

	if (text == NULL)
		text = g_strdup (" ");

	pan_lock ();
	gtk_text_buffer_get_bounds (_text_buffer, &start, &end);
	gtk_text_buffer_delete (_text_buffer, &start, &end);
	gtk_text_buffer_insert (_text_buffer, &start, text, -1);
	pan_unlock ();

	g_free (text);
	debug_exit ("text_set_raw_mainthread");
	return 0;
}

void
text_set_raw (const char * text)
{
	gui_queue_add (text_set_raw_mainthread, g_strdup(text));
}


/****
*****
*****   SETTING THE TEXT FROM AN ARTICLE
*****
****/


/**
 * Generates a GtkPixmap object from a given GMimePart that contains an image.
 * Used for displaying attached pictures inline.
 */
static GdkPixbuf*
get_pixbuf_from_gmime_part (const GMimePart * part)
{
	guint len;
	const gchar * content;
	GdkPixbuf * pixbuf = NULL;
	GdkPixbufLoader * l = NULL;

	/* create the loader */
	l = gdk_pixbuf_loader_new ();

	/* create the pixbuf */
	content = g_mime_part_get_content (part, &len);
	gdk_pixbuf_loader_write (l, (const guchar*)content, len, NULL);
	pixbuf = gdk_pixbuf_loader_get_pixbuf (l);
	gdk_pixbuf_loader_close (l, NULL);

	/* cleanup */
	g_object_ref (G_OBJECT(pixbuf));
	g_object_unref (G_OBJECT(l));

	return pixbuf;
}

typedef struct
{
	const Article * article;
	GtkTextBuffer * buffer;
}
InsertPartStruct;

static void
insert_part_partfunc (GMimeObject * obj, gpointer ips_gpointer)
{
	GMimePart * part;
	InsertPartStruct * ips;
	const GMimeContentType * type;
	YencInfo * yenc;

	/* We are only looking for leaf parts... */
	if (!GMIME_IS_PART (obj))
		return;
	
	part = GMIME_PART (obj);
	ips = (InsertPartStruct*) ips_gpointer;
	type = g_mime_object_get_content_type (GMIME_OBJECT (part));
	yenc = GMIME_OBJECT(obj)->user_data;
	
	if (yenc!= NULL && part->content!=NULL && part->content->stream!=NULL) {
		GMimeStream * stream = g_mime_stream_filter_new_with_stream (part->content->stream);
		g_mime_stream_filter_add (GMIME_STREAM_FILTER(stream),
			g_mime_filter_yenc_new (GMIME_FILTER_YENC_DIRECTION_DECODE));
		g_mime_data_wrapper_set_stream (part->content, stream);
	}

	if (g_mime_content_type_is_type (type, "image", "*"))
	{
		GdkPixbuf * pixbuf = get_pixbuf_from_gmime_part (part);

		if (pixbuf != NULL)
		{
			GtkTextIter iter;
			gtk_text_buffer_get_end_iter (ips->buffer, &iter);
			gtk_text_buffer_insert_pixbuf (ips->buffer, &iter, pixbuf);
			gtk_text_buffer_insert (ips->buffer, &iter, "\n\n", -1);
			g_object_unref (pixbuf);
		}
	}
	else if (g_mime_content_type_is_type (type, "text", "plain")
	      || g_mime_content_type_is_type (type, "text", "html"))
	{
		guint len;
		const char * content;
		const char * charset;
		const char * default_charset;
		char * str;
		GError * err = NULL;

		/* get it as a string */
		content = g_mime_part_get_content (part, &len);
		charset = g_mime_object_get_content_type_parameter (GMIME_OBJECT (part), "charset");

		/* try to convert the article to utf-8 */
		default_charset = group_get_default_charset (ips->article->group);
		if (!is_nonempty_string (charset))
			charset = default_charset;
		str = g_convert (content, len, "UTF-8", charset, NULL, NULL, &err);

		/* if conversion failed but we haven't tried the default charset yet, try it */
		if (err!=NULL && charset!=default_charset) {
			g_error_free (err);
			err = NULL;
			g_free (str);
			str = g_convert (content, len, "UTF-8", default_charset, NULL, NULL, &err);
		}

		/* report errors */
		if (err != NULL) {
			log_add_va (LOG_ERROR, "Unable to convert article to UTF-8: %s", err->message);
			log_add (LOG_ERROR, "Some parts of the article might not be displayed.");
			g_error_free (err);
		}

		if (is_nonempty_string(str))
		{
			if (text_get_wrap())
				replace_gstr (&str, fill_body(str,wrap_column));      
		}

       		append_text_buffer_nolock (ips->buffer, str, text_get_mute_quoted());
		g_free (str);
	}
}

static void
set_text_from_article_nolock (const Article * article)
{
	debug_enter ("set_text_from_article");

	/* upper headers */
	gui_headers_set_default_nolock (header_table, article);

	/* clear */
	clear_text_buffer_nolock (_text_buffer);

	/* add headers */
	if (article!=NULL && text_get_show_all_headers())
	{
		const char * charset;
		char * pch = article_get_headers (article);

		charset = article_get_extra_header (article, PAN_CHARSET);
		if (charset == NULL)
			charset = group_get_default_charset (article->group);

		if (charset != NULL) {
			char * utf8 = g_convert (pch, -1, "UTF-8", charset, NULL, NULL, NULL);

			if (pch) {
				g_free (pch);
				pch = utf8;
			}
		}
		append_text_buffer_nolock (_text_buffer, pch, FALSE);
		append_text_buffer_nolock (_text_buffer, "\n\n", FALSE);
		g_free (pch);
	}

	/* add body */
	if (current_mm)
	{
		InsertPartStruct ips;
		ips.article = article;
		ips.buffer = _text_buffer;
		if (GMIME_IS_MULTIPART (current_mm->mime_part))
			g_mime_multipart_foreach (GMIME_MULTIPART (current_mm->mime_part), insert_part_partfunc, &ips);
		else
			insert_part_partfunc (current_mm->mime_part, &ips);
	}

	gtk_adjustment_set_value (
			gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(scrolled_window)),
			0.0);

	/* cleanup */
	debug_exit ("set_text_from_article_nolock");
}

/****
*****
*****  CURRENT ARTICLE
*****
****/

Article*
get_current_article (void)
{
	return current_article;
}

static int
set_current_article_impl (Article * a, const char * text)
{
	Article * old_article;
	GMimeMessage * old_mm;

	debug_enter ("set_current_article_mainthread");

	/**
	***  Update the current article
	**/

	old_mm = current_mm;
	old_article = current_article;
	current_article = a;
	current_mm = NULL;

	/**
	***  If we have a new article, parse it
	**/

	if (text != NULL)
		current_mm = pan_g_mime_parser_construct_message (&text, 1);

	/**
	***  Fire an event that the current article has changed
	**/

	pan_callback_call (current_article_changed, old_article, current_article);

	/**
	***  Clean out the old article
	**/

	if (old_mm != NULL)
	{
		g_mime_object_unref (GMIME_OBJECT(old_mm));
	}
	if (old_article != NULL)
	{
		group_unref_articles (old_article->group, NULL);
	}

	/**
	***  Set up the new article
	**/

	if (current_article != NULL)
	{
		pan_lock ();
		set_text_from_article_nolock (current_article);
		pan_unlock ();

		articles_set_read (&current_article, 1, TRUE);
		gui_queue_add ((GSourceFunc)gui_page_set, GINT_TO_POINTER(BODY_PANE));
	}
	else
	{
		pan_lock ();
		gui_headers_set_default_nolock (header_table, NULL);
		set_text_from_article_nolock (NULL);
		pan_unlock ();
	}


	debug_exit ("set_current_article_mainthread");
	return 0;
}

static int
set_current_article_mainthread (gpointer data)
{
	gchar * text = NULL;
	Article * a = ARTICLE(data);

	if (a != NULL)
		text = article_get_message (a);

	set_current_article_impl (a, text);	

	g_free (text);
	return 0;
}

static void
set_current_article (Article * article)
{
	debug_enter ("set_current_article");

	/* sanity clause */
	if (article!=NULL)
		g_return_if_fail (article_is_valid(article));

	/**
	***  Ref the articles s.t. we know that this article will
	***  still be loaded when set_current_article_mainthread runs,
	***  then push the GUI work over into the main thread.
	**/

	if (article!=NULL)
		group_ref_articles (article->group, NULL);

	gui_queue_add (set_current_article_mainthread, article);


	debug_enter ("set_current_article");
}

void
text_clear_nolock (void)
{
	GMimeMessage * old_mm;
	Article * old_article;

	/* update model */
	old_mm = current_mm;
	old_article = current_article;
	current_article = NULL;
	current_mm = NULL;
	pan_callback_call (current_article_changed, old_article, current_article);

	/* clear out old */
	if (old_mm != NULL)
		g_mime_object_unref (GMIME_OBJECT(old_mm));
	if (old_article != NULL)
		group_unref_articles (old_article->group, NULL);

	/* update UI */
	gui_headers_set_default_nolock (header_table, NULL);
	set_text_from_article_nolock (NULL);
}

void
text_refresh (void)
{
	if (Pan.text != NULL) {
		if (current_article != NULL)
			text_set_from_cached_article (current_article);
	}
}

void
text_set_from_cached_article (Article * article)
{
	debug_enter ("text_set_from_cached_article");

	g_return_if_fail (article!=NULL);
	g_return_if_fail (article_has_body(article));
	set_current_article (article);

	debug_exit ("text_set_from_cached_article");
}

void
text_set_from_article (Article *article)
{
	debug_enter ("text_set_from_article");

	g_return_if_fail (article!=NULL);

	if (article_has_body (article))
		text_set_from_cached_article (article);
	else
		queue_add (TASK(task_body_new(article)));

	debug_exit ("text_set_from_article");
}

/****
*****
*****    CALLBACKS
*****
****/

static void
articlelist_group_changed_cb (gpointer call_obj, gpointer call_arg, gpointer client_data)
{
	Group * group = GROUP(call_arg);

	if (current_article!=NULL && current_article->group!=group)
		set_current_article (NULL);
}

static int
compare_pgchar_pparticle_msgid (const void * a, const void *b)
{
	const gchar * msgid_a = (const gchar*) a;
	const gchar * msgid_b = article_get_message_id (*(const Article**)b);
	pan_warn_if_fail (is_nonempty_string(msgid_a));
	pan_warn_if_fail (is_nonempty_string(msgid_b));
	return pan_strcmp (msgid_a, msgid_b);
}     

static void
group_articles_removed_cb (gpointer call_obj, gpointer call_arg, gpointer client_data)
{
	Group * group = GROUP(call_obj);
	GPtrArray * removed = (GPtrArray*) call_arg;
	debug_enter ("group_articles_removed_cb");

	/* sanity checks */
	g_return_if_fail (group!=NULL);
	g_return_if_fail (removed!=NULL);
	g_return_if_fail (removed->len>0u);

	/* unset the current article if we need to */
	if (current_article!=NULL)
	{
		gboolean exact_match = FALSE;

		lower_bound (article_get_message_id(current_article),
		             removed->pdata,
		             removed->len,
			     sizeof(gpointer),
			     compare_pgchar_pparticle_msgid,
			     &exact_match);

		if (exact_match)
			set_current_article (NULL);
	}

	debug_exit ("group_articles_removed_cb");
}

void
text_rot13_selected_text_nolock (void)
{
	GtkTextIter sel_start;
	GtkTextIter sel_end;

	debug_enter ("text_set_rot13");

	if (gtk_text_buffer_get_selection_bounds (_text_buffer, &sel_start, &sel_end))
	{
		char * text = gtk_text_iter_get_text (&sel_start, &sel_end);
		rot13_inplace (text);
		gtk_text_buffer_delete (_text_buffer, &sel_start, &sel_end);
		gtk_text_buffer_insert (_text_buffer, &sel_end, text, -1);
		g_free (text);
	}

	debug_exit ("text_set_rot13");
}

/****
*****
*****    TEXT WIDGET STARTUP
*****
****/


GtkWidget *
text_create (void)
{
	GtkWidget * text_view;

	/* create the text view */
        text_view = gtk_text_view_new_with_buffer (_text_buffer);
	if (!use_system_bg)
		gtk_widget_modify_base (text_view, GTK_STATE_NORMAL, &text_bg_color);
        gtk_container_set_border_width (GTK_CONTAINER(text_view), GUI_PAD_SMALL);
        gtk_text_view_set_editable (GTK_TEXT_VIEW(text_view), FALSE);
	g_signal_connect (text_view, "button_press_event",
	                  G_CALLBACK(text_button_pressed), NULL);

	/* set up the buffer tags */
	_text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(text_view));
	gtk_text_buffer_create_tag (_text_buffer, "center",
	                            "justification", GTK_JUSTIFY_CENTER,
	                            NULL);
	/* add the new tags */
	gtk_text_buffer_create_tag (_text_buffer, "url",
	                            "underline", PANGO_UNDERLINE_SINGLE,
	                            "foreground_gdk", &text_url_color,
				    NULL);
	if (use_system_fg)
		gtk_text_buffer_create_tag (_text_buffer, "quote_0",
					    NULL);
	else
		gtk_text_buffer_create_tag (_text_buffer, "quote_0",
					    "foreground_gdk", &text_fg_color,
					    NULL);
	gtk_text_buffer_create_tag (_text_buffer, "quote_1",
	                            "foreground_gdk", &text_quoted_color[0],
				    NULL);
	gtk_text_buffer_create_tag (_text_buffer, "quote_2",
	                            "foreground_gdk", &text_quoted_color[1],
				    NULL);
	gtk_text_buffer_create_tag (_text_buffer, "quote_3",
	                            "foreground_gdk", &text_quoted_color[2],
				    NULL);
#warning need some way to refresh these tags when user changes text color via prefs

	current_article_changed = pan_callback_new ();

	/* the larger scrolled window */
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scrolled_window), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	text_vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(scrolled_window), text_view);
        pan_widget_set_font (GTK_WIDGET(text_view), fontname());
	append_text_buffer_nolock (_text_buffer, welcome, FALSE);

	/* text_box holds the header info and the scrolled text window */
        text_box = gtk_vbox_new (FALSE, 4);
	gtk_container_set_border_width (GTK_CONTAINER(text_box), 4);
        header_table = gtk_table_new ( 0, 0, FALSE);
        gtk_box_pack_start (GTK_BOX(text_box), header_table, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX(text_box), scrolled_window, TRUE, TRUE, 0); 

	pan_callback_add (group_get_articles_removed_callback(),
	                  group_articles_removed_cb, NULL);
	pan_callback_add (articlelist_get_group_changed_callback(),
	                  articlelist_group_changed_cb, NULL);


	Pan.text = text_view;
	return text_box;
}

/****
*****
*****    UTILITY FUNCTIONS FOR OUTSIDE CLIENTS
*****
****/

char*
text_get_message_to_reply_to (void)
{
	GtkTextIter sel_start;
	GtkTextIter sel_end;
	char * body = NULL;
	gboolean has_selection;
	debug_enter ("text_get_message_to_reply_to");

	/* get the selected text, if any */
	pan_lock ();
	has_selection = gtk_text_buffer_get_selection_bounds (_text_buffer, &sel_start, &sel_end);
	if (has_selection)
		body = gtk_text_iter_get_text (&sel_start, &sel_end);
	else
	{
		gtk_text_buffer_get_start_iter (_text_buffer, &sel_start);
		gtk_text_buffer_get_end_iter (_text_buffer, &sel_end);
		body = gtk_text_buffer_get_text (_text_buffer, &sel_start, &sel_end, FALSE);
	}
	pan_unlock ();
 
	/* no selection, so user is replying to whole message sans signature */
	if (!has_selection && body!=NULL)
	{
		gchar * pch = pan_strstr (body, "\n-- \n");
		if (pch != NULL)
			pch[1] = '\0';
	}

	debug_exit ("text_get_message_to_reply_to");
	return body;
}

/****
*****
*****    MANIPULATORS:  WRAPPING
*****
****/

static gboolean _do_wrap = FALSE;

static PanCallback * _text_fill_body_changed_callback = NULL;

PanCallback*
text_get_fill_body_changed_callback (void)
{
	if (_text_fill_body_changed_callback == NULL)
		_text_fill_body_changed_callback = pan_callback_new ();

	return _text_fill_body_changed_callback;
}

void
text_set_wrap (gboolean wrap)
{
	debug_enter ("text_set_wrap");

	if (wrap != _do_wrap)
	{
		_do_wrap = wrap;

		pan_callback_call (text_get_fill_body_changed_callback(), NULL, GINT_TO_POINTER(wrap));
		text_refresh ();                            
	}

	debug_exit ("text_set_wrap");
}

gboolean
text_get_wrap (void)
{
	return _do_wrap;
}

/****
*****
*****    MANIPULATORS:  HEADERS
*****
****/

static gboolean _show_all_headers;

static PanCallback * _show_all_headers_changed_callback = NULL;

PanCallback*
text_get_show_all_headers_changed_callback (void)
{
	if (_show_all_headers_changed_callback == NULL)
		_show_all_headers_changed_callback = pan_callback_new ();

	return _show_all_headers_changed_callback;
}

void
text_set_show_all_headers (gboolean show)
{
	debug_enter ("text_set_show_all_headers");

	if (_show_all_headers != show)
	{
		_show_all_headers = show;
		pan_callback_call (text_get_show_all_headers_changed_callback(), NULL, GINT_TO_POINTER(show));
		text_refresh ();
	}

	debug_exit ("text_set_show_all_headers");
}

gboolean
text_get_show_all_headers (void)
{
	return _show_all_headers;
}

/****
*****
*****    MANIPULATORS:  MUTE QUOTED
*****
****/

static gboolean _mute_quoted = FALSE;

static PanCallback * _mute_quoted_changed_callback = NULL;

PanCallback*
text_get_mute_quoted_changed_callback (void)
{
	if (_mute_quoted_changed_callback == NULL)
		_mute_quoted_changed_callback = pan_callback_new ();

	return _mute_quoted_changed_callback;
}

void
text_set_mute_quoted (gboolean quoted)
{
	debug_enter ("text_set_mute_quoted");

	if (_mute_quoted != quoted)
	{
		_mute_quoted = quoted;
		pan_callback_call (text_get_mute_quoted_changed_callback(), NULL, GINT_TO_POINTER(quoted));
		text_refresh ();
	}

	debug_exit ("text_set_mute_quoted");
}

gboolean
text_get_mute_quoted (void)
{
	return _mute_quoted;
}

/**/

void
text_select_all (void)
{
	GtkTextIter start;
	GtkTextIter end;

	pan_lock();
	gtk_text_buffer_get_bounds (_text_buffer, &start, &end);
	gtk_text_buffer_move_mark (_text_buffer, gtk_text_buffer_get_insert(_text_buffer), &end);
	gtk_text_buffer_move_mark (_text_buffer, gtk_text_buffer_get_selection_bound (_text_buffer), &start);
	pan_unlock();
}

void
text_deselect_all (void)
{
	GtkTextIter start;
	GtkTextIter end;

	pan_lock();
	gtk_text_buffer_get_bounds (_text_buffer, &start, &end);
	gtk_text_buffer_move_mark (_text_buffer, gtk_text_buffer_get_insert(_text_buffer), &end);
	gtk_text_buffer_move_mark (_text_buffer, gtk_text_buffer_get_selection_bound (_text_buffer), &end);
	pan_unlock();
}
