/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <pan/base/pan-i18n.h>
#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/server.h>
#include <pan/base/serverlist.h>

#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/server-ui.h>
#include <pan/util.h>

/**
***  MENU
**/

extern GtkItemFactory * _main_menu_factory;

static gboolean server_activate_registered = FALSE;
static GHashTable * server_to_menubutton = NULL;

static int
sort_servers_by_name (gconstpointer va, gconstpointer vb, gpointer unused)
{
	const Server * a = *(const Server **)va;
	const Server * b = *(const Server **)vb;
	return g_strcasecmp (a->name, b->name);
}

static void
server_selected_cb (gpointer user_data, int index, GtkWidget * w)
{
	Server * cur_server;
	Server * sel_server;
	GPtrArray * a;

	/* get the selected server */
	a = g_ptr_array_new ();
	serverlist_get_servers (a);
	sel_server = 0<=index && index<a->len ? SERVER(g_ptr_array_index(a, index)) : NULL;
	g_ptr_array_free (a, TRUE);

	/* get the cur server */
	cur_server = serverlist_get_active_server ();

	/* set the server */
	if (sel_server!=cur_server && GTK_CHECK_MENU_ITEM(w)->active)
		serverlist_set_active_server (sel_server);
}

static void
server_activated_cb (gpointer server_gp, gpointer foo, gpointer bar)
{
	char * path = NULL;
	GtkWidget * w = NULL;
	Server * server = (Server*) server_gp;

	if (server != NULL)
		path = g_strdup_printf ("/Servers/Select Server/%s", server_get_name (server));
	if (path != NULL)
		w = gtk_item_factory_get_widget (_main_menu_factory, path);
	if (w != NULL)
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(w), TRUE);
}


void
server_menu_update (void)
{
	int i;
	Server * active_server;
	GPtrArray * servers;

	static GStaticMutex entries_mutex = G_STATIC_MUTEX_INIT;
	static GtkItemFactoryEntry * entries = NULL;
	static int qty = 0;

	g_static_mutex_lock (&entries_mutex);

	/* make sure we update the menu when a server is activated */
	if (!server_activate_registered) {
		server_activate_registered = TRUE;
		pan_callback_add (serverlist_get_server_activated_callback(),
		                  server_activated_cb,
		                  NULL);
	}

	/* remove any old server buttons */
	if (entries != NULL) {
		gtk_item_factory_delete_entries (_main_menu_factory, qty, entries);
		for (i=0; i<qty; ++i)
			g_free (entries[i].path);
		g_free (entries);
		entries = NULL;
		qty = 0;
	}

	/* build the new menu */
	if (server_to_menubutton != NULL)
		g_hash_table_destroy (server_to_menubutton);
	server_to_menubutton = g_hash_table_new (g_direct_hash, g_direct_equal);

	/* get a list of servers, sorted alphabetically */
	servers = g_ptr_array_new ();
	serverlist_get_servers (servers);
	active_server = serverlist_get_active_server ();
	if (servers->len)
	{
		const int alloc_qty = servers->len;
		char * active_path = NULL;
		char * radio_path;

		g_ptr_array_sort_with_data (servers, sort_servers_by_name, NULL);

		/* build the ItemFactoryEntries */
		entries = g_new0 (GtkItemFactoryEntry, alloc_qty);
		for (i=qty=0; i<servers->len; ++i)
		{
			const Server * server = SERVER(g_ptr_array_index(servers,i));
			const gchar  * name = server_get_name (server);

			/* skip 'invisible' servers used for folders, etc. */
			if (!strncmp (name, INTERNAL_SERVER_NAME, strlen(INTERNAL_SERVER_NAME)))
				continue;

			/* build an entry: the callback arg is the index into the servers array */
			entries[qty].path = g_strdup_printf ("/Servers/Select Server/%s", name);
			entries[qty].item_type = qty==0 ? "<RadioItem>" : radio_path;
			entries[qty].callback = server_selected_cb;
			entries[qty].callback_action = i;
			if (qty==0)
				radio_path = entries[qty].path;
			if (server == active_server)
				active_path = entries[qty].path;

			++qty;
		}

		/* add the servers */
		gtk_item_factory_create_items (_main_menu_factory, qty, entries, NULL);

		/* set the menu button that corresponds to the current active server */
		if (active_path != NULL) {
			GtkWidget * w = gtk_item_factory_get_widget (_main_menu_factory, active_path);
			gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(w), TRUE);
		}

	}

	/* cleanup */
	g_ptr_array_free (servers, TRUE);
	g_static_mutex_unlock (&entries_mutex);
}
