/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Pan - A Newsreader for X
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/filters/filter.h>
#include <pan/filters/filter-manager.h>

#include <pan/filters/filter-aggregate.h>
#include <pan/filters/filter-binary.h>
#include <pan/filters/filter-cached.h>
#include <pan/filters/filter-manager.h>
#include <pan/filters/filter-mine.h>
#include <pan/filters/filter-new.h>
#include <pan/filters/filter-phrase.h>
#include <pan/filters/filter-saved.h>
#include <pan/filters/filter-thread.h>
#include <pan/filters/filter-top.h>
#include <pan/filters/filter-manager.h>

#include <pan/xpm/pan-pixbufs.h>

#include <pan/util.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	GtkWidget * dialog;

	GtkWidget * attach_tb_complete;
	GtkWidget * attach_tb_incomplete;
	GtkWidget * attach_tb_text;

	GtkWidget * age_tb_new;
	GtkWidget * age_tb_unread;
	GtkWidget * age_tb_read;

	GtkWidget * author_tb_me;
	GtkWidget * author_tb_others;

	GtkWidget * availability_tb_local;
	GtkWidget * availability_tb_server;

	GtkWidget * action_tb_saved;
	GtkWidget * action_tb_queued;
	GtkWidget * action_tb_idle;

	GtkWidget * attention_tb_watched;
	GtkWidget * attention_tb_normal;
	GtkWidget * attention_tb_ignored;

	GtkWidget * show_thread;
	GtkWidget * show_article;
	GtkWidget * show_article_and_replies;
	GtkWidget * show_article_and_references;

	GtkWidget * negate;

	gulong bits;
	FilterShow show;
}
DialogImpl;

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
************/

/*****
******
*****/

/************
*************  PROTECTED
************/

static gboolean
is_active (GtkWidget * w)
{
	return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(w));
}

static void
set_active (GtkWidget * w, gboolean b)
{
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
}

static gulong
get_filter_bits (DialogImpl * d)
{
	gulong l = 0;
	debug_enter ("get_filter_bits");

	if (is_active (d->attach_tb_complete))
		l |= STATE_FILTER_COMPLETE_BINARIES;
	if (is_active (d->attach_tb_incomplete))
		l |= STATE_FILTER_INCOMPLETE_BINARIES;
	if (is_active (d->attach_tb_text))
		l |= STATE_FILTER_NONBINARIES;

	if (is_active (d->age_tb_new))
		l |= STATE_FILTER_NEW;
	if (is_active (d->age_tb_read))
		l |= STATE_FILTER_READ;
	if (is_active (d->age_tb_unread))
		l |= STATE_FILTER_UNREAD;

	if (is_active (d->author_tb_me))
		l |= STATE_FILTER_MINE;
	if (is_active (d->author_tb_others))
		l |= STATE_FILTER_NOT_MINE;

	if (is_active (d->availability_tb_local))
		l |= STATE_FILTER_CACHED;
	if (is_active (d->availability_tb_server))
		l |= STATE_FILTER_NOT_CACHED;

	if (is_active (d->action_tb_saved))
		l |= STATE_FILTER_SAVED;
	if (is_active (d->action_tb_queued))
		l |= STATE_FILTER_IDLE;
	if (is_active (d->action_tb_idle))
		l |= STATE_FILTER_QUEUED;

	if (is_active (d->attention_tb_watched))
		l |= STATE_FILTER_WATCHED;
	if (is_active (d->attention_tb_normal))
		l |= STATE_FILTER_NORMAL_RANK;
	if (is_active (d->attention_tb_ignored))
		l |= STATE_FILTER_IGNORED;

	debug_exit ("get_filter_bits");
	return l;
}

static void
set_filter_bit_buttons (DialogImpl * d, gulong l)
{
	set_active (d->age_tb_new, (l&STATE_FILTER_NEW)?1:0);
	set_active (d->age_tb_unread, (l&STATE_FILTER_UNREAD)?1:0);
	set_active (d->age_tb_read, (l&STATE_FILTER_READ)?1:0);

	set_active (d->attach_tb_complete, (l&STATE_FILTER_COMPLETE_BINARIES)?1:0);
	set_active (d->attach_tb_incomplete, (l&STATE_FILTER_INCOMPLETE_BINARIES)?1:0);
	set_active (d->attach_tb_text, (l&STATE_FILTER_NONBINARIES)?1:0);

	set_active (d->attention_tb_watched, (l&STATE_FILTER_WATCHED)?1:0);
	set_active (d->attention_tb_ignored, (l&STATE_FILTER_IGNORED)?1:0);
	set_active (d->attention_tb_normal, (l&STATE_FILTER_NORMAL_RANK)?1:0);

	set_active (d->action_tb_saved, (l&STATE_FILTER_SAVED)?1:0);
	set_active (d->action_tb_queued, (l&STATE_FILTER_QUEUED)?1:0);
	set_active (d->action_tb_idle, (l&STATE_FILTER_IDLE)?1:0);

	set_active (d->availability_tb_local, (l&STATE_FILTER_CACHED)?1:0);
	set_active (d->availability_tb_server, (l&STATE_FILTER_NOT_CACHED)?1:0);

	set_active (d->author_tb_me, (l&STATE_FILTER_MINE)?1:0);
	set_active (d->author_tb_others, (l&STATE_FILTER_NOT_MINE)?1:0);
}

static FilterShow
get_filter_show (DialogImpl * d)
{
	FilterShow retval = 0;

	if (is_active (d->show_article))
		retval = FILTER_SHOW_MATCHES;
	else if (is_active (d->show_article_and_replies))
		retval = FILTER_SHOW_MATCHES_AND_REPLIES;
	else if (is_active (d->show_article_and_references))
		retval = FILTER_SHOW_MATCHES_AND_REFERENCES;
	else
		retval = FILTER_SHOW_THREAD;

	return retval;
}

static void
set_filter_show_buttons (DialogImpl * d, FilterShow show)
{
	GtkWidget * w = 0;

	switch (show) {
		case FILTER_SHOW_MATCHES:
			w = d->show_article;
			break;
		case FILTER_SHOW_MATCHES_AND_REPLIES:
			w = d->show_article_and_replies;
			break;
		case FILTER_SHOW_MATCHES_AND_REFERENCES:
			w = d->show_article_and_references;
			break;
		case FILTER_SHOW_THREAD:
			w = d->show_thread;
			break;
		default:
			pan_warn_if_reached ();
	}

	if (w != NULL)
		set_active (w, TRUE);
}

/************
*************  PUBLIC
************/

void
filter_current_dialog_get_bits (GtkWidget     * dialog,
                                gulong        * setme_bits,
                                FilterShow    * setme_show,
                                gboolean      * setme_negate)
{
	DialogImpl * d = (DialogImpl*) g_object_get_data (G_OBJECT(dialog), "impl");

	*setme_bits = get_filter_bits (d);
	*setme_show = get_filter_show (d);
	*setme_negate = is_active (d->negate);
}


Filter*
filter_current_dialog_get_filter (gulong bits, gboolean negate)
{
	Filter * retval;
	FilterAggregate * top;
	const gulong l = bits;
	const gboolean show_new = (l & STATE_FILTER_NEW) != 0;
	const gboolean show_unread = (l & STATE_FILTER_UNREAD) != 0;
	const gboolean show_read = (l & STATE_FILTER_READ) != 0;
	const gboolean show_bin = (l & STATE_FILTER_COMPLETE_BINARIES) != 0;
	const gboolean show_inc_bin = (l & STATE_FILTER_INCOMPLETE_BINARIES) != 0;
	const gboolean show_nonbinary = (l & STATE_FILTER_NONBINARIES) != 0;
	const gboolean show_watched = (l & STATE_FILTER_WATCHED) != 0;
	const gboolean show_ignored = (l & STATE_FILTER_IGNORED) != 0;
	const gboolean show_normal = (l & STATE_FILTER_NORMAL_RANK) != 0;
	const gboolean show_saved = (l & STATE_FILTER_SAVED) != 0;
	const gboolean show_idle = (l & STATE_FILTER_IDLE) != 0;
	const gboolean show_queued = (l & STATE_FILTER_QUEUED) != 0;
	const gboolean show_cached = (l & STATE_FILTER_CACHED) != 0;
	const gboolean show_non_cached = (l & STATE_FILTER_NOT_CACHED) != 0;
	const gboolean show_by_me = (l & STATE_FILTER_MINE) != 0;
	const gboolean show_by_others = (l & STATE_FILTER_NOT_MINE) != 0;
	gint n;

	/* top */
	top = FILTER_AGGREGATE(filter_aggregate_new ());
	filter_aggregate_set_type (top, AGGREGATE_TYPE_AND);

	/* new/unread/read */
	n = 0;
	if (show_new) ++n;
	if (show_unread) ++n;
	if (show_read) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_new_article_new (FILTER_NEW_NEW);
		if (n==1) {
			if (show_new)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_NEW);
			else if (show_unread)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_UNREAD);
			else if (show_read)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_READ);
		} else {
			if (!show_new)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_NEW);
			else if (!show_unread)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_UNREAD);
			else if (!show_read)
				filter_new_set_state (FILTER_NEW(filter), FILTER_NEW_READ);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* attachments */
	n = 0;
	if (show_bin) ++n;
	if (show_inc_bin) ++n;
	if (show_nonbinary) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_binary_new ();
		if (n==1) {
			if (show_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_COMPLETE);
			else if (show_inc_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_INCOMPLETE);
			else if (show_nonbinary)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_NONBINARY);
		} else {
			if (!show_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_COMPLETE);
			else if (!show_inc_bin)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_INCOMPLETE);
			else if (!show_nonbinary)
				filter_binary_set_state (FILTER_BINARY(filter), FILTER_BINARY_NONBINARY);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* thread state */
	n = 0;
	if (show_watched) ++n;
	if (show_ignored) ++n;
	if (show_normal) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_thread_new ();
		if (n==1) {
			if (show_watched)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_WATCHED);
			else if (show_ignored)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_IGNORED);
			else if (show_normal)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_NOTHING);
		} else {
			if (!show_watched)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_WATCHED);
			else if (!show_ignored)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_IGNORED);
			else if (!show_normal)
				filter_thread_set_state (FILTER_THREAD(filter), FILTER_THREAD_NOTHING);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* saved/queued/idle */
	n = 0;
	if (show_saved) ++n;
	if (show_queued) ++n;
	if (show_idle) ++n;
	if (n==1 || n==2) {
		Filter * filter = filter_saved_new ();
		if (n==1) {
			if (show_saved)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_SAVED);
			else if (show_queued)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_QUEUED);
			else if (show_idle)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_NOT_SAVED);
		} else {
			if (!show_saved)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_SAVED);
			else if (!show_queued)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_QUEUED);
			else if (!show_idle)
				filter_saved_set_state (FILTER_SAVED(filter), FILTER_SAVED_NOT_SAVED);
			filter_negate (filter);
		}
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* cached */
	n = 0;
	if (show_cached) ++n;
	if (show_non_cached) ++n;
	if (n == 1) {
		Filter * filter = filter_cached_new ();
		if (show_non_cached)
			filter_negate (filter);
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* by me / others */
	n = 0;
	if (show_by_me) ++n;
	if (show_by_others) ++n;

	if (n == 1) {
		Filter * filter = filter_mine_new ();
		if (show_by_others)
			filter_negate (filter);
		filter_aggregate_add (top, &filter, 1);
		pan_object_unref (PAN_OBJECT(filter));
	}

	/* retval */
	if (filter_aggregate_child_size(top) != 1)
		retval = FILTER(top);
	else {
		retval = filter_aggregate_get_child_at (top, 0);
		pan_object_ref (PAN_OBJECT(retval));
		pan_object_unref (PAN_OBJECT(top)); /* remove previous filter */
	}

	/* negate */
	if (negate)
		filter_negate (retval);

	return retval;
}

static GtkWidget*
create_tb (const char * mnemonic, const guint8* inline_image)
{
	GtkWidget * t;
	GtkWidget * l;
	GtkWidget * v;

	t = gtk_toggle_button_new ();
	v = gtk_vbox_new (FALSE, GUI_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(t), v);

	if (inline_image != NULL) {
		GdkPixbuf * p = gdk_pixbuf_new_from_inline (-1, inline_image, FALSE, NULL);
		GtkWidget * i = gtk_image_new_from_pixbuf (p);
		g_object_unref (G_OBJECT(p));
		gtk_box_pack_start (GTK_BOX(v), i, TRUE, TRUE, 0);
	}

	l = gtk_label_new_with_mnemonic (mnemonic);
	gtk_label_set_mnemonic_widget (GTK_LABEL(l), GTK_WIDGET(t));
	gtk_box_pack_start (GTK_BOX(v), l, FALSE, FALSE, 0);

	return t;
}

GtkWidget*
filter_current_dialog_new (GtkWindow * parent,
                           gulong bits,
                           FilterShow show,
                           gboolean negate)
{
	int r = 0;
	int c = 0;
	GtkWidget * w;
	GtkWidget * table;
	GtkWidget * l;
	GtkWidget * f;
	GtkWidget * h;
	GtkWidget * v;
	GtkWidget * rb;
	GtkWidget * hwork;
	GtkWidget * frame;
	DialogImpl * d;
	
	d = g_new0 (DialogImpl, 1);
	d->bits = bits;
	d->show = show;

	/* dialog */
	w = d->dialog = gtk_dialog_new_with_buttons (_("Pan: Header Pane Filter"),
	                                             parent,
	                                             GTK_DIALOG_DESTROY_WITH_PARENT,
	                                             GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                             GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
	                                             GTK_STOCK_OK, GTK_RESPONSE_OK,
	                                             NULL);
	gtk_dialog_set_default_response (GTK_DIALOG(w), GTK_RESPONSE_OK);
	g_object_set_data_full (G_OBJECT(w), "impl", d, g_free);
	gtk_window_set_policy (GTK_WINDOW(w), TRUE, TRUE, TRUE);

	/* workarea */
	hwork = gtk_hbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(hwork), GUI_PAD_BIG);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(w)->vbox), hwork, TRUE, TRUE, 0);

	/**
	***  MATCHES
	**/

	frame = gtk_frame_new (_("Match Articles"));
	gtk_box_pack_start (GTK_BOX(hwork), frame, FALSE, FALSE, GUI_PAD);

	table = gtk_table_new (7, 2, FALSE);
	gtk_container_add (GTK_CONTAINER(frame), table);
	gtk_table_set_col_spacings (GTK_TABLE(table), GUI_PAD_BIG);
	gtk_table_set_row_spacings (GTK_TABLE(table), GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(table), GUI_PAD);


	r = c = 0;
	l = gtk_label_new (_("Match Availability:"));
	gtk_misc_set_alignment (GTK_MISC(l), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), l, c, c+1, r, r+1, GTK_FILL, 0, 0, 0);
	++c;
	f = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(f), GTK_SHADOW_IN);
	gtk_table_attach (GTK_TABLE(table), f, c, c+1, r, r+1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
	h = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER(f), h);
	w = create_tb (_("Have _Local Copy"), icon_disk);
	d->availability_tb_local = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = create_tb (_("Must _Get from Server"), icon_server);
	d->availability_tb_server = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);

	c = 0;
	++r;
	l = gtk_label_new (_("Match Author:"));
	gtk_misc_set_alignment (GTK_MISC(l), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), l, c, c+1, r, r+1, GTK_FILL, 0, 0, 0);
	++c;
	f = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(f), GTK_SHADOW_IN);
	gtk_table_attach (GTK_TABLE(table), f, c, c+1, r, r+1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
	h = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER(f), h);
	w = create_tb (_("Posted by _Me"), icon_by_me);
	d->author_tb_me = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = create_tb (_("by Someone _Else"), icon_by_others);
	d->author_tb_others = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);

	c = 0;
	++r;
	l = gtk_label_new (_("Match Age:"));
	gtk_misc_set_alignment (GTK_MISC(l), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), l, c, c+1, r, r+1, GTK_FILL, 0, 0, 0);
	++c;
	f = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(f), GTK_SHADOW_IN);
	gtk_table_attach (GTK_TABLE(table), f, c, c+1, r, r+1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
	h = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER(f), h);
	w = create_tb (_("_New"), icon_new_unread);
	d->age_tb_new = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = create_tb (_("_Unread"), icon_old_unread);
	d->age_tb_unread = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = create_tb (_("_Read"), icon_old_read);
	d->age_tb_read = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);

	c = 0;
	++r;
	l = gtk_label_new (_("Match Attachments:"));
	gtk_misc_set_alignment (GTK_MISC(l), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), l, c, c+1, r, r+1, GTK_FILL, 0, 0, 0);
	++c;
	f = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(f), GTK_SHADOW_IN);
	gtk_table_attach (GTK_TABLE(table), f, c, c+1, r, r+1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
	h = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER(f), h);
	w = create_tb (_("Complete _Binary"), icon_binary_complete);
	d->attach_tb_complete = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = create_tb (_("Incomple_te Binary"), icon_binary_incomplete);
	d->attach_tb_incomplete = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = create_tb (_("_Just Text"), icon_new_unread);
	d->attach_tb_text = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);

	c = 0;
	++r;
	l = gtk_label_new (_("Match Action:"));
	gtk_misc_set_alignment (GTK_MISC(l), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), l, c, c+1, r, r+1, GTK_FILL, 0, 0, 0);
	++c;
	f = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(f), GTK_SHADOW_IN);
	gtk_table_attach (GTK_TABLE(table), f, c, c+1, r, r+1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
	h = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER(f), h);
	w = create_tb (_("_Saved"), icon_binary);
	d->action_tb_saved = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = create_tb (_("_Queued to Save"), icon_bluecheck);
	d->action_tb_queued = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = create_tb (_("_Idle"), icon_new_unread);
	d->action_tb_idle = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);

	c = 0;
	++r;
	l = gtk_label_new (_("Match Attention:"));
	gtk_misc_set_alignment (GTK_MISC(l), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(table), l, c, c+1, r, r+1, GTK_FILL, 0, 0, 0);
	++c;
	f = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(f), GTK_SHADOW_IN);
	gtk_table_attach (GTK_TABLE(table), f, c, c+1, r, r+1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
	h = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER(f), h);
	w = create_tb (_("_Watched"), icon_watched);
	d->attention_tb_watched = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = create_tb (_("Ignore_d"), icon_bozo);
	d->attention_tb_ignored = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);
	w = create_tb (_("Ordinar_y"), icon_new_unread);
	d->attention_tb_normal = w;
	gtk_box_pack_start (GTK_BOX(h), w, TRUE, TRUE, 0);


	/**
	***  SHOW
	**/

	frame = gtk_frame_new (_("Show"));
	gtk_box_pack_start (GTK_BOX(hwork), frame, FALSE, FALSE, GUI_PAD);
	v = gtk_vbox_new (FALSE, GUI_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(v), GUI_PAD);
	gtk_container_add (GTK_CONTAINER(frame), v);

	w = rb = gtk_radio_button_new_with_label (NULL, _("Matching Articles"));
	d->show_article = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(rb), _("Matching Articles and Replies"));
	d->show_article_and_replies = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(rb), _("Matching Articles and References"));
	d->show_article_and_references = w;
	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(rb), _("Threads with Matching Articles"));
	d->show_thread = w;

	gtk_box_pack_start (GTK_BOX(v), w, FALSE, FALSE, 0);
	w = gtk_check_button_new_with_mnemonic (_("Negate Current _Filter"));
	d->negate = w;
	gtk_box_pack_end (GTK_BOX(v), w, FALSE, FALSE, 0);

	set_filter_bit_buttons (d, d->bits);
	set_filter_show_buttons (d, d->show);

	return d->dialog;
}
