/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* 
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <locale.h>

#include <glib.h>

#include <gmime/gmime-utils.h>

#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>


/***
****
****  STRINGS
****
***/

/* null-safe strstr */
char*
pan_strstr (const char * s1,
            const char * s2)
{
	g_return_val_if_fail (s1!=NULL, NULL);
	g_return_val_if_fail (s2!=NULL, NULL);

	return strstr (s1, s2);
}

/* null-safe strcmp.  NULLs < strings. */
gint 
pan_strcmp (const char * a,
            const char * b)
{
	gint retval;

	if (a == b)
		retval = 0;
	else if (!a)
		retval = -1;
	else if (!b)
		retval = 1;
	else
		retval = strcmp (a, b);

	return retval;
}

char*
pan_stristr (const char * string, const char * pattern)
{
	size_t slen;
	size_t plen;
	char * start = (char*) string;

	g_return_val_if_fail (string!=NULL, NULL);
	g_return_val_if_fail (pattern!=NULL, NULL);

 	slen = strlen (string);
	plen = strlen (pattern);

	for (; slen>=plen; ++start, --slen)
	{
		char * sptr;
		char * pptr;

		/* find start of pattern in string */
		while (toupper(*start) != toupper(*pattern))
		{
			++start;
			--slen;

			/* if pattern longer than string */
			if (slen < plen)
				return NULL;
		}

		sptr = start;
		pptr = (char *) pattern;

		while (toupper(*sptr) == toupper(*pptr))
		{
			++sptr;
			++pptr;

			if (!*pptr)
				return start;
		}
	}

	return NULL;
}

gboolean
string_ends_with (const char * str,
                  const char * end)
{
	size_t str_len;
	size_t end_len;
	gboolean retval = FALSE;

	g_return_val_if_fail (end!=NULL, FALSE);
	g_return_val_if_fail (str!=NULL, FALSE);

	str_len = strlen (str);
	end_len = strlen (end);
	if (end_len <= str_len)
		retval = !memcmp(end, str+str_len-end_len, end_len);
 
	return retval;
}


void
replace_gstr (char   ** target_gfree_old,
              char    * assign_from_me)
{
	char * gfree_old;

 	g_return_if_fail (target_gfree_old != NULL);

	gfree_old = *target_gfree_old;
	*target_gfree_old = assign_from_me;
	g_free (gfree_old);
}

/**
 * Replaces the search string inside original with the replace string.
 * This should be safe with overlapping elements of text, though I haven't
 * not tested it with elaborately cruel cases.
 */
char*
pan_substitute (const char * original,
                const char * search,
                const char * replace)
{
	size_t slen;		/* length of search */
	size_t rlen;		/* length of replace */
	size_t tlen;		/* length of target (predicted) */
	gint i;
	const char * o;
	const char * pchar;
	char * t;
	char * retval = NULL;

	g_return_val_if_fail (original!=NULL, NULL);
	g_return_val_if_fail (*original!='\0', NULL);
	g_return_val_if_fail (search!=NULL, NULL);
	g_return_val_if_fail (*search!='\0', NULL);
	g_return_val_if_fail (replace!=NULL, NULL);

	slen = strlen (search);
	rlen = strlen (replace);

	/* calculate the length */

	i = 0;
	tlen = 0;
	pchar = original;
	while ((pchar = pan_strstr (pchar, search))) {
		i++;
		pchar += slen;
	}
	tlen = strlen(original) + i*(rlen - slen);

	/**
	***  Make the substitution.
	**/

	o = original;
	t = retval = g_malloc(tlen + 1);
	while ((pchar = pan_strstr (o, search))) {
		(void) memcpy (t, o, (size_t)(pchar-o));
		t += pchar-o;
		(void) memcpy (t, replace, (size_t)rlen);
		t += rlen;
		o = pchar + slen;
	}
	(void) strcpy ( t, o );

	pan_warn_if_fail (strlen(retval)==tlen);

	return retval;
}

/***
****
****  GLIB AUGMENTATION
****
***/

void
pan_g_ptr_array_append (GPtrArray   * target,
                        gpointer    * source_ptr,
                        guint         source_qty)
{
	guint old_len;

	g_return_if_fail (target!=NULL);
	g_return_if_fail (source_ptr!=NULL);
	g_return_if_fail (source_qty>0);
       
	old_len = target->len;
	g_ptr_array_set_size (target, old_len+source_qty);
	memcpy (target->pdata+old_len, source_ptr, sizeof(gpointer)*source_qty);
}

void
pan_g_ptr_array_assign  (GPtrArray  * target,
                         gpointer   * source_ptr,
                         guint        source_qty)
{
	g_return_if_fail (target!=NULL);
	g_return_if_fail (source_qty==0 || source_ptr!=NULL);

	g_ptr_array_set_size (target, source_qty);
	memcpy (target->pdata, source_ptr, sizeof(gpointer)*source_qty);
}

GPtrArray*
pan_g_ptr_array_dup     (const GPtrArray  * source)
{
	GPtrArray * retval;

	g_return_val_if_fail (source!=NULL, NULL);

	retval = g_ptr_array_new ();
	pan_g_ptr_array_assign (retval, source->pdata, source->len);
	return retval;
}


static void
pan_hash_to_ptr_array_ghfunc (gpointer key, gpointer val, gpointer data)
{
	g_ptr_array_add ((GPtrArray*)data, val);
}
void
pan_hash_to_ptr_array  (GHashTable   * hash,
                        GPtrArray    * fillme)
{
	g_return_if_fail (fillme!=NULL);
	g_ptr_array_set_size (fillme, 0);

	g_return_if_fail (hash!=NULL);
	pan_g_ptr_array_reserve (fillme, g_hash_table_size(hash));
	g_hash_table_foreach (hash, pan_hash_to_ptr_array_ghfunc, fillme);
}


void
pan_g_ptr_array_reserve (GPtrArray  * a,
                         int          n)
{
	int len;

	g_return_if_fail (a!=NULL);

	len = a->len;
	g_ptr_array_set_size (a, MAX(len, n));
	a->len = len;
}

void
pan_g_ptr_array_insert (GPtrArray   * a,
                        gpointer      ptr,
                        int           index)
{
        g_return_if_fail (a!=NULL);

        if (index<0 || index>=a->len)
	{
                g_ptr_array_add (a, ptr);
	}
        else
        {
		pan_g_ptr_array_reserve (a, a->len+1);
                g_memmove (&a->pdata[index+1],
                           &a->pdata[index],
                           sizeof(gpointer)*(a->len-index));
                a->pdata[index] = ptr;
		++a->len;
        }
}

void
pan_g_ptr_array_foreach (GPtrArray   * a,
                         GFunc         func,
                         gpointer      user_data)
{
	guint i;

	g_return_if_fail (a!=NULL);
	g_return_if_fail (func!=NULL);

	for (i=0; i!=a->len; ++i)
	{
		gpointer call_data = g_ptr_array_index (a, i);
		(*func)(call_data, user_data);
	}
}

void
pan_g_string_replace (GString       * string,
                      const char    * search,
		      const char    * replace)
{
	const char * pch;
	size_t slen;
	size_t rlen;
	size_t offset;

	g_return_if_fail (string != NULL);
	g_return_if_fail (is_nonempty_string(search));
	g_return_if_fail (replace != NULL);

	slen = strlen (search);
	rlen = strlen (replace);

	offset = 0;
	while ((pch = pan_strstr (string->str + offset, search)) && *pch) {
		const gint pos = pch - string->str;
		g_string_erase (string, pos, slen);
		g_string_insert (string, pos, replace);
		offset = pos + rlen;
	}
}

void
pan_g_string_strstrip (GString * string)
{
	g_return_if_fail (string != NULL);

	if (string->len != 0)
	{
		g_strstrip (string->str);
		string->len = strlen (string->str);
	}
}


/***
****
****  TOKENS
****
***/

 
void
skip_next_token (const char   * pch,
                 const char     delimiter,
                 const char  ** setme_next_token)
{
	register char ch;

	if (is_nonempty_string(pch))
	{
		for (;;) {
			ch = *pch;
			if (ch == delimiter) {
				++pch; /* skip past delimiter */
				if (setme_next_token)
					*setme_next_token = pch;
				return;
			}
			if (ch)
			       ++pch;
			else {
				if (setme_next_token)
					*setme_next_token = pch;
				return;
			}
		}
	}
}
 
const char*
get_next_token_run (const char   * pch,
                    const char     delimiter,
                    const char  ** setme_next_token,
                    const char  ** setme_start,
                    int          * setme_len)
{
	const char * start = pch;

	if (pch!=NULL)
		while (*pch && *pch!=delimiter)
			++pch;

	*setme_start = start;
	*setme_len = pch!=NULL ? pch-start : 0;
	if (setme_next_token != NULL)
		*setme_next_token = pch!=NULL && *pch==delimiter ? pch+1 : pch;

	return is_nonempty_string(start) ? start : NULL;
}


glong
get_next_token_long (const char   * pch,
                     const char     delimiter,
                     const char  ** setme_next_token)
{
	glong retval = 0;

	if (is_nonempty_string(pch))
	{
		const char * m;
		const char * end = NULL;

		/* find the end of this token */
		skip_next_token (pch, delimiter, &end);
		if (setme_next_token != NULL)
			*setme_next_token = end;

		/* only extract an int of there's a digit in this token.
		   otherwise if isspace(token) is true, atoi may walk
		   right into the next token and give a wrong retval */
		for (m=pch; m!=end; ++m) {
			if (*m=='-' || *m=='+' || isdigit((guchar)*m)) {
				retval = atol (m);
				break;
			}
		}
	}

	return retval;
}

gulong
get_next_token_ulong (const char   * pch,
                      const char     delimiter,
                      const char  ** setme_next_token)
{
	gulong retval = 0;

	if (is_nonempty_string(pch))
	{
		const char * m;
		const char * end = NULL;

		/* find the end of this token */
		skip_next_token (pch, delimiter, &end);
		if (setme_next_token != NULL)
			*setme_next_token = end;

		/* only extract an int of there's a digit in this token.
		   otherwise if isspace(token) is true, strtoul may walk
		   right into the next token and give a wrong retval */
		for (m=pch; m!=end; ++m) {
			if (*m=='-' || *m=='+' || isdigit((guchar)*m)) {
				retval = strtoul (m, NULL, 10);
				break;
			}
		}
	}

	return retval;
}

int
get_next_token_int (const char   * pch,
                    const char     delimiter,
                    const char  ** setme_next_token)
{
	int retval = 0;

	if (is_nonempty_string(pch))
	{
		const char * m;
		const char * end = NULL;

		/* find the end of this token */
		skip_next_token (pch, delimiter, &end);
		if (setme_next_token != NULL)
			*setme_next_token = end;

		/* only extract an int of there's a digit in this token.
		   otherwise if isspace(token) is true, atoi may walk
		   right into the next token and give a wrong retval */
		for (m=pch; m!=end; ++m) {
			if (*m=='-' || *m=='+' || isdigit((guchar)*m)) {
				retval = atoi(m);
				break;
			}
		}
	}

	return retval;
}

char*
get_next_token_str (const char   * pch,
                    const char     delimiter,
                    const char  ** setme_next_token)
{
	char* retval = NULL;
	register const char* end = NULL;

	if (is_nonempty_string(pch)) {
		end = pch;
		while (is_nonempty_string(end) && *end!=delimiter)
			++end;
	}

	if (end != NULL)
		retval = g_strndup (pch, end-pch);

	if (setme_next_token)
		*setme_next_token = is_nonempty_string(end) ? end+1 : end;

	return retval;
}

gboolean
get_next_token_g_str (const char   * pch,
                      const char     delimiter,
                      const char  ** setme_next_token,
                      GString      * setme)
{
	register const char* end = pch;
	gboolean retval;

	while (is_nonempty_string(end) && *end!=delimiter)
		++end;

	g_string_truncate (setme, 0);

	if (end == pch)
		retval = end==NULL ? FALSE : *end==delimiter;
	else {
		g_string_append_len (setme, pch, end-pch);
		retval = TRUE;
	}

	if (setme_next_token)
		*setme_next_token = is_nonempty_string(end) ? end+1 : end;

	return retval;
}

/***
****
****  UNSORTED
****
***/


int
lower_bound (const void         * key,
             const void         * base,
             size_t               n,
             size_t               size,
             int                  (*compare)(const void *, const void *),
             gboolean           * exact_match )
{
	register int low = 0;
	register int high = n - 1;

	while (low<=high)
	{
		const int mid = (unsigned)(low+high)/2u;
		const void * checkme = (void*)(((char*)(base))+(mid*size));
		const int comp = (*compare)(key,checkme);

		if (comp>0) low = mid+1;
		else if (comp<0 ) high = mid-1;
		else {
			if ( exact_match!=NULL )
				*exact_match = TRUE;
			return mid;
		}
	}

	if (exact_match!=NULL)
		*exact_match = FALSE;

	return low;
}


void
commatize_ulong (gulong    num,
		 char    * setme)
{
	char buf[32], *src=buf;
	int len;
	int i;

	g_snprintf (buf, sizeof(buf), "%ld", num);
	len = strlen (buf);
	i = len % 3;
	for (;;)
	{
		while (*src && i--)
			*setme++ = *src++;
		if (!*src) {
			*setme = '\0';
			return;
		}
		else if (src!=buf)
		   *setme++ = ',';
		i = 3;
	}

	pan_warn_if_reached ();
}


/***
****
****  XML
****
***/

char*
pan_str_escape (const char * str)
{
	gint size_needed;
	char * retval;
	char * out;
	const char * in;

	/* sanity clause */
	if (!is_nonempty_string (str))
		return g_strdup ("");

	/* how big must the buffer be? */
	size_needed = 1;
	for (in=str; *in; ++in) {
		gint inc;
		switch (*in) {
			case '&': inc = 5; break;
			case '>': case '<': inc = 4; break;
			case '"': inc = 6; break;
			default: inc = 1; break;
		}
		size_needed += inc;
	}

	/* build the output string */
	retval = out = g_malloc (size_needed);
	for (in=str; *in; ++in) {
		switch (*in) {
			case '&': memcpy (out, "&amp;", 5); out += 5; break;
			case '>': memcpy (out, "&gt;", 4); out += 4; break;
			case '<': memcpy (out, "&lt;", 4); out += 4; break;
			case '"': memcpy (out, "&quot;", 6); out += 6; break;
			default: *out++ = *in; break;
		}
	}
	*out = '\0';

	return retval;
}

char*
pan_str_unescape (const char * escaped)
{
	const char * src;
	char * buf;
	char * tgt;

	g_return_val_if_fail (escaped!=NULL, NULL);

	src = escaped;
	tgt = buf = g_new0 (char, strlen(src)+1);
	while (*src)
	{
		if (*src!='&') { *tgt++ = *src++; }
		else if (!strncmp(src,"&#",2)) {
			char * endp = NULL;
			long ch;
			src += 2;
			if ((ch = strtoul(src, &endp, 0))) {
				*tgt++ = ch;
				src = endp + 1;
			}
		}
		else if (!strncmp(src,"&lt;",4)) { *tgt++ = '<';  src+=4; }
		else if (!strncmp(src,"&gt;",4))   { *tgt++ = '>';  src+=4; }
		else if (!strncmp(src,"&amp;",5))  { *tgt++ = '&';  src+=5; }
		else if (!strncmp(src,"&apos;",6)) { *tgt++ = '\''; src+=6; }
		else if (!strncmp(src,"&quot;",6)) { *tgt++ = '"';  src+=6; }
		else *tgt++ = *src++;
	}

	*tgt = '\0';

	return buf;
}

/***
****
***/

struct tm*
pan_localtime_r(const time_t *const timep, struct tm *p_tm)
{
	struct tm * retval;
#ifdef HAVE_LOCALTIME_R
	retval = localtime_r (timep, p_tm);
#else
	static GStaticMutex mutex = G_STATIC_MUTEX_INIT;

	g_static_mutex_lock (&mutex);
	retval = localtime (timep);
	if (retval != NULL) {
		memcpy (p_tm, retval, sizeof(struct tm));
		retval = p_tm;
	}
	g_static_mutex_unlock (&mutex);
#endif
	return retval;
}

struct tm*
pan_gmtime_r (const time_t * const timep, struct tm * p_tm)
{
	struct tm * retval;
#ifdef HAVE_GMTIME_R
	retval = gmtime_r (timep, p_tm);
#else
	static GStaticMutex mutex = G_STATIC_MUTEX_INIT;

	g_static_mutex_lock (&mutex);
	retval = gmtime (timep);
	if (retval != NULL) {
		memcpy (p_tm, retval, sizeof(struct tm));
		retval = p_tm;
	}
	g_static_mutex_unlock (&mutex);
#endif
	return retval;
}

/**
 ** Charsets
 **/

/* The language isn't used yet, but will be useful for get the default
 * language when we integrate a spellchecker.
 */

struct LocaleStruct {
    const char *locale, *charset, *lang_name;
} locales[] = {
    {"en_US",        "ISO-8859-1",       N_("English")}, 
    {"pt_BR",        "ISO-8859-1",       N_("Brazilian")},
    {"ca_ES",        "ISO-8859-15",      N_("Catalan")},
    {"zh_CN.GB2312", "gb2312",           N_("Chinese Simplified")},
    {"zh_TW.Big5",   "big5",             N_("Chinese Traditional")},
    {"cs_CZ",        "ISO-8859-2",       N_("Czech")},
    {"da_DK",        "ISO-8859-1",       N_("Danish")},
    {"de_DE",        "ISO-8859-15",      N_("German")},
    {"nl_NL",        "ISO-8859-15",      N_("Dutch")},
    {"et_EE",        "ISO-8859-15",      N_("Estonian")},
    {"fi_FI",        "ISO-8859-15",      N_("Finnish")},
    {"fr_FR",        "ISO-8859-15",      N_("French")},
    {"el_GR",        "ISO-8859-7",       N_("Greek")},
    {"hu_HU",        "ISO-8859-2",       N_("Hungarian")},
    {"it_IT",        "ISO-8859-15",      N_("Italian")},
    {"ja_JP",        "ISO-2022-jp",      N_("Japanese")},
    {"ko_KR",        "euc-kr",           N_("Korean")},
    {"lv_LV",        "ISO-8859-13",      N_("Latvian")},
    {"lt_LT",        "ISO-8859-13",      N_("Lithuanian")},
    {"no_NO",        "ISO-8859-1",       N_("Norwegian")},
    {"pl_PL",        "ISO-8859-2",       N_("Polish")},
    {"pt_PT",        "ISO-8859-15",      N_("Portuguese")},
    {"ro_RO",        "ISO-8859-2",       N_("Romanian")},
    {"ru_RU",        "KOI8-R",           N_("Russian (KOI)")},
    {"ru_SU",        "ISO-8859-5",       N_("Russian (ISO)")},
    {"sk_SK",        "ISO-8859-2",       N_("Slovak")},
    {"es_ES",        "ISO-8859-15",      N_("Spanish")},
    {"sv_SE",        "ISO-8859-1",       N_("Swedish")},
    {"tr_TR",        "ISO-8859-9",       N_("Turkish")},
    {"uk_UK",        "KOI8-U",           N_("Ukrainian")}
};

/* find_locale_index_by_locale:
 * finds the longest fit so the one who has en_GB will get en_US if en_GB
 * is not defined.
 * This function is lifted from Balsa.
 */
static gint
get_closest_locale (void)
{
	const char * locale = setlocale (LC_CTYPE, NULL);
	guint i, j, maxfit = 0, maxpos = 0;

	g_return_val_if_fail (locale != NULL, -1);

	if (!locale || strcmp(locale, "C") == 0)
		return 0;

	for (i = 0; i < G_N_ELEMENTS(locales); i++) {
		for (j=0; locale[j] && locales[i].locale[j] == locale[j]; j++);
		if (j > maxfit) {
			maxfit = j;
			maxpos = i;
		}
	}
	return maxpos;
}

#define PAN_DEFAULT_CHARSET  "ISO-8859-1"
#define PAN_DEFAULT_LANG     "English"

const char *
get_charset_from_locale (void)
{
	gint loc_idx;

	loc_idx = get_closest_locale ();

	return loc_idx != -1 ? locales[loc_idx].charset : PAN_DEFAULT_CHARSET;
}

const char *
get_lang_from_locale (void)
{
	gint loc_idx;

	loc_idx = get_closest_locale ();

	return loc_idx != -1 ? locales[loc_idx].lang_name : PAN_DEFAULT_LANG;
}

const char*
pan_utf8ize (const char    * str,
             gssize          len,
             char         ** g_freeme)
{
	return pan_g_convert_to_utf8 (str, g_freeme, len, NULL, NULL, NULL);
}

const char*
pan_g_convert_to_utf8 (const char   * str,
                       char        ** g_freeme,
                       gssize         len,
                       gsize        * bytes_read,
                       gsize        * bytes_written,
                       GError      ** error)
{
	const char * retval;

	g_return_val_if_fail (g_freeme!=NULL, NULL);

	if (!is_nonempty_string (str))
		retval = *g_freeme = g_strdup ("");
	else if  (!g_utf8_validate (str, len, NULL))
		retval = *g_freeme = g_locale_to_utf8 (str, len, bytes_read, bytes_written, error);
	else {
		*g_freeme = NULL;
		retval = str;
	}

	return retval;
}

gboolean
pan_header_is_8bit_encoded (const char * pch)
{
	return pch!=NULL && strstr(pch,"=?")!=NULL;
}


char *
pan_header_to_utf8 (const gchar * str, size_t len, const char * charset)
{
	char   * retval = NULL;
	char   * pch;

	if (str == NULL)
		return NULL;

	if (len == -1)
		len = strlen (str);
	pch = g_strndup (str, len);

	if (pan_header_is_8bit_encoded (pch))
		retval = g_mime_utils_8bit_header_decode ((const guchar *) pch);
	else
	/* naughty boys don't properly encode their 8bit string */
	if (g_mime_utils_text_is_8bit (pch, len))
	{
		if (charset == NULL)
			retval = g_locale_to_utf8 (pch, len, NULL, NULL, NULL);
		else
			retval = g_convert (pch, len, "UTF-8", charset, NULL,NULL,NULL);
	}

	/* fallback */
	if (retval == NULL)
		retval = g_strdup (pch);

	g_free (pch);

	return retval;
}

