/*
 * Copyright (c) 2007 Tamas Tevesz <ice@extreme.hu>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <err.h>
#include <bsd_auth.h>
#include <grp.h>
#include <pwd.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <utmp.h>		/* UT_NAMESIZE */

#ifndef	VPN_GROUP
#define	VPN_GROUP		"_openvpnusers"
#endif

/*
 * Sync this with OpenVPN's misc.h:USER_PASS_LEN
 *
 * It looks like the definition of (at the time of writing this, with
 * OpenVPN 2.0.9 and 2.1rc4 being current) `128' already accounts for
 * the terminating NUL, which is emphasized here as being `127 + 1'.
 */
#define	OPENVPN_USER_PASS_MAXLEN	(127 + 1)

#define	OPENVPN_AUTH_OK			0
#define	OPENVPN_AUTH_FAIL		1

extern char *__progname;
int is_ingroup(char *);
__dead void logexit(int, int, const char *, ...);

int main(int argc, char **argv) {
	char username[UT_NAMESIZE+1];
	char password[OPENVPN_USER_PASS_MAXLEN];
	char *tmp;

	(void)memset(username, 0, sizeof(username));
	(void)memset(password, 0, sizeof(password));
	openlog(__progname, LOG_PID|LOG_NDELAY, LOG_AUTH);

	if (argc == 2) {
		FILE *fp;
		if ((fp = fopen(argv[1], "r")) == NULL)
			logexit(OPENVPN_AUTH_FAIL, LOG_ERR,
			    "Can not open password file \"%s\"", argv[1]);

		if ((tmp = fgets(username, sizeof(username), fp)) == NULL)
			logexit(OPENVPN_AUTH_FAIL, LOG_ERR,
			    "Unable to read username from \"%s\"", argv[1]);

		if ((tmp = fgets(password, sizeof(password), fp)) == NULL)
			logexit(OPENVPN_AUTH_FAIL, LOG_ERR,
			    "Unable to read password from \"%s\"", argv[1]);

		(void)fclose(fp);
		username[strcspn(username, "\r\n")] = '\0';
		password[strcspn(password, "\r\n")] = '\0';
	} else {
		if ((tmp = getenv("username")) == NULL)
			logexit(OPENVPN_AUTH_FAIL, LOG_ERR,
			    "No \"username\" found in the environment");

		(void)strlcpy(username, tmp, sizeof(username));

		if ((tmp = getenv("password")) == NULL)
			logexit(OPENVPN_AUTH_FAIL, LOG_ERR,
			    "No \"password\" found in the environment");

		(void)strlcpy(password, tmp, sizeof(password));
	}

	if (!is_ingroup(username))
		logexit(OPENVPN_AUTH_FAIL, LOG_ERR,
		    "User %s is not a member of " VPN_GROUP , username);

	if (auth_userokay(username, NULL, NULL, password) != 0)
		logexit(OPENVPN_AUTH_OK, LOG_INFO,
		    "Accepted password for %s", username);

	logexit(OPENVPN_AUTH_FAIL, LOG_ERR, "Bad password for %s", username);
}

/* butchered from src/usr.bin/id/id.c */
int is_ingroup(char *username) {
	gid_t groups[NGROUPS+1], vpngroup_gid;
	struct group *gr;
	struct passwd *pw;
	int ngroups, i;
	
	if ((gr = getgrnam(VPN_GROUP)) == NULL)
		logexit(OPENVPN_AUTH_FAIL, LOG_ERR,
		    "Can not find group " VPN_GROUP " in the system group database");
	vpngroup_gid = gr->gr_gid;

	if ((pw = getpwnam(username)) == NULL)
		logexit(OPENVPN_AUTH_FAIL, LOG_ERR,
		    "Bad user %s", username);

	ngroups = NGROUPS+1;
	(void)getgrouplist(pw->pw_name, pw->pw_gid, groups, &ngroups);

	for (i=0; i<ngroups; i++)
		if (vpngroup_gid == groups[i])
			return(1);

	return(0);
}

__dead
void logexit(int exitval, int prio, const char *fmt, ...) {
	va_list va;

	va_start(va, fmt);
	vsyslog(prio, fmt, va);
	va_end(va);
	closelog();

	exit(exitval);
}

