/* $Id: slaveclienttalk.c,v 1.3 2007-04-03 08:19:24 hngott Exp $*/

/*
 * Copyright (C) 2005-2007 Thomas Birnthaler
 *                         Hermann Gottschalk
 *                         <openpoppassd@ostc.de>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Steve Dorner's description of the simple protocol:
 *
 * The server's responses should be like an FTP server's responses;
 * 1xx for in progress, 2xx for success, 3xx for more information
 * needed, 4xx for temporary failure, and 5xx for permanent failure.
 * Putting it all together, here's a sample conversation:
 *
 *   S: 200 poppassd vX.Y hello, who are you?\r\n
 *   E: user YOURLOGINNAME\r\n
 *   S: 300 your password please.\r\n
 *   E: pass YOURCURRENTPASSWORD\r\n
 *   S: 200 your new password please.\r\n
 *   E: newpass YOURNEWPASSWORD\r\n
 *   S: 200 Password changed, thank-you.\r\n
 *   E: quit\r\n
 *   S: 200 Bye.\r\n
 *   S: <closes connection>
 *   E: <closes connection>
 */

#include <sys/param.h>
#include <errno.h>
#include <poll.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <stdio.h>
#include <stdarg.h>
#include <syslog.h>

#include "poppassd.h"

/* Prototypes */
static void WriteToClient(int fd, char *fmt, ...);
static void ReadFromClient(int fd, char* line);

void
client_talk(int slave_socket, int talk_socket)
{
	char line[BUFSIZE];
	char user[BUFSIZE];
	char oldpass[BUFSIZE];
	char newpass[BUFSIZE];
	char errmsg[BUFSIZE];
	int  res;

	log_debug("slave: client_talk");

	*user = *oldpass = *newpass = '\0';

	WriteToClient(talk_socket, "200 openpoppassd v%s hello, who are you?", VERSION);
	ReadFromClient(talk_socket, line);
	// Syntax ok
	if (strncmp("user ", line, 5))
	{
		WriteToClient(talk_socket, "500 Dialog syntax wrong (use 'user XXX').");
		return;
	}
	// Username angegeben
	if (strlen(line) <= 5)
	{
		WriteToClient(talk_socket, "500 Username required.");
		return;
	}
	strncpy(user, line + 5, BUFSIZE - 5);

	WriteToClient(talk_socket, "200 your password please.");
	ReadFromClient(talk_socket, line);
	if (strncmp("pass ", line, 5))
	{
		WriteToClient(talk_socket, "500 Dialog syntax wrong (use 'pass XXX').");
		return;
	}
	if (strlen(line) <= 5)
	{
		WriteToClient(talk_socket, "500 Password required.");
		return;
	}
	strncpy(oldpass, line + 5, BUFSIZE - 5);

	WriteToClient(talk_socket, "200 your new password please.");
	ReadFromClient(talk_socket, line);
	if (strncmp("newpass ", line, 8))
	{
		WriteToClient(talk_socket, "500 Dialog syntax wrong (use 'newpass XXX').");
		return;
	}
	if (strlen(line) <= 8)
	{
		WriteToClient(talk_socket, "500 New password required.");
		return;
	}
	strncpy(newpass, line + 8, BUFSIZE - 8);

	res = ask_change_pw(slave_socket, user, oldpass, newpass, errmsg);
	switch (res)
	{
		case MSG_BUF_OVERFLOW:
		case MSG_AUTH_WRONG:
			WriteToClient(talk_socket, "500 Authentication wrong.");
			break;
		case MSG_PWD_TOO_SIMPLE:
			WriteToClient(talk_socket, "500 Password too simple.");
			break;
		case MSG_CHG_OK:
			WriteToClient(talk_socket, "200 Password changed, thank-you.");
			break;
		case MSG_PASSWD_ERROR:
			WriteToClient(talk_socket, "500 passwd error: %s", errmsg);
			break;
		default:
			break;
	}

	ReadFromClient(talk_socket, line);
	if (strncmp(line, "quit", 4) != 0)
	{
		WriteToClient(talk_socket, "500 Quit required.");
		return;
	}

	WriteToClient(talk_socket, "200 Bye.");
}

static void
WriteToClient(int fd, char *fmt, ...)
{
	va_list ap;
	int     len;
	char    buffer[BUFSIZE];

	va_start(ap, fmt);
	len = vsnprintf(buffer, BUFSIZE, fmt, ap);
	write(fd, buffer, len + 1);
	write(fd, "\r\n", 2);
	va_end(ap);
}

static void
ReadFromClient(int fd, char* line)
{
	char *sp;
	int  len;

	len = read(fd, line, BUFSIZE);
	if ((sp = strchr(line, '\n')) != NULL) *sp = '\0';
	if ((sp = strchr(line, '\r')) != NULL) *sp = '\0';
	line[BUFSIZE - 1] = '\0'; // always terminate with NUL byte

	/* convert initial keyword on line to lower case. */
	for (sp = line; isalpha(*sp); ++sp)
	{
		*sp = tolower(*sp);
	}
}
