/*	$Id: change_passwd.c,v 1.3 2007-04-03 08:19:24 hngott Exp $ */

/*
 * Copyright (C) 2005-2007 Thomas Birnthaler
 *                         Hermann Gottschalk
 *                         <openpoppassd@ostc.de>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

 * Code partly taken from
 * poppassd.c
 * 
 * A Eudora and NUPOP change password server.
 * 
 * John Norstad
 * Academic Computing and Network Services
 * Northwestern University
 * j-norstad@nwu.edu
 * 
*/


#include <sys/param.h>   // ???
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <fcntl.h>
#include <syslog.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <strings.h>
#include <errno.h>
#include <stdarg.h>
#include <pwd.h>
#include <string.h>
#include <termios.h>
#include <dirent.h>

#include "poppassd.h"

#define SUCCESS 1
#define FAILURE 0

#define OPTIONAL  0
#define NECESSARY 1

#define NOMATCH      0
#define PARTIALMATCH 1
#define FULLMATCH    2

/* Prompt strings expected from the "passwd" command. If you want
 * to port this program to yet another flavor of UNIX, you may need to add
 * more prompt strings here.
 *
 * Each prompt is defined as an array of pointers to alternate
 * strings, terminated by NULL. In the strings, '*'
 * matches any sequence of 0 or more characters. Pattern matching
 * is case-insensitive.
 */
static char *P1[] =
{
	"Old password:",
	"Changing password for *.\nOld password:",
	"Changing local password for *.\nOld password:",
	"Changing password for * on *.\nOld password:",
	"Changing NIS password for * on *.\nOld Password: ",
	"Changing password for *\n*'s Old password:",
	NULL
};

static char *P2[] =
{
	"\nNew password:",
	"New password:",
	"\n*'s New password:",
	"*'s New password:",
	NULL
};

static char *P3[] =
{
	"\nRe-enter new password:",
	"Re-enter new password:",
	"\nRetype new password:",
	"Retype new password:",
	"\nEnter the new password again:",
	"Enter the new password again:",
	"\n*Re-enter *'s new password:",
	"*Re-enter *'s new password:",
	"\nVerify:",
	"Verify:",
	NULL
};

static char *P4[] =
{
	"\n",
	"\npasswd: rebuilding the database...\npasswd: done\n",
	"NIS entry changed on *\n",
	"\n\nNIS password has been changed on *.\n",
	NULL
};

/* Prototypes */
void dochild     (int master, char* slavedev, char* user);
void geterrmsg   (int master, char** expected, char* buf);
int  chkPass     (char* user, char* pass, struct passwd* pw);
int  findpty     (char** slave);
int  talktochild (int master, char* user, char* oldpass, char* newpass, char* errmsg);
void writestring (int fd, char* s);
int  expect      (int master, char** expected, char* buf, char talk);
int  match       (char* str, char* pat);

enum err_type
change_passwd(char* user, char* oldpass, char* newpass, char* errmsg)
{
	int            master;
	char*          slavedev;
	pid_t          pid;
	pid_t          wpid;
	struct passwd* pw;
	int            wstat;

	if ((pw = getpwnam(user)) == NULL)
		return(ERR_UNKNOWN_USER);

	/* get pty to talk to password program */
	if ((master = findpty(&slavedev)) < 0)
		return(ERR_NO_PTY_FOUND);

	/* fork child process to talk to password program */
	if ((pid = fork()) < 0)     /* Error, can't fork */
		return(ERR_CANT_PASSWD_FORK);

	if (pid)   /* Parent */
	{
		sleep(1);    /* Make sure child is ready.  Is this really needed? */
		if (talktochild(master, user, oldpass, newpass, errmsg) == FAILURE)
			return(ERR_FAILED_ATTEMPT);

		if ((wpid = waitpid(pid, &wstat, 0)) < 0)
			return(ERR_WAIT_FAILED);

		if (pid != wpid)
			return(ERR_WRONG_CHILD);

		if (WIFEXITED(wstat) == 0)
			return(ERR_KILLED_CHILD);

		if (WEXITSTATUS(wstat) != 0)
			return(ERR_CHILD_EXITED);

		return(0);
	}
	else /* Child */
	{
		/* Start new session - gets rid of controlling terminal. */
		if (setsid() < 0)
		{
			log_warnx("setsid failed");
			return(0);
		}

		/* Set login name */
		if (setlogin(user) < 0)
		{
			log_warnx("setlogin failed");
			return(0);
		}
		setuid(pw->pw_uid);
		setgid(pw->pw_gid);
		dochild(master, slavedev, user);
	}

	return 0;
}

/*
 * Do child stuff - set up slave pty and execl /usr/bin/passwd.
 *
 * Code adapted from "Advanced Programming in the UNIX Environment"
 * by W. Richard Stevens.
 *
 */
void
dochild(int master, char* slavedev, char* user)
{
	int slave;
	struct termios stermios;

	/* Open slave pty and acquire as new controlling terminal. */
	if ((slave = open(slavedev, O_RDWR)) < 0)
	{
		syslog(LOG_ERR, "can't open slave pty: %m");
		return;
	}

	close(master);

	/* Make slave stdin/out/err of child. */
	if (dup2(slave, STDIN_FILENO) != STDIN_FILENO)
	{
		syslog(LOG_ERR, "dup2 error to stdin: %m");
		return;
	}
	if (dup2(slave, STDOUT_FILENO) != STDOUT_FILENO)
	{
		syslog(LOG_ERR, "dup2 error to stdout: %m");
		return;
	}
	if (dup2(slave, STDERR_FILENO) != STDERR_FILENO)
	{
		syslog(LOG_ERR, "dup2 error to stderr: %m");
		return;
	}
	if (slave > 2) close(slave);

	/* Set proper terminal attributes - no echo, canonical input processing,
		no map NL to CR/NL on output. */
	if (tcgetattr(0, &stermios) < 0)
	{
		syslog(LOG_ERR, "tcgetattr error: %m");
		return;
	}
	stermios.c_lflag &= ~(ECHO | ECHOE | ECHOK | ECHONL);
	stermios.c_lflag |= ICANON;
	stermios.c_oflag &= ~(ONLCR);
	if (tcsetattr(0, TCSANOW, &stermios) < 0)
	{
		syslog(LOG_ERR, "tcsetattr error: %m");
		return;
	}

	/* Fork /usr/bin/passwd. */
	if (execl("/usr/bin/passwd", "passwd", user, (char*)NULL) < 0)
	{
		syslog(LOG_ERR, "can't exec /usr/bin/passwd: %m");
		return;
	}
}

/*
 * Finds the first available pseudo-terminal master/slave pair.  The master
 * side is opened and a fd returned as the function value.  A pointer to the
 * name of the slave side (i.e. "/dev/ttyp0") is returned in the argument,
 * which should be a char**.  The name itself is stored in a static buffer.
 *
 * A negative value is returned on any sort of error.
 *
 * Modified by Norstad to remove assumptions about number of pty's allocated
 * on this UNIX box.
 *
 * Modified by Stephen Melvin to allocate local space for static character
 * array, rather than local space to pointer to constant string, which is
 * not kosher and was crashing FreeBSD 1.1.5.1.
 */
int
findpty(char** slave)
{
	int master;
	static char line[11];
	DIR *dirp;
	struct dirent *dp;

	strcpy(line,"/dev/ptyXX");
	dirp = opendir("/dev");
	while ((dp = readdir(dirp)) != NULL)
	{
		if (strncmp(dp->d_name, "pty", 3) == 0 && strlen(dp->d_name) == 5)
		{
			line[8] = dp->d_name[3];
			line[9] = dp->d_name[4];
			if ((master = open(line, O_RDWR)) >= 0)
			{
				line[5] = 't';
				*slave = line;
				closedir(dirp);
				return (master);
			}
		}
	}
	closedir(dirp);
	return (-1);
}

/*
 * Write a string in a single write() system call.
 */
void
writestring(int fd, char* s)
{
	int l;

	l = strlen(s);
	write(fd, s, l);
}

/*
 * Handles the conversation between the parent and child (password program)
 * processes.
 *
 * Returns SUCCESS is the conversation is completed without any problems,
 * FAILURE if any errors are encountered (in which case, it can be assumed
 * that the password wasn't changed).
 */
int
talktochild(int master, char* user, char* oldpass, char* newpass, char* errmsg)
{
	char buf[BUFSIZE];
	char pswd[BUFSIZE+1];

	*errmsg = '\0';

	if (!expect(master, P1, buf, NECESSARY))
		return FAILURE;

	snprintf(pswd, BUFSIZE, "%s\n", oldpass);
	writestring(master, pswd);

	if (!expect(master, P2, buf, NECESSARY))
		return FAILURE;

	snprintf(pswd, BUFSIZE, "%s\n", newpass);
	writestring(master, pswd);

	if (!expect(master, P3, buf, NECESSARY))
	{
		geterrmsg(master, P2, buf);
		strcpy(errmsg, buf);
		return FAILURE;
	}

	writestring(master, pswd);
	sleep(2);
	if (!expect(master, P4, buf, OPTIONAL))
		return FAILURE;

	close(master);

	return SUCCESS;
}

/*
 * Matches a string against a pattern. Wild-card characters '*' in
 * the pattern match any sequence of 0 or more characters in the string.
 * The match is case-insensitive.
 *
 * Entry: str = string.
 *        pat = pattern.
 *
 * Exit:  function result =
 *        NOMATCH      if no match.
 *        PARTIALMATCH if the string matches some initial segment of the pattern.
 *        FULLMATCH    if the string matches the full pattern.
 */
int
match(char* str, char* pat)
{
	int result;
	for (; *str && *pat && *pat != '*'; str++, pat++)
	{
		if (tolower(*str) != tolower(*pat))
		{
			return NOMATCH;
		}
	}
	if (*str == '\0')
	{
		return (*pat == '\0') ? FULLMATCH : PARTIALMATCH;
	}
	else if (*pat == '\0')
	{
		return NOMATCH;
	}
	for (; *str; str++)
	{
		if ((result = match(str, pat+1)) != NOMATCH)
		{
			return result;
		}
	}
	return NOMATCH;
}

/*
 * Reads 'passwd' command output and compares it to expected output.
 *
 * Entry: master   = fid of master pty.
 *        expected = pointer to array of pointers to alternate expected
 *                   strings, terminated by an empty string.
 *        buf      = pointer to buffer.
 *        talk     = NECESSARY/OPTIONAL string, which comes from the master
 *
 * Exit:  function result = SUCCESS if output matched, FAILURE if not.
 *        buf = the text read from the slave.
 *
 * Text is read from the slave and accumulated in buf. As long as
 * the text accumulated so far is an initial segment of at least
 * one of the expected strings, the function continues the read.
 * As soon as one of full expected strings has been read, the
 * function returns SUCCESS. As soon as the text accumulated so far
 * is not an initial segment of or exact match for at least one of
 * the expected strings, the function returns FAILURE.
 */
int
expect(int master, char** expected, char* buf, char talk)
{
	int n, m;
	char **s;
	int front_part;
	int result;

	n = 0;
	buf[0] = '\0';
	while (1)
	{
		if (n >= BUFSIZE-1)
		{
			syslog(LOG_ERR, "buffer overflow on read from child");
			return FAILURE;
		}
		m = read(master, buf+n, BUFSIZE-1-n);
		if (m == 0)
		{
			return (talk == OPTIONAL) ? SUCCESS : FAILURE;
		}
		else if (m < 0)
		{
			syslog(LOG_ERR, "read error from child: %m");
			return FAILURE;
		}
		n += m;
		buf[n] = '\0';
		front_part = NOMATCH;
		for (s = expected; *s != NULL; s++)
		{
			result = match(buf, *s);
			if (result == FULLMATCH)
			{
				return SUCCESS;
			}
			else if (result == PARTIALMATCH)
			{
				front_part = PARTIALMATCH;
			}
		}
		if (!front_part)
		{
			return FAILURE;
		}
	}
}

/*
 * This function accumulates a 'passwd' command error message issued
 * after the first copy of the password has been sent.
 *
 * Entry: master   = fid of master pty.
 *        expected = pointer to array of pointers to alternate expected
 *                   strings for first password prompt, terminated by an
 *                   empty string.
 *        buf      = pointer to buffer containing text read so far.
 *
 * Exit:  buf      = the error message read from the slave.
 *
 * Text is read from the slave and accumulated in buf until the text
 * at the end of the buffer is an exact match for one of the expected
 * prompt strings. The expected prompt string is removed from the buffer,
 * returning just the error message text. Newlines in the error message
 * text are replaced by spaces.
 */
void
geterrmsg(int master, char** expected, char* buf)
{
	int n, m;
	char **s;
	char *p, *q;

	n = strlen(buf);
	while (1)
	{
		for (s = expected; *s != NULL; s++)
		{
			/* Ignore possible leading rubbish */
			for (p = buf; *p; p++)
			{
				if (match(p, *s) == FULLMATCH)
				{
					*p = 0;
					for (q = buf; *q; q++)
					{
						if (*q == '\n')
						{
							*q = ' ';
						}
					}
					return;
				}
			}
		}
		if (n >= BUFSIZE-1)
		{
			syslog(LOG_ERR, "buffer overflow on read from child");
			return;
		}
		m = read(master, buf+n, BUFSIZE-1-n);
		if (m < 0)
		{
			syslog(LOG_ERR, "read error from child: %m");
			return;
		}
		else if (m == 0)
		{
			return;
		}
		n += m;
		buf[n] = 0;
	}
}
