(begin
;;;;;;;;;;;;;;;; 4.2 DERIVED EXPRESSION TYPES
;; Defined in order needed to define the next ones, not in R4RS order.
;;; (define a b) => (set! a b); 
;;; (define (p x) y z) => (set! p (set-procedure-name! (lambda (x) y z) 'p))
(set! define
  (set-procedure-name!
   (macro (var . body)
    (if (pair? var)
        (list 'set! (first var)
              (list 'set-procedure-name!
                    (cons 'lambda (cons (rest var) body))
		    (list 'quote (first var))))
        (cons 'set! (cons var body))))
   'define))

(define cond
  (macro clauses
    (define (process-clause clause else-part)
      (if (not (pair? clause));; atom
          (error '(bad cond clause:) clause)
          (if (null? (rest clause));; (test)
              (list 'or (first clause) else-part)
              (if (eq? (second clause) '=>);; (test => proc)
		  ((lambda (tempvar)
		     (list (list 'lambda (list tempvar)
				 (list 'if
				       tempvar
				       (list (third clause) tempvar)
				       else-part))
			   (first clause)))
		   (string->symbol "temp var"))
                  (if (member (first clause) '(#t else));; (else x y z)
                      (cons 'begin (rest clause))
                      (list 'if (first clause) (cons 'begin (rest clause)) 
                            else-part))))))
    ;; body of cond
    (if (null? clauses)
        #f
        (process-clause (first clauses) (cons 'cond (rest clauses))))))

(define tryCatch
  (macro args
    (list 'jsint.Procedure.tryCatch
	  (list 'lambda () (first args))
	  (second args))))

(define and             
  (macro args 
    (cond ((null? args) #t)
	  ((null? (rest args)) (first args))
	  (else (list 'if (first args) (cons 'and (rest args)) #f)))))

;; OR is treated in Scheme.eval(), not here.

;; Rationale: the translation to Scheme is complex: (or a b c) =>
;; ((lambda (x) (if x x ((lambda (y) (if y y c)) b) a) but the
;; implementation in Scheme.eval is easier. Also note that if OR is
;; treated as primitive, then COND can expand to OR.  The quasiquote,
;; and a few others, are from Darius Bacon <djello@well.com> (But
;; then, he started with my PAIP code, and modified it.)

(define quasiquote 
  (macro (x) 
    (define (constant? exp)
      (if (pair? exp) (eq? (car exp) 'quote) (not (symbol? exp))))
    (define (combine-skeletons left right exp)
      (cond
       ((and (constant? left) (constant? right)) 
	(if (and (eqv? (eval left) (car exp))
		 (eqv? (eval right) (cdr exp)))
	    (list 'quote exp)
	    (list 'quote (cons (eval left) (eval right)))))
       ((null? right) (list 'list left))
       ((and (pair? right) (eq? (car right) 'list))
	(cons 'list (cons left (cdr right))))
       (else (list 'cons left right))))
    (define (expand-quasiquote exp nesting)
      (cond
       ((vector? exp)
	(list 'apply 'vector (expand-quasiquote (vector->list exp) nesting)))
       ((not (pair? exp)) 
	(if (constant? exp) exp (list 'quote exp)))
       ((and (eq? (car exp) 'unquote) (= (length exp) 2))
	(if (= nesting 0)
	    (second exp)
	    (combine-skeletons ''unquote 
			       (expand-quasiquote (cdr exp) (- nesting 1))
			       exp)))
       ((and (eq? (car exp) 'quasiquote) (= (length exp) 2))
	(combine-skeletons ''quasiquote 
			   (expand-quasiquote (cdr exp) (+ nesting 1))
			   exp))
       ((and (pair? (car exp))
	     (eq? (caar exp) 'unquote-splicing)
	     (= (length (car exp)) 2))
	(if (= nesting 0)
	    (list 'append (second (first exp))
		  (expand-quasiquote (cdr exp) nesting))
	    (combine-skeletons (expand-quasiquote (car exp) (- nesting 1))
			       (expand-quasiquote (cdr exp) nesting)
			       exp)))
       (else (combine-skeletons (expand-quasiquote (car exp) nesting)
				(expand-quasiquote (cdr exp) nesting)
				exp))))
    (expand-quasiquote x 0)))

(define let 
  (macro (bindings . body) 
    (define (named-let name bindings body)
      `(let ((,name #f))
	 (set! ,name (lambda ,(map first bindings) . ,body))
	 (,name . ,(map second bindings))))
    (if (symbol? bindings) 
	(named-let bindings (first body) (rest body))
	`((lambda ,(map first bindings) . ,body) . ,(map second bindings)))))

(define let* 
  (macro (bindings . body)
    (if (null? bindings) `((lambda () . ,body))
	`(let (,(first bindings))
	   (let* ,(rest bindings) . ,body)))))

(define letrec
  (macro (bindings . body)
    (let ((vars (map first bindings))
	  (vals (map second bindings)))
    `(let ,(map (lambda (var) `(,var #f)) vars)
       ,@(map (lambda (var val) `(set! ,var ,val)) vars vals)
       . ,body))))
    
(define case
  (macro (exp . cases)
    (let ((tempvar (string->symbol "temp var")))
      (define (do-case case)
	(cond ((not (pair? case)) (error '(bad syntax in case:) case))
	      ((eq? (first case) 'else) case)
	      (else `((member ,tempvar ',(first case)) . ,(rest case)))))
      `(let ((,tempvar ,exp)) (cond . ,(map do-case cases))))))

(define do
  (macro (bindings test-and-result . body)
    (let ((variables (map first bindings))
	  (inits (map second bindings))
	  (steps (map (lambda (clause)
			(if (null? (cddr clause))
			    (first clause)   
			    (third clause)))
		      bindings))
	  (result (if (null? (cdr test-and-result)) ''unspecified
		      `(begin . ,(cdr test-and-result)))))
      (let ((tempvar '<loop>))
	`(letrec ((,tempvar
		   (lambda ,variables
		     (if ,(first test-and-result)
			 ,result
			 (begin 
			   ,@body
			   (,tempvar . ,steps))))))
	   (,tempvar . ,inits))))))

(define delay
  (macro (exp) 
    (define (make-promise proc)
      (let ((result-ready? #f)
	    (result #f))
	(lambda ()
	  (if result-ready?
	      result
	      (let ((x (proc)))
		(if result-ready?
		    result
		    (begin (set! result-ready? #t)
			   (set! result x)
			   result)))))))
    `(,make-promise (lambda () ,exp))))

;;;;;;;;;;;;;;;; Derived Expression Extensions
(define time 
  (macro (exp . rest) `(time-call (lambda () ,exp) . ,rest)))

(define define-macro 
  (macro (spec . body) 
    (if (pair? spec)
      `(define ,(first spec) (macro ,(rest spec) . ,body))
      `(define         ,spec (macro ,(second (first body)) ,@(rest (rest (first body))))))))

(define (missing-classes classes sofar)
  (if (null? classes) sofar
      (missing-classes (cdr classes)
		       (if (eq? (class (car classes)) #null)
			   (cons (car classes) sofar)
			   sofar))))

(define-macro (if-classes classes then else)
  (if (null? (missing-classes classes '()))
      then
      else))

(define-macro (when-classes classes . then)
  `(if-classes ,classes (begin ,@then) #f))


(define-macro (class-case varlist . clauses)
   (define (runtimeClassName c)
     (string->symbol (string-append (.getName (class c)) ".class")))
   (define (instanceof v c) `(.isInstance ,(runtimeClassName c) ,v))
   `(cond ,@(map (lambda (clause)
                   (if (equal? (first clause) 'else) clause
                       `((and ,@(map instanceof varlist (first clause)))
                         ,@(rest clause))))

                 clauses)))

(define (define-method-runtime name type-names f name-args)
  (let ((missing (missing-classes type-names '())))
    (if (null? missing)
	(jsint.Generic.defineMethod name type-names f)
	(jsint.E.warn (string-append "Can't define-method " name-args
			       " classes " missing " do not exist.")))))
(define define-method
  (macro (name-args . body)
    (define (arg-name x) (if (pair? x) (car x) x))
    (define (maybe-second x default)
      (if (and (pair? x) (pair? (cdr x))) (cadr x)
	  default))
    (define (arg-type x) (maybe-second x 'java.lang.Object))
    (let* ((name (car name-args))
	   (args (cdr name-args))
	   (arg-types (map arg-type args)))
      `(define-method-runtime
       ',name ',arg-types (lambda ,(map arg-name args) ,@body)
       ',name-args))))
      
(define package  (macro args #t))

(define (array a-class . args)
  (let ((v (make-array a-class (length args))))
    (let loop ((i 0)
	       (as args))
      (if (null? as) v
	  (begin
	    (vector-set! v i (car as))
	    (loop (+ i 1) (cdr as)))))))

(define (make-array a-class size)
  (java.lang.reflect.Array.newInstance a-class size))

)					; End of begin.
