;;; Example of using import and define-method.  
;;; (describe x) uses JDK 1.2 AccessibleObject to describe private
;;; fields of object x.  

;;; Example:
;;; > (describe '(1 2 3))
;;; an instance of jsint.Pair
;;;   first: 1
;;;   rest: (2 3)
;;; ()

(import "java.lang.Class")
(import "java.util.Hashtable")
(import "java.lang.System")
(import "java.lang.reflect.AccessibleObject")
(import "java.lang.reflect.Array")
(import "java.lang.reflect.Constructor")
(import "java.lang.reflect.Field")
(import "java.lang.reflect.Method")
(import "java.lang.reflect.Modifier")
(import "jsint.U")

(define make-accessible
  ;; Make Field vector accessible, punting if not in JDK 1.2
  (if (java-version>=1_2)
      (lambda (fs) (AccessibleObject.setAccessible fs #t) fs)
      (lambda (fs) fs)))

;;; (describe) requires JDK 1.2 to get access to all fields.
(if (java-version>=1_2)
    (import "java.lang.reflect.AccessibleObject"))

(define (memoize-1 f size)
  ;; Memoize (f x) using a Hashtable of size size.
  (let ((table (Hashtable. size)))
    (lambda (key)
      (let ((it (.get table key)))
	(if (not (isNull it)) it
	    (let ((it (f key)))
	      (.put table key it)
	      it))))))

(define all-fields
  ;; All fields, including private ones, most specific first.
  (memoize-1
   (lambda (c)
     (define (all-fields-1 super so-far)
       (if (isNull super) 
	   (apply append 
		  (map (lambda (fs) (make-accessible fs)
			       (vector->list fs))
		       so-far))
	   (all-fields-1 (.getSuperclass super) 
			 (cons (.getDeclaredFields super) so-far))))
     (all-fields-1 c '()))
   100))

(define all-methods
  (memoize-1
   (lambda (c)
     (define (all-methods-1 super so-far)
       (if (isNull super)
	   (apply append
		  (map (lambda (fs) (make-accessible fs)
			       (vector->list fs))
		       so-far))
	   (all-methods-1 (.getSuperclass super)
			  (cons (.getDeclaredMethods super) so-far))))
     (all-methods-1 c '()))
   100))

(define (short-toString x max)
  (let ((it (.toString x)))
    (if (<= (string-length it) max) it
	(string-append (substring it 0 (- max 4)) " ..."))))

(define-method (describe (x java.lang.Object))
  (display (short-toString x 80)) (newline)
  (dshow 
   " is an instance of " 
   (.getName (.getClass x)))
  (describe-fields x (.getClass x)))

(define (wrap-last-class d)
  (let ((last-class #null))
    (lambda (m)
      (if (not (eqv? last-class (.getDeclaringClass m)))
	  (displays "\n  // from " (.getName (.getDeclaringClass m)) "\n"))
      (d m)
      (set! last-class (.getDeclaringClass m)))))

(define (describe-fields x superclass)
  (in-1_2
   #t
   (let ((fs (.getDeclaredFields superclass)))
     (AccessibleObject.setAccessible fs #t) ; Make them all accessible.
     (for-each*
      (wrap-last-class
       (lambda (f)			; Not static fields.
	 (if (not (Modifier.isStatic (.getModifiers f)))
	     (describe-field f x))))
      fs)
     (let ((superclass (.getSuperclass superclass)))
       (if (not (isNull superclass)) (describe-fields x superclass)))))


  (in-1_2
   #f
   (let ((fs (.getFields superclass)))
     (for-each*
      (wrap-last-class
       (lambda (f)		; Not static fields.
	  (if (not (Modifier.isStatic (.getModifiers f)))
	      (describe-field f x))))
       fs))))

(define (describe-field f x)
  (displays "  " (.getName f) ": ")
  (write (.get f x))
  (newline))

(define (dshow . items)
  (apply displays items)
  (newline))

(define-method (describe (x java.lang.Class)) (describe-class x #t))

'(define-method describe ((x java.lang.String))
  (let ((it (class x)))
    (if it (describe-class it #t)
	(print x))))

(define (class-cpl c)
  ;; Return a list describing the class precedence list of class c.
  ;; > (class-cpl (class "java.lang.Class"))
  ;; (class java.lang.Class class java.lang.Object
  ;;   interface java.io.Serializable)
  (class-cpl-1 '() (list c)))

(define (class-cpl-1 so-far tail)
  (if (null? tail) (reverse so-far)
      (let* ((c (car tail))
	     (tail (cdr tail))
	     (is (vector->list (.getInterfaces c)))
	     (super (.getSuperclass c)))
	(class-cpl-1 (cons c so-far) 
		     (if (not (isNull super))
			 (cons super (append is tail))
			 (append is tail))))))

(define (displays . items) 
  (for-each 
   (lambda (i)
     (if (pair? i) (apply displays i)
	 (display i)))
   items))

(define (describe-class the-class all?)
  ;; Describe class c trying to use every method of the class Class.
  ;; if an all? argument is provided, show all public methods and 
  ;; fields, otherwise show the declared ones.
  (define (describe-items name what)
    (if (and (not (isNull what)) (not (= (vector-length what) 0)))
	(begin
	  (display name)
	  (display ":") (newline)
	  (for-each* indent-print what)
	  (newline))))
  (define (indent-print what)
    (display "  ")
    (print what))
  (define (describe-item name what)
    (if (not (isNull what))
	(begin
	  (display name)
	  (display ": ")
	  (print what))))
  (define (maybe-display x)
    (if (and x (not (isNull x)))
	(begin (display x)
	       (write-char #' '))))
  (define (class-or-error the-class) (U.toClass the-class))
  (let ((c (class-or-error the-class)))
    (print `(class: ,c))
    (if (.isPrimitive c)
	(display "primitive "))
    (display (Modifier.toString (.getModifiers c)))
    (display (if (.isInterface c) " " " class "))
    (display (.getName c))
    (let ((super (.getSuperclass c)))
      (if (not (isNull super)) (displays " extends " (.getName super))))
    (let ((interfaces (.getInterfaces c)))
      (if (and (not (isNull interfaces))
	       (> (vector-length interfaces) 0))
	  (begin (newline) (display "  implements ")
		 (for-each (lambda (x) (display x) (display " "))
			   (map* (lambda (n)
				   (short-class-name n))
				 interfaces)))))
    (newline)
    (describe-item "HashCode" (.hashCode c))
    (describe-item "ClassLoader" (.getClassLoader c))
    (if (java-version>=1_2) (describe-item "Package" (.getPackage c)))
    (describe-item "Name" (.getName c))
    (describe-item "isArray" (.isArray c))
    (describe-item "ComponentType" (.getComponentType c))
    (describe-item "DeclaringClass" (.getDeclaringClass c))
    '(if (java-version>=1_2)
	 (describe-item "ProtectionDomain" (.getProtectionDomain c)))
    (describe-items "Signers" (.getSigners c))
    (display "\n// Constructors") (newline)
    (for-each* display-constructor (.getDeclaredConstructors c))
    (display "\n// Fields")
    (for-each* (wrap-last-class display-field)
	       ((if all? all-fields .getDeclaredFields)
		c))
    (display "\n// Methods")
    (for-each* (wrap-last-class display-method)
	       ((if all? all-methods .getDeclaredMethods) c))
       ;; KRA 13JAN99: Causes access violoation on NT and W95.
    (in-1_2 #t
	    (describe-items "\n// Classes" (.getDeclaredClasses c)))
    #f))

(define (modifier-string m) (Modifier.toString (.getModifiers m)))

(define (static-final? m)
  (let ((ms (.getModifiers m)))
    (and (Modifier.isStatic ms) (Modifier.isFinal ms))))

(define (class-name-name name)
  (let ((i (.lastIndexOf name ".")))
    (if (= i -1) name
	(.substring name (+ i 1)))))

(define (short-class-name c)
  (if (.isArray c)
      (string-append (short-class-name (.getComponentType c)) "[]")
      (class-name-name (.getName c))))

(define (commacomma items)
  (define (comma1 head items)
    (cons head (if (null? items) '()
		   (cons ", " (comma1 (car items) (cdr items))))))
  (if (null? items) (list "(" ")")
      (append (list "(") (comma1 (car items) (cdr items)) (list ")"))))


(define (display-constructor m)
  (displays "  " (modifier-string m)
	    " " (short-class-name (.getDeclaringClass m))
	    (commacomma
	     (map* short-class-name (.getParameterTypes m)))
	    ";\\")
  (newline))


(define (display-field f)
  (dshow "  " (modifier-string f) " "
	    (short-class-name (.getType f)) " "
	    (.getName f)))

(define (display-method m)
  (dshow "  " (modifier-string m) " "
	    (short-class-name (.getReturnType m)) " "
	    (.getName m)
	    (commacomma
	     (map* short-class-name (.getParameterTypes m)))))
	     

