;;;
;;; Simple testing scaffolding;
;;;
(define-macro (assert p)
  ;; Usage: `(assert ,truthhood)
  ;; errors if ,truthhood is a falsehood.
  `(if (not ,p)
       ;; (error "Assertion" ',p "failed!")
       (error {Assertion [',p] failed!})
       ))

(define-macro (do-test action result)
    `(do-test-runtime ',action (lambda () ,action) (lambda () ,result)))

(define do-test-runtime
  ;; The test is run as soon as it is read in.
  ;; Prints a messge if (not (equal? (action) result))
  (lambda (name action result)
    (let ((test-result (action))
	  (expected-result (result)))
      (if (not (equal? test-result expected-result))
	  (begin
	    (display "test ") (display name) (display " failed.") 
	    (newline)
	    (display " result: ") (write test-result) (newline)
	    (display " expected: ") (write expected-result) (newline)
	    #f)
	  #t))))

(define *tests* '())

(define-macro (define-test action result)
  ;; Define a test that will run when (run-tests) is executed.
  `(define-test-runtime ',action (lambda () ,action) (lambda () ,result)))

(define (define-test-runtime name action result)
  (set! *tests* (cons (vector name action result) *tests*))
  #t)

(define (run-tests)
  (define (test-name test) (vector-ref test 0))
  (define (test-action test) (vector-ref test 1))
  (define (test-result test) (vector-ref test 2))
  (define (test-summary wins losses)
    (display (string-append " tests: " (+ wins losses) "\n"))
    (display (string-append "  wins: " wins "\n"))
    (display (string-append "losses: " losses "\n")))
  (let ((wins 0)
	(loses 0))
    (define (run-test test)
      (if (do-test-runtime
	   (test-name test) (test-action test) (test-result test))
	  (set! wins (+ wins 1))
	  (set! loses (+ loses 1))))
    (for-each run-test *tests*)
    (test-summary wins loses)))
    
(define (isNull x ) (eq? x #null))

(define java-version>=1_2
  (let ((result 
	 (jsint.Procedure.tryCatch 
	  (lambda () (and (class "java.lang.reflect.AccessibleObject") #t)) 
	  (lambda (e) #f))))
    (lambda () result)))

(define-macro (in-1_2 test . args)
  (if (eq? (java-version>=1_2) test)
      `(begin ,@args)
      '()))

;;;
;;; Utilities.
;;;
(define (print it . arg)
  ;; Like Common Lisp's print.
  (if (null? arg)
      (begin
	(write it)
	(write-char #'\n'))
      (let ((s (car arg)))
	(write it s)
	(write-char #'\n' s)))
  it)

(define-macro (dotimes iters . body)
  ;; Like Common Lisp's (dotimes).
  (let ((var (car iters))
	(max (cadr iters))
	(result (if (pair? (cddr iters)) (caddr iters) '())))
    `(let ((<L> ,max))
       (let <loop> ((,var 0))
	    (if (< ,var <L>)
		(begin ,@body
		       (<loop> (+ ,var 1)))
		,result)))))

(define-macro (dolist iters . body)
  ;; Like Common Lisp's (dolist).
  (let ((var (car iters))
	(items (cadr iters))
	(result (or (and (pair? (cddr iters)) (caddr iters)) '())))
    `(let <loop> ((.items. ,items))
	  (if (null? .items.) ,result
	      (let ((,var (car .items.)))
		,@body
		(<loop> (cdr .items.)))))))

(define (instanceof x c)
  (and (not (eq? x #null))
       (.isAssignableFrom c (.getClass x))))
