;;;
;;; Iterators
;;;

(import "java.lang.Object")
(import "java.lang.String")

;;; Original in primproc.scm
(define (missing-classes classes sofar)
  (if (null? classes) sofar
      (missing-classes (cdr classes)
		       (if (eq? (class (car classes)) #null)
			   (cons (car classes) sofar)
			   sofar))))

(define-macro (if-classes classes then else)
  (if (null? (missing-classes classes '()))
      then
      else))

(define-macro (when-classes classes . then)
  `(if-classes ,classes (begin ,@then) #f))

(define-method (iterate (mapper jsint.Procedure) action)
  (mapper action))

;;; Hashtable and Vector specialization are only needed for JDK 1.1
(define-method (iterate (items java.util.Hashtable) action)
  (iterate (.elements items) action))

(define-method (iterate (items java.util.Vector) action)
  (iterate (.elements items) action))

(define-method (iterate (items java.util.Enumeration) action)
  (let loop ()
    (if (.hasMoreElements items)
	(begin (action (.nextElement items))
	     (loop)))))

(when-classes
 (java.util.Collection)
 (define-method (iterate (items java.util.Map) action)
   (iterate (.values items) action))

 (define-method (iterate (items java.util.Iterator) action)
   (let loop ()
     (if (.hasNext items)
       (begin (action (.next items))
	      (loop)))))

 (define-method (iterate (items java.util.Collection) action)
   (iterate (.iterator items) action))
 )

(define-method (iterate (items jsint.Pair) action)
  (let loop ((items items))
    (if (pair? items)
	(begin
	  (action (car items))
	  (loop (cdr items))))))

(define-method (iterate (items Object[]) action)
  (let loop ((i 0) 
             (L (vector-length items)))
    (if (< i L) (begin (action (vector-ref items i)) (loop (+ i 1) L)))))

(define-method (iterate (items String) action)
  (let loop ((i 0) 
             (L (string-length items)))
    (if (< i L) (begin (action (string-ref items i)) (loop (+ i 1) L)))))

(define-method (iterate (items Object) action)
  (if (.isArray (.getClass items))
      (let loop ((i 0) 
		 (L (java.lang.reflect.Array.getLength items)))
	(if (< i L)
	    (begin (action (java.lang.reflect.Array.get items i))
		   (loop (+ i 1) L))))
      (error "Don't know how to iterate over " items)))

(define-method (iterate (items java.io.BufferedReader) action)
  ;; Iterate over the lines of a buffered reader.
  (let loop ((it (.readLine items)))
    (if (not (eq? it #null))
	(begin
	  (action it)
	  (loop (.readLine items))))))

(define-method (iterate (items javax.swing.text.ElementIterator) action)
  ;; Unfortunately, this Class is not an iterator!
  (let loop ((item (.next items)))
    (if (not (isNull item))
	(begin (action item)
	       (loop (.next items))))))

(define (map* f xs)
  ;; Like map but works for any container that iterate works on.
  ;; KRA 13MAY00: +++ Someday rewrite without reverse.
  (let ((results '()))
    (iterate xs (lambda (x) (set! results (cons (f x) results))))
    (reverse results)))

(define (for-each* f xs)
  ;;; Like for-each but generalized for any container that iterate works on.
  (iterate xs f))

;;;
;;; Fold to the left.
;;;
(define (foldL xs how so-far)
  (iterate xs (lambda (x) (set! so-far (how x so-far))))
  so-far)

(define identity (lambda (x) x))

(define (keep test)
  (lambda (it sofar)
    (if (test it) (cons it sofar) sofar)))

(define (find p xs)
  ;; Find first x of xs satisfying (p x).
  (call/cc (lambda (return)
	     (iterate xs (lambda (x) (if (p x) (return x)))))))

;;; Testing
(do-test (find odd? '(2 4 6 3 5)) 3)
(do-test (find odd? #(2 4 6 3 5)) 3)

(define (filter-in p xs) (foldL xs (keep p) '()))

(do-test 
  (filter-in symbol? '(3 + x f define)) 
  '(define f x +))

(define (some p xs)
  (call/cc (lambda (return)
	     (iterate xs (lambda (x) (if (p x) (return #t))))
	     #f)))

(define (every p xs)
  (call/cc (lambda (return)
	     (iterate xs (lambda (x) (if (not (p x)) (return #f))))
	     #t)))