;;; To compile .java files you need:

;;; src-base the root directory of your .java sources.
;;; class-base the root directory where you want .class files to go.

;;; (compile-class class-base src-base "package.Class") - compile's a class.

;;; (recompile class-base src-base package recurse?) - recompile all
;;; the files in a package, decending into subpackages if recurse? is true.

;;; There are also (javac), (jar), and (rmic).

(load "build/bootstrap.scm")
(load "elf/util.scm")

(import "java.io.File")
(import "java.io.FileInputStream")
(import "java.io.FileOutputStream")
(import "java.lang.Double")
(import "java.lang.String")
(import "java.lang.System")

(define-method (delete-class-files (directory java.io.File))
  (file-walk directory 
	     (lambda (f result) (if (class-file? f) (.delete f)))
	     ()))

(define (file-separator) File.separator$)

(define (separate by items)
  ;; (separate "," '(1 2 3)) -> (1 "," 2 "," 3)
  (define (separate0 head tail)
    (if (null? tail) (list head)
	(cons head (cons by (separate0 (car tail) (cdr tail))))))
  (if (null? items) items
      (separate0 (car items) (cdr items))))

(.equals (separate "," '(1 2 3)) '(1 "," 2 "," 3))

(define (class->file name)
  ;; (class->file "foo.bar.baz") -> "foo/bar/baz.java".
  (apply string-append
	 (append (separate (file-separator)
			   (crack name "." 0))
		 (list ".java"))))

(.equals (File. (class->file "foo.bar.baz")) (File. "foo/bar/baz.java"))

(define (package->file base package)
  (string-append (File. base (apply string-append
				   (separate (file-separator)
					     (crack package "." 0))))))
(.equals (package->file (File. "base/base/") "com.bbn.icis.dm")
	 "base/base/com/bbn/icis/dm")

(define (file->class name)
  ;; (file->class (File. "foo/bar.java")) -> "foo.bar"
  (let ((name (.toString name)))
    (or (and (.endsWith name ".java")
	   (apply string-append
		  (separate "."
			    (crack (substring name 0 
					      (- (string-length name) 
						 (string-length ".java")))
				   (file-separator)
				   0))))
      (error name "doesn't end with \".java\""))))

(.equals (file->class (File. "foo/bar.java")) "foo.bar")

(define (string-array args)
  (apply array (cons String.class (map .toString args))))

(define javac
  ;; KRA 10SEP01: Looks like they came out with a second compiler for JDK 1.3
  ;; We try each we know about in turn. (Thanks to AspectJ sources!)
  (cond ((not (eq? (class "com.sun.tools.javac.Main") #null))
	 (lambda args
	   (.compile (com.sun.tools.javac.Main.) (string-array args))))
	((not (eq? (class "sun.tools.javac.Main") #null))
	 (lambda args
	   (.compile (sun.tools.javac.Main. System.out$ "javac")
		     (string-array args))))
	(else (lambda args (apply run (cons "javac" args))))))

(define (jar . args)
  (sun.tools.jar.Main.run
   (sun.tools.jar.Main. System.out$ System.err$ "jscheme-jar")
   (string-array args)))

(define (rmic . args)
  (.compile (sun.rmi.rmic.Main. System.out$ "jscheme-rmic")
	    (string-array args)))

(define (remote? c)
  ;; is c a class that needs to be rmic'd?
  ;; It is if it or one of its interfaces is java.rmi.Remote.
  (define (any p xs)
    ;; Does any element of xs satisfy (p x)?
    (and (not (null? xs)) (or (p (car xs)) (any p (cdr xs)))))
  (define (remote0? c)
    (or (eq? c java.rmi.Remote.class)
	(any remote0? (vector->list (.getInterfaces c)))))
  (and (not (.isInterface c)) (remote0? c)))

(define (compile-class class-base src-base name)
  (and (compile-file class-base src-base (class->file name))
       (let ((c (class name)))
	 ;; Sometimes a java file can have the wrong package defined
	 (if (eq? c #null) (print (list "Can't find class" name))
	     (if (remote? c)
		 (begin
		   (display "rmicing ") (display name) (newline)
		   (rmic "-classpath" (System.getProperty "java.class.path")
			 "-d" class-base
			 ;; "-sourcepath" src-base
			 name))))
	 #t)))

(define (compile-file class-base src-base file)
  ;; Compile the *.java file, file using the current CLASSPATH.
  (let* ((separator (System.getProperty "path.separator")))
    (javac
     "-g" "-deprecation" "-classpath"
     (string-append class-base separator
		    (System.getProperty "java.class.path"))
     "-d" class-base
     "-sourcepath" src-base
     (.toString (make-src-file src-base file)))))


(define (file->package base file)
  (let* ((base (.toString base))
	 (L (string-length base))
	 (package (separate
		   "."
		   (crack (.substring (.toString file)
				      (if (.endsWith base (file-separator))
					  L
					  (+ L 1)))
			  (file-separator)
			  0))))
    (apply string-append package)))

(.equals (file->package (File. "base/") (File. "base/bar/baz"))
	 (file->package (File. "base") (File. "base/bar/baz")))

(.equals (file->package (File. "base/") (File. "base/bar/baz")) "bar.baz")

(define (relativize-file base file)
  ;; (relativize-file "\\frog\\prince\\" "\\frog\\prince\\bar.java")
  ;; -> bar.java.
  (let* ((string (.toString base))
	 (L (.length string)))
    (File. (.substring (.toString file)
		       (if (.endsWith string (file-separator)) L (+ L 1))))))

(define (make-class-file class-base java-file)
  (let ((name (.getName java-file)))
    (File. class-base 
	   (string-append
	    (.getParent java-file)
	    File.separator$
	    (substring name 0 (- (string-length name) 
				 (string-length ".java")))
	    ".class"))))

(define (make-src-file src-base java-file)
  (File. src-base (.toString java-file)))

(define (needs-recompile class-file java-file)
  (if (not (.exists java-file)) (error java-file " does not exist!"))
  (or (not (.exists class-file))
      (> (.lastModified java-file) (.lastModified class-file))))

(define (recompile class-base src-base package recurse?)
  ;; Recompile an entire package, and subpackages if recurse? is #t.
  ;; Try to recompile everything that needs it, but return #t
  ;; only if there are no errors.
  (display "Recompiling package ") (display package)
  (if recurse? (display " recursively"))
  (newline)
  ;; Make the class directory if necessary.
  (.mkdirs (File. (package->file class-base package)))
  (let ((directory (File. (package->file src-base package))))
    (recompile-files class-base src-base (java-files directory))
    (if recurse?
	(map* (lambda (f)
		(recompile class-base src-base 
			   (file->package src-base f)
			   recurse?))
	      (directories directory)))))

(define (recompile-files class-base src-base jfs)
  (if (null? jfs) #t
      (let* ((it (relativize-file src-base (first jfs)))
	 (jfs (cdr jfs))
	 (java-file (make-src-file src-base it))
	 (bin-file (make-class-file class-base it))
	 ;; A *.java file needs to be recompiled if it is younger than
	 ;; its *.class file or if it's class is (remote?) so rmic can be
	 ;; run.  

	 ;; +++ We could test if the stub and skeleton files are upto
	 ;; date.
	 (win? (if (or (needs-recompile bin-file java-file)
		       (let ((c (class (file->class it))))
			 (and (not (eq? c #null)) (remote? c))))
		   (begin
		     (display "Compiling ")
		     (display java-file) (newline)
		     (compile-class class-base src-base 
				    (file->class it)))
		   #t)))
    (recompile-files class-base src-base jfs))))

(define (null->empty x) (if (eq? #null x) '() x))

(define (directories directory)
  ;; Return directories, but not CVS.
  (filter-in (lambda (f)
	       (and (.isDirectory f)
		    (not (equal? (.getName f) "CVS"))))
	     (null->empty (.listFiles directory))))

(define (java-files directory)
  (filter-in java-file? (null->empty (.listFiles directory))))

(define (recompile-class class-base src-base name)
  (let* ((it (File. (class->file name)))
	 (java-file (make-src-file src-base it))
	 (bin-file (make-class-file class-base it)))
    ;; A *.java file needs to be recompiled if it is younger than
    ;; its *.class file or if it's class is (remote?) so rmic can be
    ;; run.  

    ;; +++ We could test if the stub and skeleton files are upto
    ;; date.
    (if (or (needs-recompile bin-file java-file)
	    (let ((c (class name)))
	      (and c (remote? c))))
	(begin
	  (display "Compiling ")
	  (display java-file) (newline)
	  (compile-class class-base src-base 
			 (file->class (.toString it)))))
    bin-file))

;;; KRA 23SEP99: The prototype of build.CompilingClassLoader.
'(define (compilingClassLoader class-base src-base parent)
  (SchemeClassLoader
   parent
   (lambda (name)
     (SchemeClassLoader.toBytes
      (recompile-class class-base
		       src-base
		       (symbol->string (toString name)))))))

(define-method (copyBytes (in java.lang.Object) (out java.lang.Object))
  ;; Provide default N = 1001.
  (copyBytes in out 1001))

(define-method (copyBytes (in java.io.InputStream) 
			  (out java.io.OutputStream) 
			  N)
  (let ((bs (make-array (class "byte") N)))
    (let loop ((i (.read in bs 0 N)))
      (if (not (= i -1))
	  (begin
	    (.write out bs 0 i)
	    (loop (.read in bs 0 N)))))))

(define-method (copyBytes (in java.io.File) (out java.io.File) N)
  (let ((in (FileInputStream. in))
	(out (FileOutputStream. out)))
    (copyBytes in out N)
    (.close in)
    (.close out)))

(define-method (copyBytes (in java.net.URL) (out java.io.File) N)
  (let ((in (.openStream in))
	(out (FileOutputStream. out)))
    (copyBytes in out N)
    (.close in)
    (.close out)))

(define (do-files from-directory to-directory predicate act)
  (assert (and (.isDirectory from-directory) (.isDirectory to-directory)))
  (file-walk from-directory
	     (lambda (f so-far) 
	       (if (predicate f) 
		   (act f (swing-file from-directory to-directory f))))
	     '()))

(define (swing-file from to file)
  (let ((sfile (.toString file))
	(sfrom (.toString from)))
    (assert (= (.indexOf sfile sfrom) 0))
    (let ((relative (.substring sfile (+ (.length sfrom) 1))))
      (File. to relative))))
  
;(.equals (swing-file (File. "d:/jscheme4/jschemeweb/jscheme")
;		     (File. "d:/temp/jscheme")
;		     (File. "d:/jscheme4/jschemeweb/jscheme/Invoke.java"))
;	 (File. "d:/temp/jscheme/Invoke.java"))

(define (copy-files from to predicate)
  (assert (not (eq? from to)))
  (.mkdirs to)
  (do-files from to predicate 
	    (lambda (f1 f2)
	      ;; (print `(copy ,f1 ,f2))
	      (.mkdirs (.getParentFile f2))
	      (copyBytes f1 f2))))

;;; KRA 23APR01: These should be moved into Jscheme
(define (all-files dir kind?)
  ;; Collect all files statisfying kind? under directory, dir.
  (define (collect-if p?)
    (lambda (x so-far) (if (p? x) (cons x so-far) so-far)))
  (file-walk dir (collect-if kind?) '()))

(define (needs-update? targets sources)
  ;; Update the target files from the source files when
  ;; the youngest source File is younger than the oldest target File,
  ;; or the targets don't exist.
  (define (compose f g) (lambda (a b) (f (g a) (g b))))
  (define (the-first how xs)
    ;; (the-first < '(5 22 4 3 9)) -> 3
    ;; (the-first > '(5 22 4 3 9)) -> 22
    (if (null? xs) (error "xs: " xs " can't be null!")
	(foldL (cdr xs)
	       (lambda (x so-far) (if (how so-far x) so-far x))
	       (car xs))))
  (let ((targets (filter-in .exists targets))
	(sources (filter-in .exists sources)))
    (or (null? targets)
	(and (not (null? sources))
	     (> (.lastModified (the-first (compose > .lastModified) sources))
		(.lastModified
		 (the-first (compose < .lastModified) targets)))))))

(define (mkdir dir)
  (if (not (.exists dir)) (.mkdirs dir))
  dir)

(define (directory-walk dir how so-far)
  (if (.isDirectory dir)
      (let ((so-far (how dir so-far)))
	(for-each (lambda (f)
		    (set! so-far (directory-walk f how so-far)))
		  (vector->list (.listFiles dir)))
	so-far)
      so-far))
