;;;
;;; Utilities for bootstrapping Jscheme from sources to a jar file.
;;;

(import "java.io.BufferedReader")
(import "java.io.File")
(import "java.io.InputStreamReader")
(import "java.io.OutputStreamWriter")
(import "java.io.PrintStream")
(import "java.io.PrintWriter")
(import "java.lang.Object")
(import "java.lang.Process")
(import "java.lang.Runtime")
(import "java.lang.String")
(import "java.lang.System")
(import "java.lang.Thread")
(import "java.util.jar.JarFile")
(import "jsint.E")
(import "jsint.InputPort")

(define (print x . stream)
  (let ((s (if (null? stream) System.out$ (car stream))))
    (.println s x)
    x))

(define (drain s out)
  (let ((it (read-char s)))
    (if (not (eof-object? it))
	(begin
	  (write-char it out)
	  (.flush out)
	  (drain s out)))))

(define (shell command)
  ;; Run "command" in a background shell.
  ;; Quiet Version.
  (let* ((runtime (Runtime.getRuntime))
	 (process (.exec runtime command))
	 (out (InputPort. (.getInputStream process)))
	 (err (InputPort. (.getErrorStream process)))
	 (win (PrintWriter. System.out$))
	 (err-drain (Thread. (lambda () (drain err win))))
	 )
    ;; (display command) (newline)
    (.start err-drain)
    (drain out win)
    (.join err-drain)
    (.waitFor process)
    ;; (newline win) (display "done, status " win)
    ;; (display (.exitValue process) win) (newline win)
    (.exitValue process)))

(define (list->String[] x)
  (apply array (cons String.class (map .toString x))))

(define (run . command)
  (define (flatten xs)
    (if (null? xs) '()
	(if (pair? (car xs))
	    (flatten (append (car xs) (cdr xs)))
	    (cons (car xs) (flatten (cdr xs))))))
  (= (shell (list->String[] (flatten command))) 0))

(define-method (file-walk (file File) how so-far)
  (define (file-walk-files files how so-far)
    (for-each (lambda (f) (set! so-far (file-walk f how so-far)))
              (if (equal? #null files) ()
                  (vector->list files)))
    so-far)
  (if (.isDirectory file)
      (file-walk-files (.listFiles file) how so-far)
      (how file so-far)))

(define-method (toFile (file String)) (File. file))
(define-method (toFile (file File)) file)
(define-method (toFile (file Object)) (E.typeError "File" file))

(define (files file test)
  ;; Return all the files under and including file that pass (test file).
  (file-walk (toFile file)
	     (lambda (f so-far) (if (test f) (cons f so-far) so-far))
	     '()))

(define-method (file-type? (file java.io.File) suffix)
  (.endsWith (.getName file) suffix))

(define (java-file? file) (file-type? file ".java"))
(define (class-file? file) (file-type? file ".class"))
(define (scheme-file? file) (file-type? file ".scm"))
(define (jar-file? file) (file-type? file ".jar"))

(define (crack string separator start)
  (let ((p (.indexOf string separator start)))
    (if (= p -1) (list (.substring string start))
	(cons (.substring string start p)
	      (crack string separator (+ p 1))))))

;;; (define path-files (map File. (crack path separator 0)))

(define (classes files sofar)
  (define (classes0 file files sofar)
    (if (.exists file)
	(classes files ((if (jar-file? file) jar-classes
			    directory-classes)
			file sofar))
	(classes files sofar)))
  (if (null? files) sofar
      (classes0 (car files) (cdr files) sofar)))

(define (file-name->class f)
  (.substring f 0 (- (.length f) (.length ".class"))))

(define (jar-classes file sofar)
  (let loop ((fs (.entries (JarFile. file))))
    (if (.hasMoreElements fs)
	(let ((f (.nextElement fs)))
	  (if (class-file? f)
	      (set! sofar (cons (map .intern
				     (crack (file-name->class
					     (.getName f))
					    "/" 0))
				sofar)))
	  (loop fs))))
  sofar)

(define (directory-classes file sofar)
  ;; Get the classes in a directory
  ;; if file is "fred" the the file "fred/foo/Bar.class" is the class
  ;; foo.Bar.
  (let* ((name (.toString file))
	(L (+ (.length name) 1)))
    (if (.isDirectory file)
	(append (map (lambda (f)
		       (map .intern
			    (crack (file-name->class (.substring (.toString
f) L))
				   (System.getProperty "file.separator")
0)))
		     (files file class-file?)) sofar)
	sofar)))

(define (find-javac-bin)
  ;; Find javac in one of these places:
  ;; KRA 19MAR02: New rules from Derek Upham. 
  ;; 1. java.home/bin
  ;; 2. java.home/../bin
  ;; 3. every directory in "java.library.path"
  ;; 4. the "bin" sibling directory of every directory in "java.library.path".
  (define (any-javac fs)
    (if (null? fs) #f
	(let ((name (.getName (car fs))))
	  (or (.equals name "javac")
	      (.startsWith name "javac.")
	      (any-javac (cdr fs))))))
  (define (sibling dname sname) (File. (.getParent (File. dname)) sname))
  (define (child dname cname)   (File. dname cname))
  (define (self dname) (File. dname))
  (let* ((pathdirs (crack (System.getProperty "java.library.path")
			  (System.getProperty "path.separator")
			  0))
	 (ds (append
	      (list (sibling (System.getProperty "java.home") "bin")
		    (child (System.getProperty "java.home") "bin"))
	      (map self pathdirs)
	      (map (lambda (dname) (sibling dname "bin")) pathdirs))))
    (let loop ((ds ds))
      (if (null? ds)
	  (begin (display "\nCan't find javac in path variable! \nPlease
fix, and 
try again.\nExiting !!!\n")
		 (System.exit 1))
	  (let ((d (car ds))
		(ds (cdr ds)))
	    (if (any-javac (let ((fs (.listFiles d)))
			     (if (eq? fs #null) '()
				 (vector->list fs)))) d
				 (loop ds)))))))
