/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* Generic exceptions that will be widely used: */

#include <opencm.h>

DEFEXCEPTION(ExAssert, "A run-time assertion check failed");
DEFEXCEPTION(ExOutOfMemory, "Out of memory");
DEFEXCEPTION(ExIntegrityFail, "object integrity has failed");
DEFEXCEPTION(ExMalformed, "Object violates its schema");
DEFEXCEPTION(ExBadValue, "Bad or malformed procedure argument");
DEFEXCEPTION(ExNullArg, "Null pointer passed where object expected");
DEFEXCEPTION(ExNoAccess, "I/O access violation");
DEFEXCEPTION(ExNoObject, "File/Object does not exist");
DEFEXCEPTION(ExObjectExists, "File/Object already exists");
DEFEXCEPTION(ExTruncated, "Truncated result from I/O operation");
DEFEXCEPTION(ExOverrun,   "I/O (would have) proceeded past end of object");
DEFEXCEPTION(ExSubprocess, "Error creating subprocess");
DEFEXCEPTION(ExLockFail, "Could not acquire lock");
DEFEXCEPTION(ExLostLock, "lock lost or locking mechanism failed");
DEFEXCEPTION(ExUniverseDied, "Heat death of universe has occured. Please reset universe");
DEFEXCEPTION(ExNoConnect, "network connect/session setup failed");
DEFEXCEPTION(ExNoAuth, "could not authenticate");
DEFEXCEPTION(ExEnviron, "Needed environment variable is undefined");
DEFEXCEPTION(ExWireError, "Unspecified wire error");
DEFEXCEPTION(ExVersionError, "Unknown object version");
DEFEXCEPTION(ExConnLost, "Connection lost");
DEFEXCEPTION(ExIoError, "Error during I/O");
DEFEXCEPTION(ExBadOpcode, "Bad opcode/request in protocol");
DEFEXCEPTION(ExPrngError, "Pseudo-random number generator is broken");

/* It is ALWAYS a mistake to use this! */
DEFEXCEPTION(ExUnspecified, "Unspecified exception");

excpt_unwind *exit_unwind_list = 0;

static void
throw_doremove(catch_t *cb)
{
  while(cb->unwind) {
    excpt_unwind *uw = cb->unwind;

    if (uw->isArmed) {
      path_remove_editor_file(uw->path);
      uw->isArmed = FALSE;
    }

    cb->unwind = uw->next;
  }
}

void
_throw(catch_t *_cb, const char *fname, uint32_t line, 
       Exception ex, const char *s)
{
  _curCatch = _cb;
  _cb->excpt = ex;
  _cb->fname = fname;
  _cb->line = line;
  _cb->str = s;
  throw_doremove(_cb);
  longjmp(_cb->jbuf, 1);
}

void
onthrow_remove(const char *path)
{
  excpt_unwind *uw = GC_MALLOC(sizeof(excpt_unwind));
  uw->isArmed = TRUE;
  uw->path = xstrdup(path);

  if (_curCatch == 0) {
    uw->next = exit_unwind_list;
    exit_unwind_list = uw;
  }
  else {
    uw->next = _curCatch->unwind;
    _curCatch->unwind = uw;
  }
}

void
do_exit(int status)
{
  excpt_unwind *uw = exit_unwind_list;

  while(uw) {
    if (uw->isArmed) {
      if (path_exists(uw->path))
	path_remove(uw->path);
      uw->isArmed = FALSE;
    }
    uw = uw->next;
  }

  exit(status);
}
