/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>
#include <syslog.h>

#define LOG_TIME_STAMPS

/* Until proven otherwise, all of the logging output goes to standard
   error: */
FILE *trace_channel = 0;
FILE *error_channel = 0;
FILE *access_channel = 0;

unsigned long long log_counter = 0;

#define LOGTYPE(ty, T, S, n) {ty, #T, #S, n, 0 },
struct {
  int traceType;
  const char *class;
  const char *name;
  int level;
  unsigned shouldTrace;
} traceTypes[TRACE_NUM_TRACE] = {
#include "LogTypes.h"
};
#undef LOGTYPE

extern char *
unsigned64_to_string(uint64_t ul, char *buf);

static void
do_log(FILE *chan, const char *prefix, const char *fmt, va_list args)
{
#define BUFSZ 20 /* 19 for digits + 1 for null */
  char buf[BUFSZ];
  char *logCounterString = unsigned64_to_string(log_counter, buf);

  log_counter++;

#ifdef LOG_TIME_STAMPS
  if (server_mode)
    fprintf(chan, "%s ", os_GetISOTime());
#endif

  if (server_mode)
    fprintf(chan, "%s %s %s:", appName, logCounterString, prefix);
  else
    fprintf(chan, "%s %s: ", appName, prefix);


  vfprintf(chan, fmt, args);
  fflush(chan);
}

void
log_access(const char *fmt, ...)
{
  va_list	args;
  va_start(args, fmt);

  if (access_channel == 0)
    access_channel = stderr;

  do_log(access_channel, "ACC", fmt, args);

  va_end(args);
}

void
log_vtrace(int key, const char *fmt, va_list args)
{
  FILE *chan;

  if (trace_channel == 0)
    trace_channel = stderr;
  if (error_channel == 0)
    error_channel = stderr;

  if (traceTypes[key].traceType == LTY_trace)
    chan = trace_channel;
  else
    chan = error_channel;

  if (traceTypes[key].shouldTrace)
    do_log(chan, "TRC", fmt, args);
}

void
log_trace(int key, const char *fmt, ...)
{
  va_list	args;
  va_start(args, fmt);

  if (trace_channel == 0)
    trace_channel = stderr;
  if (error_channel == 0)
    error_channel = stderr;

  if (traceTypes[key].shouldTrace)
    do_log(trace_channel, "TRC", fmt, args);

  va_end(args);
}

#define MAX_ERROR 1024

static void
vsyslog_error(const char *fmt, va_list args)
{
  if (server_mode == 0 || opt_Verbosity) {
    fprintf(stderr, "%s: ", appName);
    vfprintf(stderr, fmt, args);
  }
  if(server_mode) {
    char buf[MAX_ERROR];
    vsnprintf(buf, MAX_ERROR-1, fmt, args);
    buf[MAX_ERROR-1]=0;

    syslog(LOG_INFO, "%s", buf);
  }
} 

void
syslog_error(const char *fmt, ...)
{
  va_list   args;

  va_start(args, fmt);

  if (error_channel == 0)
    error_channel = stderr;

  do_log(error_channel, "ERR", fmt, args);

  vsyslog_error(fmt, args);

  va_end(args);
} 

void
init_log_directory(const char *logDir)
{
  const char *err_logFile = path_join(logDir, "error_log");
  const char *acc_logFile = path_join(logDir, "access_log");
  const char *trc_logFile = path_join(logDir, "trace_log");

  FILE *f;

  if ((f = fopen(err_logFile, "a")) != NULL)
    error_channel = f;

  if ((f = fopen(acc_logFile, "a")) != NULL)
    access_channel = f;

  if ((f = fopen(trc_logFile, "a")) != NULL)
    trace_channel = f;
}

void
log_set_verbosity(int level)
{
  int i;
  for (i = 0; i < TRACE_NUM_TRACE; i++) {
    /* If this is a debugging flag, changing verbosity should not
       alter it: */
    if (traceTypes[i].level == DEBUGGING_TRACE_LEVEL)
      continue;

    /* Enable or disable the tracing flag according to the new level */
    traceTypes[i].shouldTrace = (level >= traceTypes[i].level);
  }
}

void
log_set_debugging(const char *arg)
{
  int i;
  for (i = 0; i < TRACE_NUM_TRACE; i++) {
    /* If this is a debugging flag, changing verbosity should not
       alter it: */
    if (traceTypes[i].level != DEBUGGING_TRACE_LEVEL)
      continue;

    if (strcasecmp(arg, traceTypes[i].name) == 0) {
      traceTypes[i].shouldTrace = 1;
      return;
    }
  }

  THROW(ExBadValue, "Argument to -d was unrecognized");
}

void
log_help_debugging(void)
{
  int i;
  report(0, "The following debugging flags are available:\n");

  for (i = 0; i < TRACE_NUM_TRACE; i++) {
    /* If this is a debugging flag, changing verbosity should not
       alter it: */
    if (traceTypes[i].level != DEBUGGING_TRACE_LEVEL)
      continue;

    report(0, "%s\n", xstrdowncase(traceTypes[i].name));
  }
}

int
log_shouldlog(int key)
{
  return traceTypes[key].shouldTrace;
}

void
unimplemented(const char *s)
{
  log_trace(ERR_UNIMPL, s);
  exit(1);
}
