/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>
#include <client/KeyGen.h>

/* (JL 9/16/2002)
   Currently, this exec's 'openssl req' with the appropriate commands. However,
   this is bad for several reasons:
       (1) Will not work if the openssl binary is not in PATH
       (2) Will not work if the openssl config file has changed in
           certain ways

   We work around (2) by having an option to over-ride the openssl config file,
   using either the OPENCM_SSL_CONFIG environemntal variable, or a command line
   option.  If neither is set, then we use the default.

   The default OpenSSL config, as shipped with OpenSSL *today*, is fine for our
   purposes. However, a heavy SSL-using site may have changed some parts of the
   config, and there are legitimate reasons to change the config in ways which
   would break OpenCM's usage of the certs, as mentioned by Richard Levitte in
   bug #370

   There is no reasonable way to work around (1) using subprocess_Run. This
   function is the ONLY one that exececutes openssl (as of now). Thus, in the
   future, we should re-write this function to not execute openssl, using
   OpenSSL's X509 routines directly, which will solve both of these problems
   very nicely.
*/

void Generate_X509_Key(const char* keyfile, const char* certfile,
                       OC_bool encrypt_key, uint32_t days)
{
  SubProcess *proc = subprocess_create();

  subprocess_AddArg(proc, "openssl");
  subprocess_AddArg(proc, "req");
  subprocess_AddArg(proc, "-new");

  /* FIX: Take into account an alternate OpenSSL config */

  subprocess_AddArg(proc, "-newkey");
  subprocess_AddArg(proc, opt_KeyType ? opt_KeyType : "rsa:1024");
  subprocess_AddArg(proc, "-days");
  subprocess_AddArg(proc, xunsigned_str(days));
  subprocess_AddArg(proc, "-x509");

  /* -nodes means to not encrypt the secret key. Why -nodes? That option name
     makes no sense to me... (maybe it's some piece of X.509 terminology) */
  if(!encrypt_key)
    subprocess_AddArg(proc, "-nodes");

  subprocess_AddArg(proc, "-keyout");
  subprocess_AddArg(proc, keyfile);
  subprocess_AddArg(proc, "-out");
  subprocess_AddArg(proc, certfile);

  if (subprocess_Run(proc, 0, 0, 0, SPF_NORMAL) < 0)
    THROW(ExSubprocess, "Could not execute 'openssl' to create a X.509 cert");
}

