/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>

extern struct OpcodeType opcodes[];

void
request_serialize(SDR_stream *strm, const void *ob)
{
  const Request *p = ob;
  
  sdr_w_u32("opcode", strm, p->opcode);
  sdr_w_u32("reqid", strm, p->reqID);
  sdr_write("args", strm, p->args);
  
}

void *
request_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  Request *p = (Request *) GC_MALLOC(sizeof(Request));

  ser_init(p, &Request_SerType, di->ver);
  p->opcode         = sdr_r_u32("opcode", strm);
  p->reqID          = sdr_r_u32("reqid", strm);
  p->args	    = sdr_read("args", strm);

  SER_MODIFIED(p);
  return p;
}

void
request_mark(Repository *r, const void * container,
	     const void *ob, rbtree *memObs)
{
  assert(container == ob);
  /* Leaf object. Nothing to do */
}

Request *
request_create(uint32_t ocode, uint32_t reqid)
{
  Request *p = (Request *) GC_MALLOC(sizeof(Request));

  ser_init(p, &Request_SerType, Request_SerType.ver);

  p->opcode = ocode;
  p->reqID = reqid;
  p->channel = NULL;  /* will be assigned by server */

  p->args = obvec_create();

  SER_MODIFIED(p);
  return p;
} /* end request_create */

OC_bool
request_check(const void *v)
{
  return TRUE;
}

void
request_show(const void *ob)
{
  uint32_t u;
  const Request *r = ob;
  
  report_to_tracelog(DBG_PROTO);

  TRY {
    log_trace(DBG_PROTO, "Opcode:          %u [%s]\n", 
	      r->opcode, opcodes[r->opcode].name);
    log_trace(DBG_PROTO, "ReqId:           %u\n", r->reqID);

    for (u = 0; u < vec_size(r->args); u++) {
      const Serializable *s = vec_fetch(r->args, u, const Serializable*);
      report(0, "Arg[%u]: <%s>\n", u, s->ser_type->tyName);
      sdr_show(s);
    }
  } DEFAULT(ex) {
    report_to_tracelog(TRACE_NONE);
    RETHROW(ex);
  } END_CATCH;

  report_to_tracelog(TRACE_NONE);
}

void
request_trace(const Request *r)
{
  uint32_t u;
  
  log_trace(DBG_PROTO, "Opcode:          %u [%s]\n", 
	    r->opcode, opcodes[r->opcode].name);
  log_trace(DBG_PROTO, "ReqId:           %u\n", r->reqID);

  for (u = 0; u < vec_size(r->args); u++) {
    const Serializable *s = vec_fetch(r->args, u, const Serializable*);

    /* Arg can actually be NULL (especially in the case of CreateMutable) ! */
    if (s == NULL) {
      log_trace(DBG_PROTO, "Arg[%u]: <NULL> \n", u);
      continue;
    }
    else
      log_trace(DBG_PROTO, "Arg[%u]: %s <%s>\n", u, ser_getTrueName(s), s->ser_type->tyName);

    if (opt_TraceProtocol > 2) {
      report_to_tracelog(DBG_PROTO);

      TRY {
	sdr_show(s);

      } DEFAULT(ex) {
	report_to_tracelog(TRACE_NONE);
	RETHROW(ex);
      } END_CATCH;

      report_to_tracelog(TRACE_NONE);
    }
  }
}

/*
* Method for adding an argument to this request object.  Note that
* all arguments must be Serializable.
*/
void 
request_AddArgument(Request *sr, void *obj) 
{
  obvec_append(sr->args, obj);
  SER_MODIFIED(sr);
} /* end serverrequest_AddArgument */
