/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>
#include <openssl/x509v3.h>

OC_bool
pubkey_check(const void *v)
{
  return TRUE;
}

void
pubkey_show(const void *ob)
{
  const PubKey *p = ob;
  
  report(0, "Type:        %d\n", p->type);
  switch(p->type) {
  case pk_X509:
    report(0, "Cert:        0x%08x\n", p->keys.x509_cert);
    break;
  default:
    report(0, "<unknown key type>\n");
  }
}

void
pubkey_serialize(SDR_stream *strm, const void *ob)
{
  const PubKey *p = ob;
  
  sdr_w_u32("type", strm, p->type);
  
  switch(p->type) {
  case pk_X509:
    {
      unsigned num;
      void *tmpbuf = NULL;
      char *data;

      BIO *inMem = BIO_new(BIO_s_mem());

      if (!PEM_write_bio_X509(inMem, p->keys.x509_cert))
	THROW(ExTruncated, "Cannot write X509 cert to memory");

      if ((num = BIO_get_mem_data(inMem, &tmpbuf)) == 0)
	THROW(ExTruncated, "Cannot recover cert from memory BIO");

      data = (char *)GC_MALLOC_ATOMIC(sizeof(char) * (num + 1));
      memcpy(data, tmpbuf, num);
      data[num] = 0;

      sdr_w_string("x509", strm, data);

      break;
    }
  default:
    THROW(ExBadValue, "Unknown public key type");
  }
}

void *
pubkey_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  PubKey *p = (PubKey *) GC_MALLOC(sizeof(User));

  ser_init(p, &PubKey_SerType, di->ver);
  SER_MODIFIED(p);

  p->type           = sdr_r_u32("type", strm);

  switch(p->type) {
  case pk_X509:
    {
      char *s = sdr_r_string("x509", strm);
      BIO *inMem = BIO_new(BIO_s_mem());

      BIO_puts(inMem, s);
      p->keys.x509_cert = PEM_read_bio_X509(inMem, NULL, NULL, NULL);
      BIO_set_close(inMem, BIO_NOCLOSE);
      BIO_free(inMem);

      break;
    }
  default:
    THROW(ExBadValue, "Unknown public key type");
  }

  return p;
}

void
pubkey_mark(Repository *r, const void *container,
	    const void *ob, rbtree *memObs)
{
  assert(container != ob);
  /* do nothing */
}

OC_bool
pubkey_isvalid(PubKey *pk)
{
  int i;
  int result;

  X509_LOOKUP *lookup = NULL;
  X509_STORE *cert_ctx = X509_STORE_new();
  X509_STORE_CTX *csc = X509_STORE_CTX_new();

  if (!cert_ctx || !csc)
    THROW(ExOutOfMemory, 
	  "Could not allocate cert_ctx or csc structure");

  lookup = X509_STORE_add_lookup(cert_ctx, X509_LOOKUP_hash_dir());
  if (!lookup)
    THROW(ExOutOfMemory, 
	  format("Couldn't create lookup structure when "
		 "validating User [%s]",
		 ERR_reason_error_string(ERR_get_error())));

  X509_STORE_CTX_init(csc, cert_ctx, pk->keys.x509_cert, NULL);
  csc->error = X509_V_OK;
  i = X509_verify_cert(csc);
  result = csc->error;

  X509_STORE_free(cert_ctx);
  X509_STORE_CTX_free(csc);

  if (result != X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT)
    return FALSE;

  return TRUE;
}

PubKey *
pubkey_from_X509(X509 *cert)
{
  PubKey *p = (PubKey *) GC_MALLOC(sizeof(User));

  ser_init(p, &PubKey_SerType, PubKey_SerType.ver);
  SER_MODIFIED(p);

  p->type = pk_X509;
  p->keys.x509_cert = cert;

  return p;
}

const char *
pubkey_getid(PubKey *pk)
{
  long num = 0;
  char *data = NULL;
  BIO *bio;
  EVP_PKEY *pubkey;
  void *tmpbuf = NULL;
  OPENCM_SHA *sha;

  bio = BIO_new(BIO_s_mem());

  pubkey = X509_get_pubkey(pk->keys.x509_cert);
  if (!PEM_write_bio_PUBKEY(bio, pubkey))
    THROW(ExNoAccess,
	  format("Error retrieving server public key [%s]",
		 ERR_reason_error_string(ERR_get_error())));

  if ((num = BIO_get_mem_data(bio, &tmpbuf)) == 0)
    THROW(ExNoAccess,
	  format("BIO Error when reading server public key [%s]",
		 ERR_reason_error_string(ERR_get_error())));

  /* Now allocate enough space for a string-ified version of tmpbuf*/
  data = (char *)GC_MALLOC_ATOMIC(sizeof(char) * (num + 1));
  memcpy(data, tmpbuf, num);
  data[num] = 0;  /* null-terminate the string */

  BIO_set_close(bio, BIO_NOCLOSE);
  BIO_free(bio);
    
  sha = sha_from_string(data);
  
  return sha_hexdigest(sha);
}

PubKey *
pubkey_from_PemFile(const char *fileName)
{
  FILE *f = NULL;
  PubKey *p = (PubKey *) GC_MALLOC(sizeof(User));
  TRY {
    f = xfopen(fileName, 'r', 't');

    ser_init(p, &PubKey_SerType, PubKey_SerType.ver);
    SER_MODIFIED(p);

    p->type = pk_X509;
    p->keys.x509_cert = PEM_read_X509(f, NULL, NULL, NULL);

    xfclose(f);
  }
  DEFAULT(ex) {
    xfclose(f);
    RETHROW(ex);
  }
  END_CATCH;

  return p;
}

	    /* Get the expiration date of cert */
const char *
pubkey_GetExpiration(PubKey *pk)
{
  switch(pk->type) {
  case pk_X509:
    {
      BIO *inMem = BIO_new(BIO_s_mem());
      unsigned num;
      char *efix;
      char *tmpbuf;
      char *expires;

      ASN1_TIME_print(inMem, X509_get_notAfter(pk->keys.x509_cert));
      num = BIO_get_mem_data(inMem, &tmpbuf);

      expires = (char *)GC_MALLOC_ATOMIC(sizeof(char) * (num + 1));
      memcpy(expires, tmpbuf, num);
      expires[num] = 0;

      /* Get rid of extraneous characters */
      efix = strstr(expires, "GMT");
      if (efix) {
	efix = efix + 3;
	*efix = 0;
      }

      return expires;
    }
  default:
    return 0;
  }
}

const char *
pubkey_GetEmail(PubKey *pk)
{
  if(pk == 0)
     THROW(ExNullArg, "pubkey_GetEmail: null PubKey pointer");

  switch(pk->type) {
  case pk_X509:
    {
      X509 *x = pk->keys.x509_cert;
      STACK* emails;
      const char* email = 0;

      emails = X509_get1_email(x);

      if(sk_num(emails) > 0)
         email = xstrdup(sk_value(emails, 0));

      return email;
    }
  default:
    THROW(ExBadValue, "Unknown key type in pubkey_GetEmail");
  }
}

#if 0
const char *
pubkey_GetName(PubKey *pk, const char *fallback)
{
  const char* email = pubkey_GetEmail(pk);
  if(email)
     return email;

  switch(pk->type) {
  case pk_X509:
    {
      X509 *x = pk->keys.x509_cert;

      char commonName[100];

      commonName[0] = '\0';

      X509_NAME_get_text_by_NID(X509_get_subject_name(x),
				NID_commonName, commonName, 100);

      if (commonName[0])
	return xstrdup(commonName);
    }
  default:
    return fallback;
  }
}
#endif
