/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>

char *RepositoryURI = 0;

int
validate_pet_name(const char *s)
{
  if (s == 0)
    return 0;
  if (strlen(s) == 0)
    return 0;

  if (s[0] == '-' || s[0] == '.')
    return 0;
      
  while (*s) {
    char c = *s++;
    if (isalnum(c) == 0 &&
	c != '_' &&
	c != '.' &&
	c != '/' &&
	c != '-')
      return 0;
  }

  return 1;  
}

char *
xstrcat(const char *s1, const char *s2)
{
  char *news =
    (char *) GC_MALLOC_ATOMIC (sizeof(char) * (strlen(s1) + strlen(s2) + 1));

  strcpy(news, s1);
  strcat(news, s2);
  return news;
}

char *
xstrdup(const char *s)
{
  char *news =
    (char *) GC_MALLOC_ATOMIC(sizeof(char) * (strlen(s) + 1));

  strcpy(news, s);
  return news;
}

char *
xstrndup(const char *s, int len)
{
  char *news;
  
  if (strlen(s) < len)
    return xstrdup(s);

  news = (char *) GC_MALLOC_ATOMIC(sizeof(char) * (len + 1));

  strncpy(news, s, len);
  news[len] = 0;
  return news;
}

/* This is needful because the precision of an unsigned value varies
 * from platform to platform, so it is very easy to get this
 * conversion wrong by mis-sizing the buffer. The logic of the buffer
 * size selection is as follows:
 *
 * Every byte of the unsigned contributes 8 bits. Every decimal digit
 * conveys 3 bits (plus some). Therefore, if we reserve space for
 * three decimal digits per input byte in the scratch buffer we will
 * be guaranteed to have more than we need. */
char *
xunsigned_str(unsigned u)
{
  char buf[sizeof(unsigned)*3 + 1];

  sprintf(buf, "%u", u);

  return xstrdup(buf);
}

/* bc says that 2^64-1==18446744073709551615, which looks to me like
   19 characters.
*/
char *
xunsigned64_str(oc_uint64_t ul)
{
#define BUFSZ 20 /* 19 for digits + 1 for null */
  char buf[BUFSZ];
  char *p;

  buf[BUFSZ-1] = 0;
  p = &buf[BUFSZ-1];

  if (ul == 0)
    return xstrdup("0");

  while(ul) {
    *(--p) = '0' + (ul % 10);
    ul = ul / 10;
  }

  return xstrdup(p);
}

oc_uint64_t
atollu(const char *s)
{
  oc_uint64_t ull = 0;

  for ( ; *s; s++) {
    ull *= 10;
    ull += (*s - '0');
  }

  return ull;
}

OC_bool
nmequal(const char *s1, const char *s2)
{
  /* unnamed and unnamed are considered NOT to match. */
  if (s1 == 0)
    return FALSE;
  if (s2 == 0)
    return FALSE;

  return (strcmp(s1, s2) == 0) ? TRUE : FALSE;
}

/* The only reason this exists is because I don't want to have to
 * bother being careful about printing null strings. Some
 * implementations of xprintf() behave gracelessly about that.
 */

/* This is derived from the EROS kprintf code */
size_t
xprintf(const char *fmt, ...)
{
  unsigned long len;
  unsigned long width = 0;
  OC_bool sign;
  OC_bool rightAdjust;
  char fillchar;
  char buf[20];
  char *p, *pend;
  size_t output_count = 0;
  
  va_list ap;

  va_start(ap, fmt);
    
  for( ; *fmt; fmt++) {
    if (*fmt != '%') {
      putchar(*fmt);
      output_count ++;
      continue;
    }

    /* largest thing we might convert fits in 20 digits (unsigned long
     * long as decimal */
    
    pend = &buf[20];
    p = pend;

    fmt++;			/* now looking at specification */

    /* check for left adjust.. */
    rightAdjust = TRUE;
    if (*fmt == '-') {
      rightAdjust = FALSE;
      fmt++;
    }
      
    fillchar = ' ';
    
    /* we just saw a format character.  See if what follows
     * is a width specifier:
     */
    width = 0;

    if (*fmt == '0')
      fillchar = '0';

    while (*fmt && *fmt >= '0' && *fmt <= '9') {
      width *= 10;
      width += (*fmt - '0');
      fmt++;
    }
    
    assert (*fmt);		/* check bogus fmt */

    sign = FALSE;
    
    switch (*fmt) {
    case '%':
      {
	putchar(*fmt);
	output_count ++;
	break;
      }
    case 's':
      {
	p = pend = va_arg(ap, char *);
      
	if (pend == 0)
	  p = pend = "<null>";

	while (*pend)
	  pend++;
	  
	break;
      }
    case 'c':
      {
	long c;
	c = va_arg(ap, long);
	*(--p) = (char) c;
	break;
      }	    
    case 'd':
      {
	long l;
	unsigned long ul;

	l = va_arg(ap, long);
	      
	if (l == 0) {
	  *(--p) = '0';
	}
	else {
	  if (l < 0)
	    sign = '-';

	  ul = (l < 0) ? (unsigned) -l : (unsigned) l;

	  if (l == LONG_MIN)
	    ul = ((unsigned long) LONG_MAX) + 1ul;

	  while(ul) {
	    *(--p) = '0' + (ul % 10);
	    ul = ul / 10;
	  }
	}
	break;
      }
    case 'u':
      {
	unsigned long ul;

	ul = va_arg(ap, unsigned long);
	      
	if (ul == 0) {
	  *(--p) = '0';
	}
	else {
	  while(ul) {
	    *(--p) = '0' + (ul % 10);
	    ul = ul / 10;
	  }
	}
	break;
      }
    case 'x':
      {
	unsigned long ul;
	static char *hex_digits = "0123456789abcdef";

	ul = va_arg(ap, unsigned long);
	      
	if (ul == 0) {
	  *(--p) = '0';
	}
	else {
	  while(ul) {
	    *(--p) = hex_digits[ul % 16];
	    ul = ul / 16;
	  }
	}
	break;
      }
    default:
      {
	assert(FALSE);
      }
    }
  
    len = pend - p;
    if (sign)
      len++;

    /* do padding with initial spaces for right justification: */
    if (width && rightAdjust && len < width) {
      while (len < width) {
	putchar(fillchar);
	output_count ++;
	width--;
      }
    }

    if (sign)
      putchar(sign);

    /* output the text */
    while (p != pend) {
      putchar(*p++);
      output_count ++;
    }
    
    /* do padding with trailing spaces for left justification: */
    if (width && rightAdjust == FALSE && len < width) {
      while (len < width) {
	putchar(fillchar);
	output_count ++;
	width--;
      }
    }
  }
    
  va_end(ap);

  return output_count;
}

int
xassert_fail(const char * expr, const char *file, unsigned line)
{
  THROW(ExAssert,
        format("Assertion %s is false, in file %s, line %d\n",
               expr, file, line));
}

/* This is _exactly_ like bsearch, except that if base == NULL, simply return NULL
   Normally this should work anyway, but it seems Solaris has 'issues' with
   this -- thus, this wrapper function.
*/
void* 
xbsearch(const void *key, const void *base, size_t nmemb,
	 size_t size, int (*compar)(const void *, const void *))
{
  /* FIX: Should we check for base==NULL and nmemb > 0? */
  if(base == NULL)
    return NULL;
  return bsearch(key, base, nmemb, size, compar);
}

const char *
strnchr(const char *s, size_t len, char c)
{
  size_t pos;

  for (pos = 0; pos < len; pos++) {
    if (s[pos] == c)
      return s + pos;
  }

  return 0;
}
