#ifndef COMMON_SDR_H
#define COMMON_SDR_H

/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* Serialization logic. This generates the XDR format (or it should),
 * but uses dynamic memory allocation
 */

#define SDR_AUTO   0		/* try to autodetect stream type */
#define SDR_RAW    1		/* not self identifying! */
#define SDR_BINARY 2
#define SDR_TEXTY  3
#define SDR_DTEXTY 4

#define STREAM_FILE    1
#define STREAM_MEM     2
#define STREAM_BUFFER  3

typedef struct SDR_stream SDR_stream;

SDR_stream*    stream_createfile(const char *fileName, unsigned format);

SDR_stream*    stream_createMemory(unsigned format);
SDR_stream*    stream_createBuffer(unsigned format);

SDR_stream*    stream_create_sha1(void);

SDR_stream*    stream_fromfile(const char *fileName, unsigned formet);
SDR_stream*    stream_fromMemory(const char *buffer, unsigned len, unsigned format);
SDR_stream*    stream_fromBuffer(Buffer *buf, unsigned format);

ocmoff_t       stream_length(SDR_stream *);
ocmoff_t       stream_position(SDR_stream *);
Buffer *       stream_asBuffer(SDR_stream *);

void           stream_reread(SDR_stream *);
void           stream_close(SDR_stream *);

int            stream_putc(SDR_stream*, int c);
int            stream_getc(SDR_stream*);
int            stream_ungetc(SDR_stream*, int c);

ocmoff_t       stream_write(SDR_stream*, const void *, ocmoff_t len);
ocmoff_t       stream_read(SDR_stream*, void *, ocmoff_t len);

Buffer *       stream_read_buffer(SDR_stream*, ocmoff_t len);

void           stream_write_buffer(SDR_stream *, const Buffer *);
void           stream_write_partial_buffer(SDR_stream *, const Buffer *, 
					   ocmoff_t start, ocmoff_t len);

size_t         stream_vprintf(SDR_stream*, const char *fmt, va_list);
size_t         stream_printf(SDR_stream*, const char *fmt, ...);
size_t         stream_scanf(SDR_stream*, const char *fmt, ...);




void           sdr_w_u8(const char *elem, SDR_stream *,
			unsigned char);
unsigned char  sdr_r_u8(const char *elem, SDR_stream *);

void           sdr_w_u16(const char *elem, SDR_stream *,
			 unsigned short);
unsigned short sdr_r_u16(const char *elem, SDR_stream *);

void           sdr_w_u32(const char *elem, SDR_stream *,
			   unsigned long);
unsigned long  sdr_r_u32(const char *elem, SDR_stream *);

void           sdr_w_u64(const char *elem, SDR_stream *,
			 oc_uint64_t);
oc_uint64_t    sdr_r_u64(const char *elem, SDR_stream *);

void           sdr_w_bytes(const char *elem, SDR_stream *,
			   unsigned long len, const void *);
void *         sdr_r_bytes(const char *elem, SDR_stream *,
			   unsigned long len);

void           sdr_w_buffer(const char *elem, SDR_stream *, const Buffer*);
Buffer *       sdr_r_buffer(const char *elem, SDR_stream *, ocmoff_t len);

void           sdr_w_string(const char *elem, SDR_stream *, const char *);
char *         sdr_r_string(const char *elem, SDR_stream *);

void           sdr_w_obname(const char *elem, SDR_stream *, const char *);
char *         sdr_r_obname(const char *elem, SDR_stream *);

/* Interface for reading/writing serializable objects. Write is
   non-const because serializer may canonicalize. */
void           sdr_write(const char *elem, SDR_stream *, const void *);
void *         sdr_read(const char *elem, SDR_stream *);

void           sdr_show(const void *);

/* Format the print-style arguments into a dynamically allocated
   string. This is a wrapper for stream_vprintf(), intended for use
   with the exceptions package to format the explanation string. */
const char *format(const char *fmt, ...);

#endif /* COMMON_SDR_H */
