/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>
#ifdef MALLOC_DEBUG
#include <mcheck.h>
#endif

const unsigned growBy = 64;

void *
vec_dup(void *vp)
{
  unsigned u;

  ObVec *vec = vp;
  ObVec *newvec = (ObVec *) GC_MALLOC(sizeof(ObVec));
  
  newvec->serType    = vec->serType;
  SER_MODIFIED(newvec);

  newvec->size = vec->size;
  newvec->maxSize = vec->size;

  newvec->elements = 
      (const void **) GC_MALLOC(sizeof(void *) *newvec->maxSize);

  for (u = 0; u < vec->size; u++)
    newvec->elements[u] = vec->elements[u];

  return newvec;
}


static void
vec_MaybeGrow(ObVec *vec)
{
  unsigned u;
  unsigned sz = vec->size;
  
  if (vec->maxSize == vec->size) {
    vec->maxSize += growBy;
    vec->elements =
      (const void **) GC_realloc(vec->elements, sizeof(void *) * vec->maxSize);

    for (u = sz; u < vec->maxSize; u++)
      vec->elements[u] = 0;
  }
}

void
vec_reset(void *v)
{
  unsigned i;
  ObVec *vec = v;
  
  for (i = 0; i < vec->size; i++)
    vec->elements[i] = 0;
    
  vec->size = 0;
  SER_MODIFIED(vec);
}
  
ObVec *
obvec_shallow_copy(ObVec *vec)
{
  int i;
  ObVec *newVec = obvec_create();
  for (i = 0; i < vec_size(vec); i++)
    obvec_append(newVec, vec_fetch(vec, i));

  return newVec;
}

void
obvec_append(ObVec *vec, const void *vp)
{
  vec_MaybeGrow(vec);
  vec->elements[vec->size++] = vp;
  SER_MODIFIED(vec);
}

void
strvec_append(StrVec *vec, const char *s)
{
  obvec_append((ObVec *) vec, s);
}

void
tnvec_append(TnVec *vec, const char *s)
{
  obvec_append((ObVec *) vec, s);
}

void
obvec_insert(ObVec *vec, const void *vp, unsigned ndx)
{
  unsigned i;
  
  assert (ndx <= vec->size);
  
  SER_MODIFIED(vec);
  vec_MaybeGrow(vec);

  for (i = vec->size; i > ndx; i++)
    vec->elements[i] = vec->elements[i-1];

  vec->size++;

  vec->elements[ndx] = vp;
}

void
obvec_set(ObVec *vec, const void *vp, unsigned ndx)
{
  assert (ndx < vec->size);
  vec->elements[ndx] = vp;
  SER_MODIFIED(vec);
}

void
strvec_insert(StrVec *vec, const char *s, unsigned ndx)
{
  obvec_insert((ObVec *)vec, s, ndx);
}

void
tnvec_insert(TnVec *vec, const char *s, unsigned ndx)
{
  obvec_insert((ObVec *)vec, s, ndx);
}

void
strvec_set(StrVec *vec, const char *vp, unsigned ndx)
{
  assert (ndx < vec->size);
  vec->elements[ndx] = vp;
  SER_MODIFIED(vec);
}

void
tnvec_set(TnVec *vec, const char *vp, unsigned ndx)
{
  assert (ndx < vec->size);
  vec->elements[ndx] = vp;
  SER_MODIFIED(vec);
}

void
vec_remove(void *vd_vec, unsigned ndx)
{
  unsigned i;
  ObVec *vec = vd_vec;
  
  assert (ndx < vec->size);
  
  for (i = ndx; i < (vec->size - 1); i++)
    vec->elements[i] = vec->elements[i+1];

  vec->elements[vec->size - 1] = 0;
  
  vec->size--;
  SER_MODIFIED(vec);
}

void
obvec_serialize(SDR_stream *strm, const void *vp)
{
  unsigned i;
  const ObVec *vec = vp;
  
  sdr_w_u32("len", strm, vec->size);

  for (i = 0; i < vec->size; i++)
    sdr_write("elem", strm, vec->elements[i]);
}

void *
obvec_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  ObVec *vec = obvec_create();
  unsigned size, i;
  
  vec->serType       = &ObVec_SerType;
  SER_MODIFIED(vec);

  size = sdr_r_u32("len", strm);
  
  for (i = 0; i < size; i++) {
    void *elem = sdr_read("elem", strm);
    obvec_append(vec, elem);
  }

  return vec;
}

void
obvec_show(const void *v)
{
  const ObVec *vec = v;

  xprintf("<obvec[%u]>\n", vec->size);
}

void
strvec_serialize(SDR_stream *strm, const void *v)
{
  const StrVec *vec = v;
  unsigned i;
  
  sdr_w_u32("len", strm, vec->size);

  for (i = 0; i < vec->size; i++)
    sdr_w_string("elem", strm, vec->elements[i]);
}

void *
strvec_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  StrVec *vec = strvec_create();
  unsigned size, i;
  
  vec->serType       = &StrVec_SerType;
  SER_MODIFIED(vec);

  size = sdr_r_u32("len", strm);
  
  for (i = 0; i < size; i++) {
    char *elem = sdr_r_string("elem", strm);
    strvec_append(vec, elem);
  }

  return vec;
}

void
tnvec_serialize(SDR_stream *strm, const void *v)
{
  const StrVec *vec = v;
  unsigned i;
  
  sdr_w_u32("len", strm, vec->size);

  for (i = 0; i < vec->size; i++)
    sdr_w_obname("elem", strm, vec->elements[i]);
}

void *
tnvec_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  StrVec *vec = tnvec_create();
  unsigned size, i;
  
  vec->serType       = &TnVec_SerType;
  SER_MODIFIED(vec);

  size = sdr_r_u32("len", strm);
  
  for (i = 0; i < size; i++) {
    char *elem = sdr_r_obname("elem", strm);
    tnvec_append(vec, elem);
  }

  return vec;
}

void
strvec_show(const void *v)
{
  const StrVec *vec = v;

  xprintf("<strvec[%u]>\n", vec->size);
}

void
tnvec_show(const void *v)
{
  const StrVec *vec = v;

  xprintf("<strvec[%u]>\n", vec->size);
}

StrVec *
strvec_create(void)
{
  StrVec *vec = (StrVec *) GC_MALLOC(sizeof(StrVec));

  vec->serType = &StrVec_SerType;
  SER_MODIFIED(vec);

  vec->size = 0;
  vec->maxSize = 0;
  vec->elements = 0;

  return vec;
}

TnVec *
tnvec_create(void)
{
  TnVec *vec = (TnVec *) GC_MALLOC(sizeof(TnVec));

  vec->serType = &TnVec_SerType;
  SER_MODIFIED(vec);

  vec->size = 0;
  vec->maxSize = 0;
  vec->elements = 0;

  return vec;
}

ObVec *
obvec_create()
{
  ObVec *vec = (ObVec *) GC_MALLOC(sizeof(ObVec));

  vec->serType = &ObVec_SerType;
  SER_MODIFIED(vec);

  vec->size = 0;
  vec->maxSize = 0;
  vec->elements = 0;

  return vec;
}

char *
strvec_flatten(StrVec *vec)
{
  unsigned i;
  unsigned cumlen = 0;
  char *s;
  
  for (i = 0; i < vec->size; i++) {
    cumlen = cumlen + strlen(vec->elements[i]);
  }

  cumlen += 1;			/* for trailing null */

  s = GC_MALLOC_ATOMIC(cumlen);
  cumlen = 0;

  for (i = 0; i < vec->size; i++) {
    unsigned len = strlen(vec->elements[i]);
    strcpy(&s[cumlen], vec->elements[i]);
    cumlen += len;
  }

  return s;
}

static int
cmp_ent(const void *e1, const void *e2)
{
  const Serializable *s1 = *((const Serializable **) e1);
  const Serializable *s2 = *((const Serializable **) e2);

  assert(s1 != 0);
  assert(s2 != 0);

  return strcmp(ser_getTrueName(s1), ser_getTrueName(s2));
}

static int
cmp_name(const void *v1, const void *v2)
{
  const char *s1 = *((const char **) v1);
  const char *s2 = *((const char **) v2);

  assert(s1 != 0);
  assert(s2 != 0);
  return strcmp(s1, s2);
}

void
strvec_sort(StrVec *vec)
{
  SER_MODIFIED(vec);
  /* FIX: mergeHist */
  qsort(vec->elements, vec->size, sizeof(const char *), cmp_name);
}

void
tnvec_sort(StrVec *vec)
{
  SER_MODIFIED(vec);
  /* FIX: mergeHist */
  qsort(vec->elements, vec->size, sizeof(const char *), cmp_name);
}

void
obvec_sort(ObVec *vec)
{
  SER_MODIFIED(vec);
  qsort(vec->elements, vec->size, sizeof(const void *), cmp_ent);
}

void
obvec_sort_using(ObVec *vec, int (*cmp)(const void *, const void *))
{
  SER_MODIFIED(vec);
  qsort(vec->elements, vec->size, sizeof(const void *), cmp);
}

static int
strvec_keycmp(const void *vkey, const void *vmember)
{
  const char *key = (const char *) vkey;
  const char **member = (const char **) vmember;
  return strcmp(key, (*member));
}

void
strvec_bremove(StrVec *vec, const char *s)
{
  const char **entry = 
    xbsearch(s, vec->elements, vec->size, sizeof(char*), strvec_keycmp);

  if (entry) {
    vec_remove(vec, (entry - (const char **) vec->elements));
    SER_MODIFIED(vec);
  }
}

int
strvec_bsearch(StrVec *vec, const char *s)
{
  char **entry = 
    xbsearch(s, vec->elements, vec->size, sizeof(char*), strvec_keycmp);

  if (entry)
    return entry - (char **)vec->elements;
  else
    return -1;
}

void
tnvec_bremove(TnVec *vec, const char *s)
{
  const char **entry = 
    xbsearch(s, vec->elements, vec->size, sizeof(char*), strvec_keycmp);

  if (entry) {
    vec_remove(vec, (entry - (const char **) vec->elements));
    SER_MODIFIED(vec);
  }
}

int
tnvec_bsearch(StrVec *vec, const char *s)
{
  char **entry = 
    xbsearch(s, vec->elements, vec->size, sizeof(char*), strvec_keycmp);

  if (entry)
    return entry - (char **)vec->elements;
  else
    return 0;
}
