/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>

Directory *
directory_create(void)
{
  Directory *n = (Directory *)GC_MALLOC(sizeof(Directory));

  n->serType = &Directory_SerType;
  SER_MODIFIED(n);

  n->entries = obvec_create();

  return n;
}

void
directory_Clear(Directory *ns)
{
  vec_reset(ns->entries);
  SER_MODIFIED(ns);
}

static OC_bool
key_exists(Directory *ns, const char *check)
{
  int u;

  for (u = 0; u < vec_size(ns->entries); u++) {
    Dirent *entry = (Dirent *)vec_fetch(ns->entries, u);
    if (nmequal(entry->key, check))
      return TRUE;
  }
  return FALSE;
}

int
directory_Add(Directory *ns, const char *key, const char *value)
{
  if (key_exists(ns, key))
    return -1;   /* that key is already used */
  else {
    obvec_append(ns->entries, dirent_create(key, value));
    SER_MODIFIED(ns);
    return 0;
  }
}

int
directory_Remove(Directory *ns, const char *key)
{
  int u;

  for (u = 0; u < vec_size(ns->entries); u++) {
    Dirent *entry = (Dirent *)vec_fetch(ns->entries, u);
    if (nmequal(entry->key, key)) {
      vec_remove(ns->entries, u);
      SER_MODIFIED(ns);
      return 0;
    }
  }
  return -1;   /* key not found */ 
}

const char *
directory_Lookup(Directory *ns, const char *key)
{
  int u;

  for (u = 0; u < vec_size(ns->entries); u++) {
    Dirent *entry = (Dirent *)vec_fetch(ns->entries, u);
    if (nmequal(entry->key, key))
      return entry->value;
  }
  return NULL;
}

void
directory_show(const void *vp)
{
  int u;
  const Directory *n = vp;

  for (u=0; u < vec_size(n->entries); u++) {
    sdr_show((Dirent *)vec_fetch(n->entries, u));
  }

}

void
directory_serialize(SDR_stream *strm, const void *ob)
{
  const Directory *n = ob;

  sdr_write("entries", strm, n->entries);
}

void *
directory_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  Directory * ns = (Directory *)GC_MALLOC(sizeof(Directory));

  ns->serType = &Directory_SerType;
  SER_MODIFIED(ns);

  ns->entries = sdr_read("entries", strm);

  return ns;
}
