/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>
#include "../../zlib/zlib.h"

#define BUFFER_BLOCK_SIZE 1024

struct Buffer {
  Serializable ser;

  OC_bool frozen;

  ocmoff_t bias;		/* for window buffers */
  ocmoff_t end;

  struct ObVec *vec;
};


#define BLKNDX(pos) ((pos) / BUFFER_BLOCK_SIZE)
#define BLKOFF(pos) ((pos) % BUFFER_BLOCK_SIZE)

void
buffer_show(const void *vp)
{
  const Buffer *buf = vp;
  
  xprintf("Frozen:   %c\n", buf->frozen ? 'Y' : 'N');
  xprintf("Bias:     %s\n", xunsigned64_str(buf->bias));
  xprintf("End:      %s\n", xunsigned64_str(buf->end));
  xprintf("Descrip:\n");
}

/* This is mildly tricky. When we serialize a buffer OUT, we must
   serialize it in such a way that it reads back in as an unwindowed
   buffer. This means that the /end/ value that is serialized out must
   be reduced by the /bias/ field. */
void
buffer_serialize(SDR_stream *strm, const void *vp)
{
  const Buffer *buf = vp;

  sdr_w_u64("end", strm, buf->end - buf->bias);
  sdr_w_buffer("bits", strm, (Buffer *)vp);
}

void *
buffer_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  ocmoff_t len;
  Buffer *buf;

  len           = sdr_r_u64("end", strm);
  buf           = sdr_r_buffer("bits", strm, len);

  SER_MODIFIED(buf);
  
  return buf;
}

Buffer *
buffer_create(void)
{
  Buffer *buf = (Buffer *) GC_MALLOC(sizeof(Buffer));
  buf->serType        = &Buffer_SerType;
  SER_MODIFIED(buf);

  buf->frozen = FALSE;
  buf->bias = 0;
  buf->end = 0;
  buf->vec = obvec_create();

  return buf;
}

Buffer *
buffer_fromBuffer(const Buffer *in, ocmoff_t start, ocmoff_t len)
{
  Buffer *buf = buffer_create();

  if (!in->frozen)
    THROW(ExBadValue, "buffer_fromBuffer(): Input buffer is not frozen");

  buf->frozen = TRUE;
  buf->bias = start + in->bias;
  buf->end = buf->bias + len;
  buf->vec = in->vec;

  if (buf->end > in->end)
    THROW(ExOverrun, "buffer_fromBuffer(): new buffer would overrun old");
  return buf;
}

void
buffer_freeze(Buffer *buf)
{
  buf->frozen = TRUE;
}

ocmoff_t
buffer_length(const Buffer *buf)
{
  return buf->end - buf->bias;
}

/* buffer_append() can only be applied on non-frozen buffers. Mutable
   buffers always have a bias value of zero. */
void
buffer_append(Buffer *buf, const void *vp, ocmoff_t len)
{
  ocmoff_t end = buf->end + len;
  unsigned char *bp = (unsigned char *) vp;

  while (BLKNDX(end) >= vec_size(buf->vec)) {
    unsigned char* block = GC_MALLOC_ATOMIC(BUFFER_BLOCK_SIZE);
    obvec_append(buf->vec, block);
  }
  
  while (len > 0) {
    ocmoff_t start = buf->end;
    size_t blkoff = BLKOFF(start);
    size_t take = min(BUFFER_BLOCK_SIZE - blkoff, len);
    unsigned char *block = 
      (unsigned char *) vec_fetch(buf->vec, BLKNDX(start));

    block += blkoff;
    memcpy(block, bp, take);
    bp += take;
    buf->end += take;
    len -= take;
  }
}

void
buffer_appendString(Buffer *buf, const char *s)
{
  buffer_append(buf, s, strlen(s));
}

/* getChunk returns the next linear byte subsequence in a Buffer,
   along with the length of that sequence. It honors both the length
   restrictions on the buffer and also any windowing that has been
   applied (the bias field). */
BufferChunk
buffer_getChunk(const Buffer *buf, ocmoff_t pos, ocmoff_t len)
{
  pos += buf->bias;

  if (pos >= buf->end)
    THROW(ExOverrun, "Buffer length exceeded in buffer_getChunk\n");

  {
    BufferChunk bc;
    unsigned blkoff = BLKOFF(pos);
    ocmoff_t take = min(BUFFER_BLOCK_SIZE - blkoff, buf->end - pos);
    unsigned char *block = (unsigned char *) vec_fetch(buf->vec, BLKNDX(pos));

    bc.ptr = block + blkoff;
    bc.len = min(take, len);

    return bc;
  }
}

Buffer *
buffer_FromFilePtr(const char *name, FILE *f)
{
  Buffer *buf = buffer_create();

  rewind(f);

  for ( ;; ) {
    unsigned char *block = GC_MALLOC_ATOMIC(BUFFER_BLOCK_SIZE);

    size_t take = fread(block, 1, BUFFER_BLOCK_SIZE, f);
    if (take == 0 && feof(f))
      break;
    else if (take == 0)
      THROW(ExTruncated, format("Could not read all of \"%s\" (errno %d)",
				name, errno));

    obvec_append(buf->vec, block);
    buf->end += take;
  }

  buf->frozen = TRUE;

  return buf;
}

Buffer *
buffer_FromFile(const char *fileName, unsigned char eType)
{
  FILE *f = NULL;
  Buffer *buf = NULL;

  assert(eType == 'T' || eType == 'B');
  
  TRY {
    f = xfopen(fileName, 'r', (eType == 'T') ? 't' : 'b');
  
    buf = buffer_FromFilePtr(fileName, f);

    xfclose(f);
  }
  DEFAULT(ex) {
    xfclose(f);
    RETHROW(ex);
  }
  END_CATCH;

  return buf;
}

void 
buffer_ToFile(const Buffer *buf, const char *fileName, 
	      unsigned char eType)
{
  FILE *f = NULL;
  const char *dir;

  assert(eType == 'T' || eType == 'B');
  
  dir = path_dirname(fileName);
  path_smkdir(dir);
			
  TRY {
    ocmoff_t end = buffer_length(buf);
    ocmoff_t pos = 0;

    f = xfopen(fileName, 'w', (eType == 'T') ? 't' : 'b');

    while (pos < end) {
      BufferChunk bc = buffer_getChunk(buf, pos, end - pos);

      if (fwrite(bc.ptr, 1, bc.len, f) < bc.len)
	THROW(ExTruncated, format("Could not write all of \"%s\"", fileName));

      pos += bc.len;
    }

    xfclose(f);
  }
  DEFAULT(ex) {
    xfclose(f);
    RETHROW(ex);
  }
  END_CATCH;

}

int
buffer_getc(const Buffer *buf, ocmoff_t pos)
{
  pos += buf->bias;

  if (pos >= buf->end)
    THROW(ExOverrun, "Buffer size exceeded");


  {
    unsigned char *block = (unsigned char *) vec_fetch(buf->vec, BLKNDX(pos));
    size_t blkoff = BLKOFF(pos);

    return block[blkoff];
  }
}

void 
buffer_read(const Buffer *buf, void *vp, ocmoff_t pos, ocmoff_t len)
{
  ocmoff_t end = pos + len;
  unsigned char *bp = vp;

  while (pos < end) {
    BufferChunk bc = buffer_getChunk(buf, pos, end - pos);
    assert(bc.len <= (end - pos));

    memcpy(bp, bc.ptr, bc.len);

    bp += bc.len;
    pos += bc.len;
  }
}

/* Note that buffer_asString ALWAYS NUL-terminates whatever it
   returns. If you wanted a binary bytestring, the extra NUL won't
   really hurt you. If you needed a NUL, though, it's absence could
   prove quite irritating. */
char *
buffer_asString(const Buffer *buf)
{
  ocmoff_t len = buffer_length(buf);
  char *s = GC_MALLOC_ATOMIC(len+1);
  s[len] = 0;

  buffer_read(buf, s, 0, len);

  return s;
}

Buffer *
buffer_compress(Buffer *inBuf, int level)
{
  char buffer[BUFFER_BLOCK_SIZE];
  z_stream zstrm;
  ocmoff_t pos = 0;
  ocmoff_t end = buffer_length(inBuf);
  Buffer *outBuf = buffer_create();

  memset(&zstrm, 0, sizeof(zstrm));

  zstrm.zalloc = Z_NULL;
  zstrm.zfree = Z_NULL;
  zstrm.opaque = Z_NULL;
  zstrm.total_out= 0;
  zstrm.total_in= 0;
  zstrm.data_type = Z_BINARY;

  zstrm.next_out = buffer;
  zstrm.avail_out = BUFFER_BLOCK_SIZE;

  zstrm.avail_in = 0;

  deflateInit(&zstrm, level);

  {
    int result = Z_OK;

    while (pos < end) {

      if (zstrm.avail_in == 0) {
	BufferChunk bc = buffer_getChunk(inBuf, pos, end - pos);
	assert(bc.len <= (end - pos));

	zstrm.next_in = (char *) bc.ptr;
	zstrm.avail_in = bc.len;

	pos += bc.len;
      }

      result = deflate(&zstrm, Z_NO_FLUSH);
      if (result < 0)
	THROW(ExBadValue, format("Deflate failed (error %d)", result));

      if (zstrm.avail_out < BUFFER_BLOCK_SIZE) {
	buffer_append(outBuf, buffer, BUFFER_BLOCK_SIZE - zstrm.avail_out);
	zstrm.next_out = buffer;
	zstrm.avail_out = BUFFER_BLOCK_SIZE;
      }
    }

    /* Finish deflating: */
    while (result != Z_STREAM_END) {
      result = deflate(&zstrm, Z_FINISH);
      if (result < 0)
	THROW(ExBadValue, format("Deflate failed (error %d)", result));

      if (zstrm.avail_out < BUFFER_BLOCK_SIZE) {
	buffer_append(outBuf, buffer, BUFFER_BLOCK_SIZE - zstrm.avail_out);
	zstrm.next_out = buffer;
	zstrm.avail_out = BUFFER_BLOCK_SIZE;
      }
    }

    deflateEnd(&zstrm);
  }

  return outBuf;
}

Buffer *
buffer_decompress(Buffer *inBuf)
{
  char buffer[BUFFER_BLOCK_SIZE];
  z_stream zstrm;
  ocmoff_t pos = 0;
  ocmoff_t end = buffer_length(inBuf);
  Buffer *outBuf = buffer_create();

  memset(&zstrm, 0, sizeof(zstrm));

  zstrm.zalloc = Z_NULL;
  zstrm.zfree = Z_NULL;
  zstrm.opaque = Z_NULL;
  zstrm.total_out= 0;
  zstrm.total_in= 0;
  zstrm.data_type = Z_BINARY;

  zstrm.next_out = buffer;
  zstrm.avail_out = BUFFER_BLOCK_SIZE;

  zstrm.avail_in = 0;

  inflateInit(&zstrm);

  {
    int result = Z_OK;

    while (pos < end) {
      if (zstrm.avail_in == 0) {
	BufferChunk bc = buffer_getChunk(inBuf, pos, end - pos);
	assert(bc.len <= (end - pos));

	zstrm.next_in = (char *) bc.ptr;
	zstrm.avail_in = bc.len;

	pos += bc.len;
      }

      result = inflate(&zstrm, Z_NO_FLUSH);
      if (result < 0)
	THROW(ExBadValue, format("Inflate failed (error %d)", result));

      if (zstrm.avail_out < BUFFER_BLOCK_SIZE) {
	buffer_append(outBuf, buffer, BUFFER_BLOCK_SIZE - zstrm.avail_out);
	zstrm.next_out = buffer;
	zstrm.avail_out = BUFFER_BLOCK_SIZE;
      }
    }

    /* Finish deflating: */
    while (result != Z_STREAM_END) {
      result = inflate(&zstrm, Z_NO_FLUSH);
      if (result < 0)
	THROW(ExBadValue, format("Inflate failed (error %d)", result));

      if (zstrm.avail_out < BUFFER_BLOCK_SIZE) {
	buffer_append(outBuf, buffer, BUFFER_BLOCK_SIZE - zstrm.avail_out);
	zstrm.next_out = buffer;
	zstrm.avail_out = BUFFER_BLOCK_SIZE;
      }
    }

    inflateEnd(&zstrm);
  }

  return outBuf;
}
