/* $Id: proclist.c,v 1.4 2002/08/30 18:58:00 nino Exp $ */

/* Copyright (C) 2002 Nils Nordman <nino@nforced.com>

   LibGTop is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License,
   or (at your option) any later version.

   LibGTop is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with LibGTop; see the file COPYING. If not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <glibtop.h>
#include <glibtop/error.h>
#include <glibtop/proclist.h>
#include <glibtop/xmalloc.h>

#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/proc.h>

static const unsigned long _glibtop_sysdeps_proclist =
(1L << GLIBTOP_PROCLIST_TOTAL) +
(1L << GLIBTOP_PROCLIST_NUMBER) +
(1L << GLIBTOP_PROCLIST_SIZE);

/* Fetch list of currently running processes.
 * The interface of this function is a little bit different from the others:
 * buf->flags is only set if the call succeeded, in this case pids_chain,
 * a list of the pids of all currently running processes is returned,
 * buf->number is the number of elements of this list and buf->size is
 * the size of one single element (sizeof (unsigned)). The total size is
 * stored in buf->total.
 *
 * The calling function has to free the memory to which a pointer is returned.
 *
 * IMPORTANT NOTE:
 *   On error, this function MUST return NULL and set buf->flags to zero !
 *   On success, it returns a pointer to a list of buf->number elements
 *   each buf->size big. The total size is stored in buf->total.
 * The calling function has to free the memory to which a pointer is returned.
 *
 * On error, NULL is returned and buf->flags is zero. */

/* Init function. */

void
glibtop_init_proclist_s (glibtop *server)
{
	server->sysdeps.proclist = _glibtop_sysdeps_proclist;
}

unsigned *
glibtop_get_proclist_s (glibtop *server, glibtop_proclist *buf,
			int64_t which, int64_t arg)
{
	/* XXX - Also include kernel threads if not masked out? or not? gaah */
	int proc_mib[4];
	size_t mib_size;
	size_t proc_size;
	struct kinfo_proc *p;
	int real_which;
	int i, j;
	unsigned *pids;

	glibtop_init_s(&server, (1L << GLIBTOP_SYSDEPS_PROCLIST), 0);
	memset (buf, 0, sizeof (glibtop_proclist));

	real_which = (int)(which & GLIBTOP_KERN_PROC_MASK);
	proc_mib[0] = CTL_KERN;
	proc_mib[1] = KERN_PROC;
	mib_size = 4; /* will be 4 for all but KERN_PROC_ALL */

	/* 
	   Bleah. Now we have to go through and remap each and everyone of
	   these. We _can't_ rely on any correspondance with the values as they
	   currently are in the system headers. Also, we can't specify all flags
	   to sysctl initially so some of the flags have to be enforced after
	   fetching the procs.
	*/
	switch(real_which) {
	      case GLIBTOP_KERN_PROC_ALL:
		proc_mib[2] = KERN_PROC_ALL;
		mib_size = 3;
		break;
	      case GLIBTOP_KERN_PROC_PID:
		proc_mib[2] = KERN_PROC_PID;
		proc_mib[3] = (int)arg;
		break;
	      case GLIBTOP_KERN_PROC_PGRP:
		proc_mib[2] = KERN_PROC_PGRP;
		proc_mib[3] = (int)arg;
		break;
	      case GLIBTOP_KERN_PROC_SESSION:
		proc_mib[2] = KERN_PROC_SESSION;
		proc_mib[3] = (int)arg;
		break;
	      case GLIBTOP_KERN_PROC_TTY:
		proc_mib[2] = KERN_PROC_TTY;
		proc_mib[3] = (int)arg;
		break;
	      case GLIBTOP_KERN_PROC_UID:
		proc_mib[2] = KERN_PROC_UID;
		proc_mib[3] = (int)arg;
		break;
	      case GLIBTOP_KERN_PROC_RUID:
		proc_mib[2] = KERN_PROC_RUID;
		proc_mib[3] = (int)arg;
		break;
	      default: /* they want all */
		proc_mib[2] = KERN_PROC_ALL;
		mib_size = 3;
		break;
	}

	/* get size of procs */
	if (sysctl(proc_mib, mib_size, NULL, &proc_size, NULL, 0) == -1) {
	        glibtop_warn_io_r (server, "sysctl kern.kern_proc");
                return NULL;
	}
	
	if( !(p = glibtop_malloc_r( server, proc_size ) ) )
                return NULL;

	if( !(pids = glibtop_malloc_r( server, proc_size / sizeof(struct kinfo_proc) ) ) )
                return NULL;

	/* fetch the procs */
	if (sysctl(proc_mib, mib_size, p, &proc_size, NULL, 0) == -1) {
	        glibtop_warn_io_r(server, "sysctl kern.kern_proc");
                return NULL;
	}

	buf->number = proc_size / sizeof(struct kinfo_proc);

	/* build pid chain */
	for( i = j = 0; i < buf->number; ++i ) {
		if( which & GLIBTOP_EXCLUDE_IDLE ) {
			if( p[i].kp_proc.p_stat != SRUN )
				continue;
		}
/*  		if( which & GLIBTOP_EXCLUDE_SYSTEM ) { */
/*  			if( p[i].kp_proc.p_stat != SRUN ) */
/*  				continue; */
/*  		} */
		if( which & GLIBTOP_EXCLUDE_NOTTY ) {
			if( !(p[i].kp_eproc.e_tdev) )
				continue;
		}
		pids[j++] = p[i].kp_proc.p_pid;
	}
	buf->size = sizeof(unsigned);
	buf->total = buf->number * buf->size;
	buf->flags = _glibtop_sysdeps_proclist;
	return pids;
}
