/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include <glib/gi18n.h>
#include <glade/glade.h>

#include "ogmrip-progress.h"
#include "ogmrip-helper.h"

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-progress.glade"

typedef struct
{
  OGMJobSpawn *spawn;
  GtkWidget *dialog;

  GtkWidget *title_label;
  GtkWidget *stage_label;
  GtkWidget *eta_label;

  GtkWidget *stage_progress;
  GtkWidget *total_progress;

  guint stage;
  guint completed;
  gulong start_time;
} DialogData;

static void
ogmrip_progress_responsed (DialogData *data, gint response_id, GtkWidget *dialog)
{
  if (response_id != GTK_RESPONSE_NONE)
  {
    response_id = ogmrip_message_dialog (GTK_WINDOW (dialog), GTK_MESSAGE_QUESTION,
        _("Are you sure you want to cancel the encoding process?"));

    if (response_id == GTK_RESPONSE_YES)
      ogmjob_spawn_cancel (data->spawn);
  }
}

static void
ogmrip_progress_destroyed (DialogData *data)
{
  if (data->spawn)
    g_object_unref (data->spawn);
  data->spawn = NULL;

  g_free (data);
}

static void
ogmrip_progress_run (DialogData *data)
{
  GTimeVal tv;

  g_get_current_time (&tv);
  data->start_time = tv.tv_sec;
}

static void
ogmrip_progress_progress (DialogData *data, gdouble fraction)
{
  GTimeVal tv;
  gulong eta;
  gchar *str;

  if (!data->start_time)
  {
    g_get_current_time (&tv);
    data->start_time = tv.tv_sec;
  }

  gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (data->stage_progress), fraction);

  g_get_current_time (&tv);
  eta = (1.0 - fraction) * (tv.tv_sec - data->start_time) / fraction;

  if (eta >= 3600)
    str = g_strdup_printf ("%ld hour(s) %ld minute(s)", eta / 3600, (eta % 3600) / 60);
  else
    str = g_strdup_printf ("%ld minute(s)", eta / 60);

  gtk_label_set_text (GTK_LABEL (data->eta_label), str);
  g_free (str);

  fraction = (data->completed + fraction) / (gdouble) data->stage;
  gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (data->total_progress), CLAMP (fraction, 0.0, 1.0));
}

GtkWidget *
ogmrip_progress_new (const gchar *title, guint stage)
{
  DialogData *data;
  GladeXML *xml;

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, NULL, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return NULL;
  }

  data = g_new0 (DialogData, 1);
  data->completed = -1;
  data->stage = stage;

  data->dialog = glade_xml_get_widget (xml, "progress-dialog");
  g_signal_connect (data->dialog, "delete-event", G_CALLBACK (gtk_true), NULL);
  g_signal_connect_swapped (data->dialog, "response", G_CALLBACK (ogmrip_progress_responsed), data);
  g_signal_connect_swapped (data->dialog, "destroy", G_CALLBACK (ogmrip_progress_destroyed), data);

  gtk_window_set_icon_from_stock (GTK_WINDOW (data->dialog), GTK_STOCK_EXECUTE);
  g_object_set_data (G_OBJECT (data->dialog), "__dialog_data__", data);

  data->title_label = glade_xml_get_widget (xml, "title-label");
  gtk_label_set_text (GTK_LABEL (data->title_label), title);

  data->stage_label = glade_xml_get_widget (xml, "stage-label");
  data->eta_label = glade_xml_get_widget (xml, "eta-label");

  data->stage_progress = glade_xml_get_widget (xml, "stage-progress");
  data->total_progress = glade_xml_get_widget (xml, "total-progress");

  g_object_set_data (G_OBJECT (data->dialog), "__ogmrip_progress__", data);

  g_object_unref (xml);

  return data->dialog;
}

void
ogmrip_progress_set_spawn (GtkWidget *dialog, OGMJobSpawn *spawn, const gchar *title)
{
  DialogData *data;

  if (dialog)
  {
    data = g_object_get_data (G_OBJECT (dialog), "__dialog_data__");
    data->start_time = 0;
    data->completed ++;

    if (spawn != data->spawn)
    {
      g_object_ref (spawn);
      if (data->spawn)
        g_object_unref (data->spawn);
      data->spawn = spawn;

      g_signal_connect_swapped (spawn, "run", G_CALLBACK (ogmrip_progress_run), data);
      g_signal_connect_swapped (spawn, "progress", G_CALLBACK (ogmrip_progress_progress), data);
    }

    if (title)
      gtk_label_set_markup (GTK_LABEL (data->stage_label), title);

    gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (data->stage_progress), 0.0);
  }
}

