/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include "ogmrip-helper.h"

#include <stdlib.h>

gint
gconf_client_get_int_default (GConfClient *client, const gchar *key, gint def)
{
  GConfValue* val;

  val = gconf_client_get (client, key, NULL);
  if (!val)
    gconf_client_set_int (client, key, def, NULL);
  else
  {
    def = gconf_value_get_int (val);
    gconf_value_free (val);
  }

  return def;
}

gboolean
gconf_client_get_bool_default (GConfClient *client, const gchar *key, gboolean def)
{
  GConfValue* val;

  val = gconf_client_get (client, key, NULL);
  if (!val)
    gconf_client_set_bool (client, key, def, NULL);
  else
  {
    def = gconf_value_get_bool (val);
    gconf_value_free (val);
  }

  return def;
}

gchar *
gconf_client_get_string_default (GConfClient *client, const gchar *key, const gchar *def)
{
  GConfValue* val;
  gchar *str;

  val = gconf_client_get (client, key, NULL);
  if (!val)
  {
    gconf_client_set_string (client, key, def, NULL);
    str = g_strdup (def);
  }
  else
  {
    str = g_strdup (gconf_value_get_string (val));
    gconf_value_free (val);
  }

  return str;
}

gchar *
gconf_client_get_filename_default (GConfClient *client, const gchar *key, const gchar *def)
{
  gchar *utf8, *path;

  utf8 = gconf_client_get_string_default (client, key, def);
  path = g_filename_from_utf8 (utf8, -1, NULL, NULL, NULL);
  g_free (utf8);

  return path;
}

void
gtk_window_set_parent (GtkWindow *window, GtkWindow *parent)
{
  g_return_if_fail (GTK_IS_WINDOW (window));
  g_return_if_fail (GTK_IS_WINDOW (parent));
  g_return_if_fail (window != parent);

  gtk_window_set_transient_for (window, parent);
  gtk_window_set_position (window, GTK_WIN_POS_CENTER_ON_PARENT);
  gtk_window_set_gravity (window, GDK_GRAVITY_CENTER);
  gtk_window_set_destroy_with_parent (window, TRUE);
}

void
gtk_window_set_icon_from_stock (GtkWindow *window, const gchar *stock_id)
{
  GdkPixbuf *pixbuf;

  g_return_if_fail (GTK_IS_WINDOW (window));
  g_return_if_fail (stock_id && *stock_id);
  
  pixbuf = gtk_widget_render_icon (GTK_WIDGET (window), stock_id, GTK_ICON_SIZE_DIALOG, NULL);
  gtk_window_set_icon (window, pixbuf);
  gdk_pixbuf_unref (pixbuf);
}

gint
gtk_radio_button_get_active (GtkRadioButton *radio)
{
  GSList *link;
  gint i;

  g_return_val_if_fail (GTK_IS_RADIO_BUTTON (radio), -1);

  link = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));

  for (i = g_slist_length (link) - 1; link; i--, link = link->next)
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (link->data)))
      return i;

  return -1;
}

void
gtk_radio_button_set_active (GtkRadioButton *radio, guint index)
{
  GSList *link;
  guint i;

  g_return_if_fail (GTK_IS_RADIO_BUTTON (radio));

  link = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));
  for (i = g_slist_length (link) - 1;  link; i--, link = link->next)
    if (i == index)
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (link->data), TRUE);
}

gboolean
gtk_tree_model_iter_prev (GtkTreeModel *tree_model, GtkTreeIter *iter)
{
  gboolean retval = FALSE;
  GtkTreePath *path;

  path = gtk_tree_model_get_path (tree_model, iter);
  if (path)
  {
    if (gtk_tree_path_prev (path))
      retval = gtk_tree_model_get_iter (tree_model, iter, path);
    gtk_tree_path_free (path);
  }

  return retval;
}

void
gtk_label_set_int (GtkLabel *label, gint value)
{
  gchar *text;

  g_return_if_fail (GTK_IS_LABEL (label));

  text = g_strdup_printf ("%d", value);
  gtk_label_set_text (label, text);
  g_free (text);
}

gint
gtk_label_get_int (GtkLabel *label)
{
  const gchar *text;

  g_return_val_if_fail (GTK_IS_LABEL (label), G_MININT);

  text = gtk_label_get_text (label);
  
  return atoi (text);
}

GtkWidget *
gtk_box_get_child (GtkBox *box, guint n)
{
  GtkBoxChild *child;

  g_return_val_if_fail (GTK_IS_BOX (box), NULL);

  child = g_list_nth_data (box->children, n);
  if (!child)
    return NULL;

  return child->widget;
}

void
ogmrip_statusbar_push (GtkStatusbar *statusbar, const gchar *text)
{
  guint id;

  g_return_if_fail (GTK_IS_STATUSBAR (statusbar));
  g_return_if_fail (text != NULL);

  id = gtk_statusbar_get_context_id (statusbar, "__menu_hint__");
  gtk_statusbar_push (statusbar, id, text);
}

void
ogmrip_statusbar_pop (GtkStatusbar *statusbar)
{
  guint id;

  g_return_if_fail (GTK_IS_STATUSBAR (statusbar));

  id = gtk_statusbar_get_context_id (statusbar, "__menu_hint__");
  gtk_statusbar_pop (statusbar, id);
}

GQuark
ogmrip_error_quark (void)
{
  static GQuark quark = 0;

  if (quark == 0)
    quark = g_quark_from_static_string ("ogmrip-error-quark");

  return quark;
}

gint
ogmrip_message_dialog (GtkWindow *parent, GtkMessageType type, const gchar *format, ...)
{
  GtkWidget *dialog = NULL;
  gchar *message;
  va_list args;

  g_return_val_if_fail (parent == NULL || GTK_IS_WINDOW (parent), GTK_RESPONSE_NONE);

  va_start (args, format);
  message = g_strdup_vprintf (format, args);
  va_end (args);

  switch (type)
  {
    case GTK_MESSAGE_ERROR:
      dialog = gtk_message_dialog_new (parent, 
          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, type, GTK_BUTTONS_CLOSE, message);
      gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_DIALOG_ERROR);
      break;
    case GTK_MESSAGE_QUESTION:
      dialog = gtk_message_dialog_new (parent, 
          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, type, GTK_BUTTONS_YES_NO, message);
      gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_DIALOG_QUESTION);
      break;
    case GTK_MESSAGE_INFO:
      dialog = gtk_message_dialog_new (parent, 
          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, type, GTK_BUTTONS_CLOSE, message);
      gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_DIALOG_INFO);
      break;
    case GTK_MESSAGE_WARNING:
      dialog = gtk_message_dialog_new (parent, 
          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT, type, GTK_BUTTONS_CLOSE, message);
      gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_DIALOG_WARNING);
      break;
    default:
      break;
  }

  g_free (message);

  if (dialog)
  {
    gint response;

    gtk_window_set_gravity (GTK_WINDOW (dialog), GDK_GRAVITY_CENTER);
    gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ON_PARENT);
    gtk_label_set_use_markup (GTK_LABEL (GTK_MESSAGE_DIALOG (dialog)->label), TRUE);
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);

    return response;
  }

  return GTK_RESPONSE_NONE;
}

