/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include "ogmdvd.h"

#include "ogmrip-crop.h"
#include "ogmrip-helper.h"
#include "ogmrip-fs.h"

#include "ogmjob-enums.h"
#include "ogmjob-exec.h"

#include <unistd.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>
#include <glade/glade.h>

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-crop.glade"

#define SCALE_FACTOR 2 / 3

typedef struct
{
  OGMDvdTitle *title;

  GtkWidget *dialog;

  GtkWidget *left_spin;
  GtkWidget *right_spin;
  GtkWidget *top_spin;
  GtkWidget *bottom_spin;

  GtkWidget *image;
  GtkWidget *scale;
  GtkWidget *label;
  GtkWidget *aspect;

  GdkPixbuf *pixbuf;

  gulong length;
  guint rate_numerator;
  guint rate_denominator;

  guint raw_width;
  guint raw_height;

} DialogData;

static gint
ogmrip_crop_get_chapter (OGMDvdTitle *title, gulong *frame)
{
  guint numerator, denominator;
  gint i, last_chapter;
  gulong length;

  g_return_val_if_fail (title != NULL, -1);

  ogmdvd_title_get_framerate (title, &numerator, &denominator);
  last_chapter = ogmdvd_title_get_n_chapters (title);

  for (i = 0; i < last_chapter; i++)
  {
    length = ogmdvd_title_get_chapters_length (title, i, i, NULL);

    if (length >= *frame)
      return i + 1;

    *frame -= length;
  }

  return i;
}

static gchar **
ogmrip_crop_grab_command (DialogData *data, gulong frame)
{
  GPtrArray *argv;
  const gchar *device;
  gint chap, vid, time_;

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mplayer"));
  g_ptr_array_add (argv, g_strdup ("-nolirc"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-nosound"));
  g_ptr_array_add (argv, g_strdup ("-nozoom"));

  g_ptr_array_add (argv, g_strdup ("-vo"));

#if MPLAYER_PRE >= 6
  g_ptr_array_add (argv, g_strdup_printf ("jpeg:outdir=%s", ogmrip_fs_get_tmp_dir ()));
#else /* MPLAYER_PRE */
  g_ptr_array_add (argv, g_strdup ("jpeg"));
  g_ptr_array_add (argv, g_strdup ("-jpeg"));
  g_ptr_array_add (argv, g_strdup_printf ("outdir=%s", ogmrip_fs_get_tmp_dir ()));
#endif /* MPLAYER_PRE */

#if MPLAYER_PRE >= 8
  g_ptr_array_add (argv, g_strdup ("-vc"));
  g_ptr_array_add (argv, g_strdup ("ffmpeg12"));
  g_ptr_array_add (argv, g_strdup ("-frames"));
  g_ptr_array_add (argv, g_strdup ("1"));
#else /* MPLAYER_PRE */
  g_ptr_array_add (argv, g_strdup ("-frames"));
  g_ptr_array_add (argv, g_strdup ("3"));
  g_ptr_array_add (argv, g_strdup ("-sstep"));
  g_ptr_array_add (argv, g_strdup ("1"));
#endif /* MPLAYER_PRE */

  chap = ogmrip_crop_get_chapter (data->title, &frame);
  g_ptr_array_add (argv, g_strdup ("-chapter"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", chap));

  time_ = (gint) (frame * data->rate_denominator / (gdouble) data->rate_numerator);
  g_ptr_array_add (argv, g_strdup ("-ss"));
  g_ptr_array_add (argv, g_strdup_printf ("%u", time_));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (data->title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (data->title);

#if MPLAYER_MAJOR > 0
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_MAJOR */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_MAJOR */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static void
ogmrip_crop_crop_frame (DialogData *data)
{
  GdkPixbuf *pixbuf;
  gint left, top, right, bottom, w, h;

  left = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->left_spin)) * SCALE_FACTOR;
  top = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->top_spin)) * SCALE_FACTOR;
  right = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->right_spin)) * SCALE_FACTOR;
  bottom = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->bottom_spin)) * SCALE_FACTOR;

  w = gdk_pixbuf_get_width (data->pixbuf)- left - right;
  h = gdk_pixbuf_get_height (data->pixbuf) - top - bottom;

  pixbuf = gdk_pixbuf_new_subpixbuf (data->pixbuf, left, top, w, h);
  gtk_image_set_from_pixbuf (GTK_IMAGE (data->image), pixbuf);
  g_object_unref (pixbuf);
}

static void
ogmrip_crop_grab_frame (DialogData *data, gulong frame)
{
  OGMJobSpawn *spawn;
  gchar **cmd;

  cmd = ogmrip_crop_grab_command (data, frame);

  spawn = ogmjob_exec_newv (cmd);
  if (ogmjob_spawn_run (spawn, NULL) == OGMJOB_RESULT_COMPLETED)
  {
    gchar *filename;

#if MPLAYER_PRE >= 8
    filename = g_build_filename (ogmrip_fs_get_tmp_dir (), "00000001.jpg", NULL);
#else /* MPLAYER_PRE */
    filename = g_build_filename (ogmrip_fs_get_tmp_dir (), "00000001.jpg", NULL);
    g_unlink (filename);
    g_free (filename);

    filename = g_build_filename (ogmrip_fs_get_tmp_dir (), "00000003.jpg", NULL);
    g_unlink (filename);
    g_free (filename);

    filename = g_build_filename (ogmrip_fs_get_tmp_dir (), "00000002.jpg", NULL);
#endif /* MPLAYER_PRE */

    if (g_file_test (filename, G_FILE_TEST_IS_REGULAR))
    {
      if (data->pixbuf)
        g_object_unref (data->pixbuf);
      data->pixbuf = gdk_pixbuf_new_from_file_at_size (filename, 
          data->raw_width * SCALE_FACTOR, data->raw_height * SCALE_FACTOR, NULL);
      g_unlink (filename);
    }
    g_free (filename);

    if (data->pixbuf)
      ogmrip_crop_crop_frame (data);
  }
  g_object_unref (spawn);
}

static void
ogmrip_crop_destroyed (DialogData *data)
{
  if (data->title)
    ogmdvd_title_unref (data->title);
  data->title = NULL;

  if (data->pixbuf)
    g_object_unref (data->pixbuf);
  data->pixbuf = NULL;

  g_free (data);
}

static void
ogmrip_crop_spin_value_changed (DialogData *data)
{
  ogmrip_crop_crop_frame (data);
}

static void
ogmrip_crop_scale_value_changed (DialogData *data, GtkWidget *scale)
{
  gulong frame;
  gchar *text;

  frame = (guint) gtk_range_get_value (GTK_RANGE (scale));
  text = g_strdup_printf (_("Frame %lu of %lu"), frame, data->length);
  gtk_label_set_text (GTK_LABEL (data->label), text);
  g_free (text);

  ogmrip_crop_grab_frame (data, frame);
}

GtkWidget *
ogmrip_crop_new (OGMDvdTitle *title, guint left, guint top, guint right, guint bottom)
{
  DialogData *data;
  GladeXML *xml;

  gint32 frame;
/*
  gchar *text;
*/
  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, NULL, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return NULL;
  }

  data = g_new0 (DialogData, 1);

  ogmdvd_title_ref (title);
  data->title = title;

  ogmdvd_title_get_size (title, &data->raw_width, &data->raw_height);

  data->length = ogmdvd_title_get_length (title, NULL);
  ogmdvd_title_get_framerate (title, &data->rate_numerator, &data->rate_denominator);

  data->dialog = glade_xml_get_widget (xml, "crop-dialog");
  g_object_set_data (G_OBJECT (data->dialog), "__ogmrip_crop__", data);
  gtk_window_set_icon_from_stock (GTK_WINDOW (data->dialog), GTK_STOCK_PROPERTIES);
  g_signal_connect_swapped (data->dialog, "destroy", G_CALLBACK (ogmrip_crop_destroyed), data);
  g_signal_connect_after (data->dialog, "response", G_CALLBACK (gtk_widget_hide), NULL);

  data->left_spin = glade_xml_get_widget (xml, "left-spin");
  gtk_spin_button_set_range (GTK_SPIN_BUTTON (data->left_spin), 0.0, (gdouble) data->raw_width);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->left_spin), (gdouble) left);
  g_signal_connect_swapped (data->left_spin, "value-changed", G_CALLBACK (ogmrip_crop_spin_value_changed), data);

  data->top_spin = glade_xml_get_widget (xml, "top-spin");
  gtk_spin_button_set_range (GTK_SPIN_BUTTON (data->top_spin), 0.0, (gdouble) data->raw_height);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->top_spin), (gdouble) top);
  g_signal_connect_swapped (data->top_spin, "value-changed", G_CALLBACK (ogmrip_crop_spin_value_changed), data);

  data->right_spin = glade_xml_get_widget (xml, "right-spin");
  gtk_spin_button_set_range (GTK_SPIN_BUTTON (data->right_spin), 0.0, (gdouble) data->raw_width);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->right_spin), (gdouble) right);
  g_signal_connect_swapped (data->right_spin, "value-changed", G_CALLBACK (ogmrip_crop_spin_value_changed), data);

  data->bottom_spin = glade_xml_get_widget (xml, "bottom-spin");
  gtk_spin_button_set_range (GTK_SPIN_BUTTON (data->bottom_spin), 0.0, (gdouble) data->raw_height);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->bottom_spin), (gdouble) bottom);
  g_signal_connect_swapped (data->bottom_spin, "value-changed", G_CALLBACK (ogmrip_crop_spin_value_changed), data);

  data->image = glade_xml_get_widget (xml, "frame-image");
  data->label = glade_xml_get_widget (xml, "frame-label");

  data->scale = glade_xml_get_widget (xml, "frame-scale");
  gtk_range_set_range (GTK_RANGE (data->scale), 1.0, data->length);
  gtk_range_set_increments (GTK_RANGE (data->scale), 1.0, data->length / 25);
  gtk_range_set_update_policy (GTK_RANGE (data->scale), GTK_UPDATE_DELAYED);

  g_signal_connect_swapped (data->scale, "value-changed", G_CALLBACK (ogmrip_crop_scale_value_changed), data);

  g_object_unref (xml);

  frame = g_random_int_range (1, data->length);
  gtk_range_set_value (GTK_RANGE (data->scale), frame);

  return data->dialog;
}

void
ogmrip_crop_get_crop (GtkWidget *dialog, guint *left, guint *top, guint *right, guint *bottom)
{
  DialogData *data;

  g_return_if_fail (GTK_IS_DIALOG (dialog));
  g_return_if_fail (left && top && right && bottom);

  data = g_object_get_data (G_OBJECT (dialog), "__ogmrip_crop__");
  g_return_if_fail (data != NULL);

  *left = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->left_spin));
  *top = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->top_spin));
  *right = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->right_spin));
  *bottom = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->bottom_spin));
}

