/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n-lib.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtktooltips.h>

#include "ogmrip-chooser-list.h"
#include "ogmdvd-audio-chooser.h"
#include "ogmdvd-subtitle-chooser.h"

#define OGMRIP_CHOOSER_LIST_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_CHOOSER_LIST, OGMRipChooserListPriv))

struct _OGMRipChooserListPriv
{
  GType child_type;
  gint max;
};

typedef struct _OGMRipChooserListChild OGMRipChooserListChild;

static void ogmrip_chooser_list_show            (GtkWidget    *widget);
static void ogmrip_chooser_list_add_internal    (GtkContainer *container,
                                                 GtkWidget    *widget);
static void ogmrip_chooser_list_remove_internal (GtkContainer *container,
                                                 GtkWidget    *widget);

static GtkWidget *
ogmrip_chooser_list_child_new (GtkWidget *chooser)
{
  GtkWidget *child, *hbox, *button, *image;
  GtkTooltips *tooltips;

  child = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (child), chooser, TRUE, TRUE, 0);
  gtk_widget_show (child);

  tooltips = gtk_tooltips_new ();

  hbox = gtk_hbox_new (TRUE, 0);
  gtk_box_pack_start (GTK_BOX (child), hbox, FALSE, FALSE, 0);
  gtk_widget_show (hbox);

  button = gtk_button_new ();
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);

  gtk_tooltips_set_tip (tooltips, button, _("Add a stream"), NULL);

  image = gtk_image_new_from_stock (GTK_STOCK_ADD, GTK_ICON_SIZE_BUTTON);
  gtk_container_add (GTK_CONTAINER (button), image);
  gtk_widget_show (image);

  button = gtk_button_new ();
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);

  gtk_tooltips_set_tip (tooltips, button, _("Remove the stream"), NULL);

  image = gtk_image_new_from_stock (GTK_STOCK_REMOVE, GTK_ICON_SIZE_BUTTON);
  gtk_container_add (GTK_CONTAINER (button), image);
  gtk_widget_show (image);

  return child;
}

static GtkWidget *
ogmrip_chooser_list_child_get_chooser (GtkWidget *child)
{
  GtkBoxChild *box_child;

  box_child = GTK_BOX (child)->children->data;

  return box_child->widget;
}

static GtkWidget *
ogmrip_chooser_list_child_get_add_button (GtkWidget *child)
{
  GtkBoxChild *box_child;

  box_child = GTK_BOX (child)->children->next->data;
  box_child = GTK_BOX (box_child->widget)->children->data;

  return box_child->widget;
}

static GtkWidget *
ogmrip_chooser_list_child_get_rem_button (GtkWidget *child)
{
  GtkBoxChild *box_child;

  box_child = GTK_BOX (child)->children->next->data;
  box_child = GTK_BOX (box_child->widget)->children->next->data;

  return box_child->widget;
}

G_DEFINE_TYPE (OGMRipChooserList, ogmrip_chooser_list, GTK_TYPE_VBOX);

static void
ogmrip_chooser_list_class_init (OGMRipChooserListClass *klass)
{
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  widget_class = GTK_WIDGET_CLASS (klass);
  container_class = GTK_CONTAINER_CLASS (klass);

  widget_class->show = ogmrip_chooser_list_show;
  container_class->add = ogmrip_chooser_list_add_internal;
  container_class->remove = ogmrip_chooser_list_remove_internal;

  g_type_class_add_private (klass, sizeof (OGMRipChooserListPriv));
}

static void
ogmrip_chooser_list_init (OGMRipChooserList *list)
{
  list->priv = OGMRIP_CHOOSER_LIST_GET_PRIVATE (list);

  list->priv->max = -1;

  gtk_box_set_spacing (GTK_BOX (list), 6);
}

static void
ogmrip_chooser_list_add_clicked (OGMRipChooserList *list, GtkWidget *button)
{
  GtkWidget *child;

  child = g_object_new (list->priv->child_type, NULL);
  gtk_container_add (GTK_CONTAINER (list), child);
  gtk_widget_show (child);
}

static void
ogmrip_chooser_list_remove_clicked (OGMRipChooserList *list, GtkWidget *button)
{
  gtk_container_remove (GTK_CONTAINER (list), button->parent->parent);
}

static void
ogmrip_chooser_list_show (GtkWidget *widget)
{
  if (widget && widget->parent)
    gtk_widget_show (widget->parent);

  (*GTK_WIDGET_CLASS (ogmrip_chooser_list_parent_class)->show) (widget);
}

static void
ogmrip_chooser_list_add_internal (GtkContainer *container, GtkWidget *chooser)
{
  OGMRipChooserList *list;
  guint len;

  list = OGMRIP_CHOOSER_LIST (container);

  if (G_TYPE_CHECK_INSTANCE_TYPE (chooser, list->priv->child_type))
  {
    len = g_list_length (GTK_BOX (container)->children);
    if (list->priv->max < 0 || len < list->priv->max)
    {
      GtkWidget *child, *button;
      GList *last;

      last = g_list_last (GTK_BOX (container)->children);
      if (last)
      {
        GtkBoxChild *box_child;

        box_child = last->data;

        button = ogmrip_chooser_list_child_get_add_button (box_child->widget);
        gtk_widget_hide (button);

        button = ogmrip_chooser_list_child_get_rem_button (box_child->widget);
        gtk_widget_show (button);
      }

      child = ogmrip_chooser_list_child_new (chooser);

      button = ogmrip_chooser_list_child_get_add_button (child);
      g_signal_connect_swapped (button, "clicked", 
          G_CALLBACK (ogmrip_chooser_list_add_clicked), list);
      if (list->priv->max > 0 && len + 1 == list->priv->max)
        gtk_widget_set_sensitive (button, FALSE);

      button = ogmrip_chooser_list_child_get_rem_button (child);
      g_signal_connect_swapped (button, "clicked", 
          G_CALLBACK (ogmrip_chooser_list_remove_clicked), list);

      (*GTK_CONTAINER_CLASS (ogmrip_chooser_list_parent_class)->add) (container, child);
    }
  }
}

static void
ogmrip_chooser_list_remove_internal (GtkContainer *container, GtkWidget *child)
{
  GList *last;

  (*GTK_CONTAINER_CLASS (ogmrip_chooser_list_parent_class)->remove) (container, child);

  last = g_list_last (GTK_BOX (container)->children);
  if (last)
  {
    GtkBoxChild *box_child;
    GtkWidget *button;

    box_child = last->data;

    button = ogmrip_chooser_list_child_get_add_button (box_child->widget);
    gtk_widget_show (button);

    button = ogmrip_chooser_list_child_get_rem_button (box_child->widget);
    gtk_widget_hide (button);
  }
}

GtkWidget *
ogmrip_chooser_list_new (GType type)
{
  OGMRipChooserList *list;

  g_return_val_if_fail (g_type_is_a (type, OGMDVD_TYPE_AUDIO_CHOOSER) ||
      g_type_is_a (type, OGMDVD_TYPE_SUBTITLE_CHOOSER), NULL);

  list = g_object_new (OGMRIP_TYPE_CHOOSER_LIST, NULL);
  list->priv->child_type = type;

  return GTK_WIDGET (list);
}

void
ogmrip_chooser_list_set_max (OGMRipChooserList *list, guint max)
{
  GList *link, *children;
  GtkWidget *button;
  GtkBoxChild *child;
  guint i;

  g_return_if_fail (OGMRIP_IS_CHOOSER_LIST (list));

  list->priv->max = MAX (max, 1);

  children = g_list_copy (GTK_BOX (list)->children);
  for (i = 0, link = children; link; i ++, link = link->next)
  {
    child = link->data;

    if (i >= max)
      gtk_container_remove (GTK_CONTAINER (list), child->widget);
    else
    {
      button = ogmrip_chooser_list_child_get_add_button (child->widget);
      gtk_widget_set_sensitive (button, max < 0 || i < max - 1);
    }
  }
  g_list_free (children);
}

gint
ogmrip_chooser_list_get_max (OGMRipChooserList *list)
{
  g_return_val_if_fail (OGMRIP_IS_CHOOSER_LIST (list), -1);

  return list->priv->max;
}

void
ogmrip_chooser_list_add (OGMRipChooserList *list, GtkWidget *chooser)
{
  g_return_if_fail (OGMRIP_CHOOSER_LIST (list));
  g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (chooser, list->priv->child_type));

  gtk_container_add (GTK_CONTAINER (list), chooser);
}

void
ogmrip_chooser_list_remove (OGMRipChooserList *list, GtkWidget *chooser)
{
  g_return_if_fail (OGMRIP_CHOOSER_LIST (list));
  g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (chooser, list->priv->child_type));

  if (chooser->parent)
    gtk_container_remove (GTK_CONTAINER (list), chooser->parent);
}

void
ogmrip_chooser_list_clear (OGMRipChooserList *list)
{
  GList *link, *children;
  GtkBoxChild *box_child;

  g_return_if_fail (OGMRIP_IS_CHOOSER_LIST (list));

  children = g_list_copy (GTK_BOX (list)->children);
  for (link = children; link; link = link->next)
  {
    box_child = link->data;
    gtk_container_remove (GTK_CONTAINER (list), box_child->widget);
  }
  g_list_free (children);
}

GList *
ogmrip_chooser_list_get_children (OGMRipChooserList *list)
{
  GList *link, *children = NULL;
  GtkBoxChild *box_child;

  g_return_val_if_fail (OGMRIP_IS_CHOOSER_LIST (list), NULL);

  for (link = GTK_BOX (list)->children; link; link = link->next)
  {
    box_child = link->data;
    children = g_list_append (children, ogmrip_chooser_list_child_get_chooser (box_child->widget));
  }

  return children;
}

static GList *
ogmrip_chooser_list_get_audio_streams (OGMRipChooserList *list)
{
  GList *link, *streams = NULL;
  GtkBoxChild *box_child;
  GtkWidget *chooser;

  g_return_val_if_fail (OGMRIP_IS_CHOOSER_LIST (list), NULL);

  for (link = GTK_BOX (list)->children; link; link = link->next)
  {
    box_child = link->data;
    chooser = ogmrip_chooser_list_child_get_chooser (box_child->widget);
    streams = g_list_append (streams, ogmdvd_audio_chooser_get_active (OGMDVD_AUDIO_CHOOSER (chooser)));
  }

  return streams;
}

static GList *
ogmrip_chooser_list_get_subp_streams (OGMRipChooserList *list)
{
  GList *link, *streams = NULL;
  OGMDvdSubpStream *stream;
  GtkBoxChild *box_child;
  GtkWidget *chooser;

  g_return_val_if_fail (OGMRIP_IS_CHOOSER_LIST (list), NULL);

  for (link = GTK_BOX (list)->children; link; link = link->next)
  {
    box_child = link->data;
    chooser = ogmrip_chooser_list_child_get_chooser (box_child->widget);
    stream = ogmdvd_subtitle_chooser_get_active (OGMDVD_SUBTITLE_CHOOSER (chooser));
    if (stream)
      streams = g_list_append (streams, stream);
  }

  return streams;
}

GList *
ogmrip_chooser_list_get_streams (OGMRipChooserList *list)
{
  g_return_val_if_fail (OGMRIP_IS_CHOOSER_LIST (list), NULL);

  if (g_type_is_a (list->priv->child_type, OGMDVD_TYPE_AUDIO_CHOOSER))
    return ogmrip_chooser_list_get_audio_streams (list);
  else
    return ogmrip_chooser_list_get_subp_streams (list);
}

