/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "ogmrip-xml.h"

#include "ogmrip-subp.h"
#include "ogmrip-audio.h"
#include "ogmrip-video.h"
#include "ogmrip-container.h"

#include <string.h>
#include <libxml/tree.h>
#include <libxml/xmlschemas.h>

#define OGMRIP_XSD_FILE "ogmrip/ogmrip.xsd"

#define OGMRIP_XML_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_XML, OGMRipXmlPriv))

struct _OGMRipXmlPriv
{
  GSList *video_codecs;
  GSList *audio_codecs;
  GSList *subp_codecs;
  GSList *containers;
};

typedef struct
{
  GType   type;
  xmlChar *name;
  xmlChar *description;
} OGMRipXmlCodec;

typedef struct
{
  OGMRipXmlCodec codec;
  gint passes;
} OGMRipXmlVideoCodec;

typedef OGMRipXmlCodec OGMRipXmlAudioCodec;

typedef struct
{
  OGMRipXmlCodec codec;
  gboolean text;
} OGMRipXmlSubpCodec;

typedef struct
{
  GType   type;
  xmlChar *name;
  xmlChar *description;
  gint    max_audio;
  gint    max_subp;
  GSList  *codecs;
} OGMRipXmlContainer;

static void ogmrip_xml_finalize (GObject *gobject);

G_DEFINE_TYPE (OGMRipXml, ogmrip_xml, G_TYPE_OBJECT)

static void
ogmrip_xml_class_init (OGMRipXmlClass *klass)
{
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->finalize = ogmrip_xml_finalize;

  g_type_class_add_private (klass, sizeof (OGMRipXmlPriv));
}

static void
ogmrip_xml_init (OGMRipXml *xml)
{
  xml->priv = OGMRIP_XML_GET_PRIVATE (xml);
}

static void
ogmrip_xml_free_codec (OGMRipXmlCodec *codec)
{
  g_free (codec->name);
  g_free (codec->description);
}

static void
ogmrip_xml_free_container (OGMRipXmlContainer *container)
{
  g_free (container->name);
  g_free (container->description);
  g_slist_free (container->codecs);
  g_free (container);
}

static void
ogmrip_xml_finalize (GObject *gobject)
{
  OGMRipXml *xml;

  xml = OGMRIP_XML (gobject);

  g_slist_foreach (xml->priv->containers, 
      (GFunc) ogmrip_xml_free_container, NULL);
  g_slist_free (xml->priv->containers);

  g_slist_foreach (xml->priv->video_codecs, 
      (GFunc) ogmrip_xml_free_codec, NULL);
  g_slist_free (xml->priv->video_codecs);

  g_slist_foreach (xml->priv->audio_codecs, 
      (GFunc) ogmrip_xml_free_codec, NULL);
  g_slist_free (xml->priv->audio_codecs);

  g_slist_foreach (xml->priv->subp_codecs, 
      (GFunc) ogmrip_xml_free_codec, NULL);
  g_slist_free (xml->priv->subp_codecs);

  G_OBJECT_CLASS (ogmrip_xml_parent_class)->finalize (gobject);
}

static xmlChar *
ogmrip_xml_get_description (GHashTable *hash)
{
  static const gchar * const *languages;

  if (!languages)
    languages = g_get_language_names ();

  if (g_hash_table_size (hash) > 0)
  {
    xmlChar *description;
    guint i;

    for (i = 0; languages[i]; i++)
    {
      description = g_hash_table_lookup(hash, languages[i]);
      if (description)
        return xmlStrdup (description);
    }
  }

  return NULL;
}

static OGMRipXmlCodec *
ogmrip_xml_get_codec_by_name (OGMRipXml *xml, const xmlChar *name)
{
  OGMRipXmlCodec *codec;
  GSList *link;

  for (link = xml->priv->video_codecs; link; link = link->next)
  {
    codec = link->data;
    if (xmlStrEqual (codec->name, name))
      return codec;
  }

  for (link = xml->priv->audio_codecs; link; link = link->next)
  {
    codec = link->data;
    if (xmlStrEqual (codec->name, name))
      return codec;
  }

  for (link = xml->priv->subp_codecs; link; link = link->next)
  {
    codec = link->data;
    if (xmlStrEqual (codec->name, name))
      return codec;
  }

  return NULL;
}

static void
ogmrip_xml_parse_contain (OGMRipXml *xml, OGMRipXmlContainer *container, xmlNode *node)
{
  OGMRipXmlCodec *codec;
  xmlChar *name;

  name = xmlGetProp (node, (xmlChar *) "name");
  codec = ogmrip_xml_get_codec_by_name (xml, name);
  xmlFree (name);

  container->codecs = g_slist_append (container->codecs, codec);
}

static void
ogmrip_xml_parse_contains (OGMRipXml *xml, OGMRipXmlContainer *container, xmlNode *node)
{
  xmlNode *child;

  for (child = node->children; child; child = child->next)
    if (xmlStrEqual (child->name, (xmlChar *) "codec"))
      ogmrip_xml_parse_contain (xml, container, child);
}

static void
ogmrip_xml_parse_codec (OGMRipXmlCodec *codec, xmlNode *node)
{
  GHashTable *hash;
  xmlChar *str, *lang;
  xmlNode *child;

  str = xmlGetProp (node, (xmlChar *) "type");
  codec->type = g_type_from_name ((gchar *) str);
  xmlFree (str);

  codec->name = xmlGetProp (node, (xmlChar *) "name");

  hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);

  for (child = node->children; child; child = child->next)
  {
    if (xmlStrEqual (child->name, (xmlChar *) "description"))
    {
      lang = xmlNodeGetLang (child);
      if (!lang)
        lang = xmlStrdup ((xmlChar *) "C");

      if (g_hash_table_lookup(hash, lang))
        xmlFree (lang);
      else
      {
        str = xmlNodeGetContent (child);
        g_hash_table_insert(hash, lang, str);
      }
    }
  }

  codec->description = ogmrip_xml_get_description (hash);

  g_hash_table_destroy (hash);
}

static void
ogmrip_xml_parse_video_codec (OGMRipXml *xml, xmlNode *node)
{
  OGMRipXmlVideoCodec *codec;

  codec = g_new0 (OGMRipXmlVideoCodec, 1);
  xml->priv->video_codecs = g_slist_append (xml->priv->video_codecs, codec);

  ogmrip_xml_parse_codec ((OGMRipXmlCodec *) codec, node);
}

static void
ogmrip_xml_parse_audio_codec (OGMRipXml *xml, xmlNode *node)
{
  OGMRipXmlAudioCodec *codec;

  codec = g_new0 (OGMRipXmlAudioCodec, 1);
  xml->priv->audio_codecs = g_slist_append (xml->priv->audio_codecs, codec);

  ogmrip_xml_parse_codec ((OGMRipXmlCodec *) codec, node);
}

static void
ogmrip_xml_parse_subp_codec (OGMRipXml *xml, xmlNode *node)
{
  OGMRipXmlSubpCodec *codec;
  xmlChar *str;

  codec = g_new0 (OGMRipXmlSubpCodec, 1);
  xml->priv->subp_codecs = g_slist_append (xml->priv->subp_codecs, codec);

  str = xmlGetProp (node, (xmlChar *) "text");
  codec->text = xmlStrEqual (str, (xmlChar *) "true");
  xmlFree (str);

  ogmrip_xml_parse_codec ((OGMRipXmlCodec *) codec, node);
}

static void
ogmrip_xml_parse_container (OGMRipXml *xml, xmlNode *node)
{
  GHashTable *hash;
  OGMRipXmlContainer *container;
  xmlChar *str, *lang;
  xmlNode *child;

  container = g_new0 (OGMRipXmlContainer, 1);

  str = xmlGetProp (node, (xmlChar *) "type");
  container->type = g_type_from_name ((gchar *) str);
  xmlFree (str);

  container->name = xmlGetProp (node, (xmlChar *) "name");

  xml->priv->containers = g_slist_append (xml->priv->containers, container);

  hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);

  for (child = node->children; child; child = child->next)
  {
    if (xmlStrEqual (child->name, (xmlChar *) "description"))
    {
      lang = xmlNodeGetLang (child);
      if (!lang)
        lang = xmlStrdup ((xmlChar *) "C");

      if (g_hash_table_lookup(hash, lang))
        xmlFree (lang);
      else
      {
        str = xmlNodeGetContent (child);
        g_hash_table_insert(hash, lang, str);
      }
    }
    else if (xmlStrEqual (child->name, (xmlChar *) "contains"))
    {
      str = xmlGetProp (child, (xmlChar *) "audio");
      if (xmlStrEqual (str, (xmlChar *) "unbounded"))
        container->max_audio = G_MAXINT;
      else
        container->max_audio = atoi ((gchar *) str);
      xmlFree (str);

      str = xmlGetProp (child, (xmlChar *) "subp");
      if (xmlStrEqual (str, (xmlChar *) "unbounded"))
        container->max_subp = G_MAXINT;
      else
        container->max_subp = atoi ((gchar *) str);
      xmlFree (str);

      ogmrip_xml_parse_contains (xml, container, child);
    }
  }

  container->description = ogmrip_xml_get_description (hash);

  g_hash_table_destroy (hash);
}

static void
ogmrip_xml_parse_video_codecs (OGMRipXml *xml, xmlNode *node)
{
  xmlNode *child;

  for (child = node->children; child; child = child->next)
    if (xmlStrEqual (child->name, (xmlChar *) "video-codec"))
      ogmrip_xml_parse_video_codec (xml, child);
}

static void
ogmrip_xml_parse_audio_codecs (OGMRipXml *xml, xmlNode *node)
{
  xmlNode *child;

  for (child = node->children; child; child = child->next)
    if (xmlStrEqual (child->name, (xmlChar *) "audio-codec"))
      ogmrip_xml_parse_audio_codec (xml, child);
}

static void
ogmrip_xml_parse_subp_codecs (OGMRipXml *xml, xmlNode *node)
{
  xmlNode *child;

  for (child = node->children; child; child = child->next)
    if (xmlStrEqual (child->name, (xmlChar *) "subp-codec"))
      ogmrip_xml_parse_subp_codec (xml, child);
}

static void
ogmrip_xml_parse_containers (OGMRipXml *xml, xmlNode *node)
{
  xmlNode *child;

  for (child = node->children; child; child = child->next)
    if (xmlStrEqual (child->name, (xmlChar *) "container"))
      ogmrip_xml_parse_container (xml, child);
}

static void
ogmrip_xml_parse_doc (OGMRipXml *xml, xmlDoc *doc)
{
  xmlNode *root, *child;

  root = xmlDocGetRootElement (doc);
  for (child = root->children; child; child = child->next)
  {
    if (xmlStrEqual (child->name, (xmlChar *) "video-codecs"))
      ogmrip_xml_parse_video_codecs (xml, child);
    if (xmlStrEqual (child->name, (xmlChar *) "audio-codecs"))
      ogmrip_xml_parse_audio_codecs (xml, child);
    if (xmlStrEqual (child->name, (xmlChar *) "subp-codecs"))
      ogmrip_xml_parse_subp_codecs (xml, child);
    else if (xmlStrEqual (child->name, (xmlChar *) "containers"))
      ogmrip_xml_parse_containers(xml, child);
  }
}

static gboolean
ogmrip_xml_validate_doc (xmlDoc *doc)
{
  xmlSchemaParserCtxt *ctxt;
  gboolean result = FALSE;

  ctxt = xmlSchemaNewParserCtxt (OGMRIP_DATA_DIR "/" OGMRIP_XSD_FILE);
  if (ctxt)
  {
    xmlSchema *schema;

    xmlSchemaSetParserErrors (ctxt, (xmlSchemaValidityErrorFunc) fprintf, 
        (xmlSchemaValidityWarningFunc) fprintf, stderr);

    schema = xmlSchemaParse (ctxt);
    if (schema)
    {
      xmlSchemaValidCtxtPtr valid;

      valid = xmlSchemaNewValidCtxt (schema);
      if (valid)
      {
        xmlSchemaSetValidErrors (valid, (xmlSchemaValidityErrorFunc) fprintf,
            (xmlSchemaValidityWarningFunc) fprintf, stderr);

        result = xmlSchemaValidateDoc (valid, doc) == 0;
        xmlSchemaFreeValidCtxt (valid);
      }
      xmlSchemaFree (schema);
    }
    xmlSchemaFreeParserCtxt (ctxt);
  }

  return result;
}

OGMRipXml *
ogmrip_xml_new (const gchar *filename)
{
  OGMRipXml *xml = NULL;
  xmlDoc *doc;

  g_return_val_if_fail (filename != NULL, NULL);
  g_return_val_if_fail (g_file_test (filename, G_FILE_TEST_IS_REGULAR), NULL);

  doc = xmlParseFile (filename);
  if (doc)
  {
    if (ogmrip_xml_validate_doc (doc))
    {
      xml = g_object_new (OGMRIP_TYPE_XML, NULL);
      ogmrip_xml_parse_doc (xml, doc);
    }
    xmlFreeDoc (doc);
  }

  return xml;
}

static OGMRipXmlContainer *
ogmrip_xml_find_container_by_type (GSList *list, GType type)
{
  OGMRipXmlContainer *container;

  while (list)
  {
    container = (OGMRipXmlContainer *) list->data;
    if (container->type == type)
      return container;
    list = list->next;
  }

  return NULL;
}

static GType
ogmrip_xml_get_nth_codec (GSList *list, guint n)
{
  OGMRipXmlCodec *codec;

  if (!list)
    return G_TYPE_NONE;

  codec = g_slist_nth_data (list, n);
  if (!codec)
    codec = list->data;

  return codec->type;
}

static void
ogmrip_xml_foreach_codec (GSList *list, OGMRipXmlFunc func, gpointer data)
{
  OGMRipXmlCodec *codec;

  while (list)
  {
    codec = list->data;
    func (codec->type, (const gchar *) codec->name, (const gchar *) codec->description, data);
    list = list->next;
  }
}

static OGMRipXmlCodec *
ogmrip_xml_find_codec_by_type (GSList *list, GType type)
{
  OGMRipXmlCodec *codec;

  while (list)
  {
    codec = list->data;
    if (codec->type == type)
      return codec;
    list = list->next;
  }

  return NULL;
}

static G_CONST_RETURN gchar *
ogmrip_xml_get_codec_name (GSList *list, GType type)
{
  OGMRipXmlCodec *codec;

  codec = ogmrip_xml_find_codec_by_type (list, type);
  if (codec)
    return (gchar *) codec->name;

  return NULL;
}

static GType
ogmrip_xml_find_codec (GSList *list, OGMRipXmlCmpFunc func, gconstpointer data)
{
  OGMRipXmlCodec *codec;

  while (list)
  {
    codec = list->data;
    if (func (codec->type, (gchar *) codec->name, (gchar *) codec->description, data) == 0)
      return codec->type;
    list = list->next;
  }

  return G_TYPE_NONE;
}

gint
ogmrip_xml_get_n_containers (OGMRipXml *xml)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), -1);

  return g_slist_length (xml->priv->containers);
}

void
ogmrip_xml_foreach_container (OGMRipXml *xml, OGMRipXmlFunc func, gpointer data)
{
  OGMRipXmlContainer *container;
  GSList *link;

  g_return_if_fail (OGMRIP_IS_XML (xml));
  g_return_if_fail (func != NULL);

  for (link = xml->priv->containers; link; link = link->next)
  {
    container = link->data;
    func (container->type, (const gchar *) container->name, (const gchar *) container->description, data);
  }
}

GType
ogmrip_xml_find_container (OGMRipXml *xml, OGMRipXmlCmpFunc func, gconstpointer data)
{
  OGMRipXmlContainer *container;
  GSList *link;

  g_return_val_if_fail (OGMRIP_IS_XML (xml), G_TYPE_NONE);
  g_return_val_if_fail (func != NULL, G_TYPE_NONE);

  for (link = xml->priv->containers; link; link = link->next)
  {
    container = link->data;
    if (func (container->type, (gchar *) container->name, (gchar *) container->description, data) == 0)
      return container->type;
  }

  return G_TYPE_NONE;
}

GType
ogmrip_xml_get_nth_container (OGMRipXml *xml, guint n)
{
  OGMRipXmlContainer *container;

  g_return_val_if_fail (OGMRIP_IS_XML (xml), G_TYPE_NONE);

  if (!xml->priv->containers)
    return G_TYPE_NONE;

  container = g_slist_nth_data (xml->priv->containers, n);
  if (!container)
    container = xml->priv->containers->data;

  return container->type;
}

G_CONST_RETURN gchar *
ogmrip_xml_get_container_name (OGMRipXml *xml, GType container)
{
  OGMRipXmlContainer *cont;

  g_return_val_if_fail (OGMRIP_IS_XML (xml), NULL);
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), NULL);

  cont = ogmrip_xml_find_container_by_type (xml->priv->containers, container);
  if (cont)
    return (gchar *) cont->name;

  return NULL;
}

gint
ogmrip_xml_get_n_video_codecs (OGMRipXml *xml)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), -1);

  return g_slist_length (xml->priv->video_codecs);
}

void
ogmrip_xml_foreach_video_codec (OGMRipXml *xml, OGMRipXmlFunc func, gpointer data)
{
  g_return_if_fail (OGMRIP_IS_XML (xml));
  g_return_if_fail (func != NULL);

  ogmrip_xml_foreach_codec (xml->priv->video_codecs, func, data);
}

GType
ogmrip_xml_find_video_codec (OGMRipXml *xml, OGMRipXmlCmpFunc func, gconstpointer data)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), G_TYPE_NONE);
  g_return_val_if_fail (func != NULL, G_TYPE_NONE);

  return ogmrip_xml_find_codec (xml->priv->video_codecs, func, data);
}

GType
ogmrip_xml_get_nth_video_codec (OGMRipXml *xml, guint n)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), G_TYPE_NONE);

  return ogmrip_xml_get_nth_codec (xml->priv->video_codecs, n);
}

G_CONST_RETURN gchar *
ogmrip_xml_get_video_codec_name (OGMRipXml *xml, GType codec)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), NULL);
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_VIDEO), NULL);

  return ogmrip_xml_get_codec_name (xml->priv->video_codecs, codec);
}

gint
ogmrip_xml_get_n_audio_codecs (OGMRipXml *xml)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), -1);

  return g_slist_length (xml->priv->audio_codecs);
}

void
ogmrip_xml_foreach_audio_codec (OGMRipXml *xml, OGMRipXmlFunc func, gpointer data)
{
  g_return_if_fail (OGMRIP_IS_XML (xml));
  g_return_if_fail (func != NULL);

  ogmrip_xml_foreach_codec (xml->priv->audio_codecs, func, data);
}

GType
ogmrip_xml_find_audio_codec (OGMRipXml *xml, OGMRipXmlCmpFunc func, gconstpointer data)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), G_TYPE_NONE);
  g_return_val_if_fail (func != NULL, G_TYPE_NONE);

  return ogmrip_xml_find_codec (xml->priv->audio_codecs, func, data);
}

GType
ogmrip_xml_get_nth_audio_codec (OGMRipXml *xml, guint n)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), G_TYPE_NONE);

  return ogmrip_xml_get_nth_codec (xml->priv->audio_codecs, n);
}

G_CONST_RETURN gchar *
ogmrip_xml_get_audio_codec_name (OGMRipXml *xml, GType codec)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), NULL);
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_AUDIO), NULL);

  return ogmrip_xml_get_codec_name (xml->priv->audio_codecs, codec);
}

gint
ogmrip_xml_get_n_subp_codecs (OGMRipXml *xml)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), -1);

  return g_slist_length (xml->priv->subp_codecs);
}

void
ogmrip_xml_foreach_subp_codec (OGMRipXml *xml, OGMRipXmlFunc func, gpointer data)
{
  g_return_if_fail (OGMRIP_IS_XML (xml));
  g_return_if_fail (func != NULL);

  ogmrip_xml_foreach_codec (xml->priv->subp_codecs, func, data);
}

GType
ogmrip_xml_find_subp_codec (OGMRipXml *xml, OGMRipXmlCmpFunc func, gconstpointer data)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), G_TYPE_NONE);
  g_return_val_if_fail (func != NULL, G_TYPE_NONE);

  return ogmrip_xml_find_codec (xml->priv->subp_codecs, func, data);
}

GType
ogmrip_xml_get_nth_subp_codec (OGMRipXml *xml, guint n)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), G_TYPE_NONE);

  return ogmrip_xml_get_nth_codec (xml->priv->subp_codecs, n);
}

G_CONST_RETURN gchar *
ogmrip_xml_get_subp_codec_name (OGMRipXml *xml, GType codec)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), NULL);
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_SUBP), NULL);

  return ogmrip_xml_get_codec_name (xml->priv->subp_codecs, codec);
}

static gboolean
ogmrip_xml_can_contain (GSList *containers, GType acontainer, GSList *codecs, GType acodec)
{
  OGMRipXmlContainer *container;
  OGMRipXmlCodec *codec;
  GSList *link;

  container = ogmrip_xml_find_container_by_type (containers, acontainer);
  if (!container)
    return FALSE;

  codec = ogmrip_xml_find_codec_by_type (codecs, acodec);
  if (!codec)
    return FALSE;

  for (link = container->codecs; link; link = link->next)
    if (link->data == codec)
      return TRUE;

  return FALSE;
}

gboolean
ogmrip_xml_can_contain_video (OGMRipXml *xml, GType container, GType codec)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), FALSE);
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_VIDEO), FALSE);

  return ogmrip_xml_can_contain (xml->priv->containers, container, xml->priv->video_codecs, codec);
}

gboolean
ogmrip_xml_can_contain_audio (OGMRipXml *xml, GType container, GType codec)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), FALSE);
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_AUDIO), FALSE);

  return ogmrip_xml_can_contain (xml->priv->containers, container, xml->priv->audio_codecs, codec);
}

gboolean
ogmrip_xml_can_contain_subp (OGMRipXml *xml, GType container, GType codec)
{
  g_return_val_if_fail (OGMRIP_IS_XML (xml), FALSE);
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);
  g_return_val_if_fail (g_type_is_a (codec, OGMRIP_TYPE_SUBP), FALSE);

  return ogmrip_xml_can_contain (xml->priv->containers, container, xml->priv->subp_codecs, codec);
}

gboolean
ogmrip_xml_can_contain_n_audio (OGMRipXml *xml, GType container, guint ncodec)
{
  OGMRipXmlContainer *cont;

  g_return_val_if_fail (OGMRIP_IS_XML (xml), FALSE);
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);

  cont = ogmrip_xml_find_container_by_type (xml->priv->containers, container);
  if (!cont)
    return FALSE;

  return ncodec <= cont->max_audio;
}

gboolean
ogmrip_xml_can_contain_n_subp (OGMRipXml *xml, GType container, guint ncodec)
{
  OGMRipXmlContainer *cont;

  g_return_val_if_fail (OGMRIP_IS_XML (xml), FALSE);
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);

  cont = ogmrip_xml_find_container_by_type (xml->priv->containers, container);
  if (!cont)
    return FALSE;

  return ncodec <= cont->max_subp;
}

gboolean
ogmrip_xml_get_subp_codec_text (OGMRipXml *xml, GType type)
{
  OGMRipXmlCodec *codec;

  g_return_val_if_fail (OGMRIP_IS_XML (xml), FALSE);
  g_return_val_if_fail (g_type_is_a (type, OGMRIP_TYPE_SUBP), FALSE);

  codec = ogmrip_xml_find_codec_by_type (xml->priv->subp_codecs, type);
  if (!codec)
    return FALSE;

  return ((OGMRipXmlSubpCodec *) codec)->text;
}

gint
ogmrip_xml_get_container_max_audio (OGMRipXml *xml, GType container)
{
  OGMRipXmlContainer *cont;

  g_return_val_if_fail (OGMRIP_IS_XML (xml), FALSE);
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);

  cont = ogmrip_xml_find_container_by_type (xml->priv->containers, container);
  if (!cont)
    return FALSE;

  return cont->max_audio;
}

gint
ogmrip_xml_get_container_max_subp (OGMRipXml *xml, GType container)
{
  OGMRipXmlContainer *cont;

  g_return_val_if_fail (OGMRIP_IS_XML (xml), FALSE);
  g_return_val_if_fail (g_type_is_a (container, OGMRIP_TYPE_CONTAINER), FALSE);

  cont = ogmrip_xml_find_container_by_type (xml->priv->containers, container);
  if (!cont)
    return FALSE;

  return cont->max_subp;
}

