/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "ogmrip-codec.h"

#include <unistd.h>
#include <glib/gstdio.h>

#define OGMRIP_CODEC_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_CODEC, OGMRipCodecPriv))

struct _OGMRipCodecPriv
{
  OGMDvdTitle *title;
  OGMDvdTime time_;
  gchar *output;

  guint framerate_numerator;
  guint framerate_denominator;
  guint framestep;

  gboolean do_unlink;
  gboolean dirty;
  gulong length;

  guint start;
  gint end;
};

enum 
{
  PROP_0,
  PROP_INPUT,
  PROP_OUTPUT,
  PROP_LENGTH,
  PROP_START_CHAPTER,
  PROP_END_CHAPTER,
  PROP_FRAMERATE_NUMERATOR,
  PROP_FRAMERATE_DENOMINATOR,
  PROP_FRAMESTEP
};

static void ogmrip_codec_finalize     (GObject      *gobject);
static void ogmrip_codec_set_property (GObject      *gobject,
                                       guint        property_id,
                                       const GValue *value,
                                       GParamSpec   *pspec);
static void ogmrip_codec_get_property (GObject      *gobject,
                                       guint        property_id,
                                       GValue       *value,
                                       GParamSpec   *pspec);

G_DEFINE_ABSTRACT_TYPE (OGMRipCodec, ogmrip_codec, OGMJOB_TYPE_BIN)

static void
ogmrip_codec_class_init (OGMRipCodecClass *klass)
{
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->finalize = ogmrip_codec_finalize;
  gobject_class->set_property = ogmrip_codec_set_property;
  gobject_class->get_property = ogmrip_codec_get_property;

  g_object_class_install_property (gobject_class, PROP_INPUT, 
        g_param_spec_pointer ("input", "Input property", "Set input title", 
           G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_OUTPUT, 
        g_param_spec_string ("output", "Output property", "Set output file", 
           NULL, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_LENGTH, 
        g_param_spec_long ("length", "Length property", "Get length", 
           0, G_MAXLONG, 0, G_PARAM_READABLE));

  g_object_class_install_property (gobject_class, PROP_START_CHAPTER, 
        g_param_spec_int ("start-chapter", "Start chapter property", "Set start chapter", 
           0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_END_CHAPTER, 
        g_param_spec_int ("end-chapter", "End chapter property", "Set end chapter", 
           -1, G_MAXINT, -1, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_FRAMERATE_NUMERATOR, 
        g_param_spec_uint ("framerate-numerator", "Framerate numerator property", "Set framerate numerator", 
           0, G_MAXUINT, 25, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_FRAMERATE_DENOMINATOR, 
        g_param_spec_uint ("framerate-denominator", "Framerate denominator property", "Set framerate denominator", 
           0, G_MAXUINT, 1, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_FRAMESTEP, 
        g_param_spec_uint ("framestep", "Framestep property", "Set framestep", 
           0, G_MAXUINT, 1, G_PARAM_READWRITE));

  g_type_class_add_private (klass, sizeof (OGMRipCodecPriv));
}

static void
ogmrip_codec_init (OGMRipCodec *codec)
{
  codec->priv = OGMRIP_CODEC_GET_PRIVATE (codec);
  codec->priv->framestep = 1;
  codec->priv->end = -1;
}

static void
ogmrip_codec_finalize (GObject *gobject)
{
  OGMRipCodec *codec;

  codec = OGMRIP_CODEC (gobject);

  if (codec->priv->output)
  {
    if (codec->priv->do_unlink && g_file_test (codec->priv->output, G_FILE_TEST_IS_REGULAR))
      g_unlink (codec->priv->output);

    g_free (codec->priv->output);
    codec->priv->output = NULL;
  }

  if (codec->priv->title)
    ogmdvd_title_unref (codec->priv->title);
  codec->priv->title = NULL;

  G_OBJECT_CLASS (ogmrip_codec_parent_class)->finalize (gobject);
}

static void
ogmrip_codec_set_property (GObject *gobject, guint property_id, const GValue *value, GParamSpec *pspec)
{
  OGMRipCodec *codec;

  codec = OGMRIP_CODEC (gobject);

  switch (property_id) 
  {
    case PROP_INPUT:
      ogmrip_codec_set_input (codec, g_value_get_pointer (value));
      break;
    case PROP_OUTPUT:
      ogmrip_codec_set_output (codec, g_value_get_string (value));
      break;
    case PROP_START_CHAPTER: 
      ogmrip_codec_set_chapters (codec, g_value_get_int (value), codec->priv->end);
      break;
    case PROP_END_CHAPTER: 
      ogmrip_codec_set_chapters (codec, codec->priv->start, g_value_get_int (value));
      break;
    case PROP_FRAMERATE_NUMERATOR: 
      ogmrip_codec_set_framerate (codec, g_value_get_uint (value), codec->priv->framerate_numerator);
      break;
    case PROP_FRAMERATE_DENOMINATOR: 
      ogmrip_codec_set_framerate (codec, codec->priv->framerate_denominator, g_value_get_uint (value));
      break;
    case PROP_FRAMESTEP: 
      ogmrip_codec_set_framestep (codec, g_value_get_uint (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

static void
ogmrip_codec_get_property (GObject *gobject, guint property_id, GValue *value, GParamSpec *pspec)
{
  OGMRipCodec *codec;

  codec = OGMRIP_CODEC (gobject);

  switch (property_id) 
  {
    case PROP_INPUT:
      g_value_set_pointer (value, codec->priv->title);
      break;
    case PROP_OUTPUT:
      g_value_set_static_string (value, codec->priv->output);
      break;
    case PROP_LENGTH: 
      g_value_set_long (value, ogmrip_codec_get_length (codec, NULL));
      break;
    case PROP_START_CHAPTER: 
      g_value_set_int (value, codec->priv->start);
      break;
    case PROP_END_CHAPTER: 
      g_value_set_int (value, codec->priv->end);
      break;
    case PROP_FRAMERATE_NUMERATOR: 
      g_value_set_uint (value, codec->priv->framerate_numerator);
      break;
    case PROP_FRAMERATE_DENOMINATOR: 
      g_value_set_uint (value, codec->priv->framerate_denominator);
      break;
    case PROP_FRAMESTEP: 
      g_value_set_uint (value, codec->priv->framestep);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

void
ogmrip_codec_set_output (OGMRipCodec *codec, const gchar *output)
{
  g_return_if_fail (OGMRIP_IS_CODEC (codec));

  g_free (codec->priv->output);
  codec->priv->output = g_strdup (output);
}

G_CONST_RETURN gchar *
ogmrip_codec_get_output (OGMRipCodec *codec)
{
  g_return_val_if_fail (OGMRIP_IS_CODEC (codec), NULL);

  return codec->priv->output;
}

void
ogmrip_codec_set_input (OGMRipCodec *codec, OGMDvdTitle *title)
{
  g_return_if_fail (OGMRIP_IS_CODEC (codec));
  g_return_if_fail (title != NULL);

  ogmdvd_title_ref (title);

  if (codec->priv->title)
    ogmdvd_title_unref (codec->priv->title);

  ogmdvd_title_get_framerate (title, 
      &codec->priv->framerate_numerator, 
      &codec->priv->framerate_denominator);

  codec->priv->title = title;
  codec->priv->dirty = TRUE;

  codec->priv->start = 0;
  codec->priv->end = -1;
}

OGMDvdTitle *
ogmrip_codec_get_input (OGMRipCodec *codec)
{
  g_return_val_if_fail (OGMRIP_IS_CODEC (codec), NULL);

  return codec->priv->title;
}

void
ogmrip_codec_set_chapters (OGMRipCodec *codec, guint start, gint end)
{
  g_return_if_fail (OGMRIP_IS_CODEC (codec));

  if (codec->priv->start != start || codec->priv->end != end)
  {
    gint nchap;

    nchap = ogmdvd_title_get_n_chapters (codec->priv->title);

    if (end < 0)
      end = nchap - 1;

    codec->priv->start = MIN ((gint) start, nchap - 1);
    codec->priv->end = CLAMP (end, (gint) start, nchap - 1);

    codec->priv->dirty = TRUE;
  }
}

void
ogmrip_codec_get_chapters (OGMRipCodec *codec, guint *start, guint *end)
{
  g_return_if_fail (OGMRIP_IS_CODEC (codec));
  g_return_if_fail (start != NULL);
  g_return_if_fail (end != NULL);

  *start = codec->priv->start;

  if (codec->priv->end < 0)
    *end = ogmdvd_title_get_n_chapters (codec->priv->title) - 1;
  else
    *end = codec->priv->end;
}

glong
ogmrip_codec_get_length (OGMRipCodec *codec, OGMDvdTime *time_)
{
  g_return_val_if_fail (OGMRIP_IS_CODEC (codec), -1);

  if (!codec->priv->title)
    return -1;

  if (codec->priv->dirty)
  {
    guint numerator, denominator;

    if (codec->priv->start == 0 && codec->priv->end == -1)
      codec->priv->length = ogmdvd_title_get_length (codec->priv->title, &codec->priv->time_);
    else
      codec->priv->length = ogmdvd_title_get_chapters_length (codec->priv->title, 
            codec->priv->start, codec->priv->end, &codec->priv->time_);

    ogmdvd_title_get_framerate (codec->priv->title, &numerator, &denominator);
    if (numerator != codec->priv->framerate_numerator || denominator != codec->priv->framerate_denominator)
      codec->priv->length *= ((denominator * codec->priv->framerate_numerator) / 
          (gdouble) (numerator * codec->priv->framerate_denominator));

    codec->priv->dirty = FALSE;
  }

  if (time_)
    *time_ = codec->priv->time_;

  return codec->priv->length;
}

void
ogmrip_codec_set_framerate (OGMRipCodec *codec, guint numerator, guint denominator)
{
  g_return_if_fail (OGMRIP_IS_CODEC (codec));
  g_return_if_fail (numerator > 0 && denominator > 0);

  codec->priv->framerate_numerator = numerator;
  codec->priv->framerate_denominator = denominator;
}

void
ogmrip_codec_get_framerate (OGMRipCodec *codec, guint *numerator, guint *denominator)
{
  g_return_if_fail (OGMRIP_IS_CODEC (codec));
  g_return_if_fail (denominator != NULL);
  g_return_if_fail (numerator != NULL);

  *numerator = codec->priv->framerate_numerator;
  *denominator = codec->priv->framerate_denominator;

  if (!numerator || !denominator)
  {
    OGMDvdTitle *title;

    title = ogmrip_codec_get_input (OGMRIP_CODEC (codec));
    ogmdvd_title_get_framerate (title, numerator, denominator);
  }
}

void
ogmrip_codec_set_framestep (OGMRipCodec *codec, guint framestep)
{
  g_return_if_fail (OGMRIP_IS_CODEC (codec));

  codec->priv->framestep = MAX (framestep, 1);
}

gint
ogmrip_codec_get_framestep (OGMRipCodec *codec)
{
  g_return_val_if_fail (OGMRIP_IS_CODEC (codec), -1);

  return codec->priv->framestep;
}

void
ogmrip_codec_set_unlink_on_unref (OGMRipCodec *codec, gboolean do_unlink)
{
  g_return_if_fail (OGMRIP_IS_CODEC (codec));

  codec->priv->do_unlink = do_unlink;
}

gboolean
ogmrip_codec_get_unlink_on_unref (OGMRipCodec *codec)
{
  g_return_val_if_fail (OGMRIP_IS_CODEC (codec), FALSE);

  return codec->priv->do_unlink;
}

