/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "ogmrip-audio.h"

#define OGMRIP_AUDIO_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_AUDIO, OGMRipAudioPriv))

struct _OGMRipAudioPriv
{
  guint srate;
  guint quality;
  guint samples_per_frame;

  gboolean normalize;

  OGMDvdAudioChannels channels;
  OGMDvdAudioStream *stream;
};

enum 
{
  PROP_0,
  PROP_STREAM,
  PROP_QUALITY,
  PROP_NORMALIZE,
  PROP_CHANNELS,
  PROP_SPF
};

static void ogmrip_audio_finalize     (GObject      *gobject);
static void ogmrip_audio_set_property (GObject      *gobject,
                                       guint        property_id,
                                       const GValue *value,
                                       GParamSpec   *pspec);
static void ogmrip_audio_get_property (GObject      *gobject,
                                       guint        property_id,
                                       GValue       *value,
                                       GParamSpec   *pspec);

G_DEFINE_ABSTRACT_TYPE (OGMRipAudio, ogmrip_audio, OGMRIP_TYPE_CODEC)

static void
ogmrip_audio_class_init (OGMRipAudioClass *klass)
{
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->finalize = ogmrip_audio_finalize;
  gobject_class->set_property = ogmrip_audio_set_property;
  gobject_class->get_property = ogmrip_audio_get_property;

  g_object_class_install_property (gobject_class, PROP_STREAM, 
        g_param_spec_pointer ("stream", "Audio stream property", "Set audio stream", 
           G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_QUALITY, 
        g_param_spec_uint ("quality", "Quality property", "Set quality", 
           0, 10, 3, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_NORMALIZE, 
        g_param_spec_boolean ("normalize", "Normalize property", "Set normalize", 
           FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_CHANNELS, 
        g_param_spec_uint ("channels", "Channels property", "Set channels", 
           0, 10, OGMDVD_AUDIO_CHANNELS_STEREO, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_SPF, 
        g_param_spec_uint ("samples-per-frame", "Samples per frame property", "Set samples per frame", 
           0, G_MAXUINT, 512, G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

  g_type_class_add_private (klass, sizeof (OGMRipAudioPriv));
}

static void
ogmrip_audio_init (OGMRipAudio *audio)
{
  audio->priv = OGMRIP_AUDIO_GET_PRIVATE (audio);
  audio->priv->channels = OGMDVD_AUDIO_CHANNELS_STEREO;

  audio->priv->srate = 48000;
  audio->priv->samples_per_frame = 512;
  audio->priv->quality = 3;
}

static void
ogmrip_audio_finalize (GObject *gobject)
{
  OGMRipAudio *audio;

  audio = OGMRIP_AUDIO (gobject);
  if (audio->priv->stream)
    ogmdvd_audio_stream_unref (audio->priv->stream);
  audio->priv->stream = NULL;

  G_OBJECT_CLASS (ogmrip_audio_parent_class)->finalize (gobject);
}

static void
ogmrip_audio_set_property (GObject *gobject, guint property_id, const GValue *value, GParamSpec *pspec)
{
  OGMRipAudio *audio;

  audio = OGMRIP_AUDIO (gobject);

  switch (property_id) 
  {
    case PROP_STREAM:
      ogmrip_audio_set_dvd_audio_stream (audio, g_value_get_pointer (value));
      break;
    case PROP_QUALITY: 
      ogmrip_audio_set_quality (audio, g_value_get_uint (value));
      break;
    case PROP_NORMALIZE: 
      ogmrip_audio_set_normalize (audio, g_value_get_boolean (value));
      break;
    case PROP_CHANNELS: 
      ogmrip_audio_set_channels (audio, g_value_get_uint (value));
      break;
    case PROP_SPF: 
      audio->priv->samples_per_frame = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

static void
ogmrip_audio_get_property (GObject *gobject, guint property_id, GValue *value, GParamSpec *pspec)
{
  OGMRipAudio *audio;

  audio = OGMRIP_AUDIO (gobject);

  switch (property_id) 
  {
    case PROP_STREAM:
      g_value_set_pointer (value, audio->priv->stream);
      break;
    case PROP_QUALITY:
      g_value_set_uint (value, audio->priv->quality);
      break;
    case PROP_NORMALIZE:
      g_value_set_boolean (value, audio->priv->normalize);
      break;
    case PROP_CHANNELS:
      g_value_set_uint (value, audio->priv->channels);
      break;
    case PROP_SPF:
      g_value_set_uint (value, audio->priv->samples_per_frame);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

void
ogmrip_audio_set_quality (OGMRipAudio *audio, guint quality)
{
  g_return_if_fail (OGMRIP_IS_AUDIO (audio));

  audio->priv->quality = MIN (quality, 10);
}

gint
ogmrip_audio_get_quality (OGMRipAudio *audio)
{
  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), -1);

  return audio->priv->quality;
}

void
ogmrip_audio_set_normalize (OGMRipAudio *audio, gboolean normalize)
{
  g_return_if_fail (OGMRIP_IS_AUDIO (audio));

  audio->priv->normalize = normalize;
}

gboolean
ogmrip_audio_get_normalize (OGMRipAudio *audio)
{
  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), FALSE);

  return audio->priv->normalize;
}

void
ogmrip_audio_set_dvd_audio_stream (OGMRipAudio *audio, OGMDvdAudioStream *stream)
{
  g_return_if_fail (OGMRIP_IS_AUDIO (audio));
  g_return_if_fail (stream != NULL);

  ogmdvd_audio_stream_ref (stream);

  if (audio->priv->stream)
    ogmdvd_audio_stream_unref (audio->priv->stream);
  audio->priv->stream = stream;

  ogmrip_codec_set_input (OGMRIP_CODEC (audio), 
      ogmdvd_audio_stream_get_title (stream));

  ogmrip_audio_set_channels (audio, audio->priv->channels);
}

OGMDvdAudioStream *
ogmrip_audio_get_dvd_audio_stream (OGMRipAudio *audio)
{
  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), NULL);

  return audio->priv->stream;
}

void
ogmrip_audio_set_channels (OGMRipAudio *audio, OGMDvdAudioChannels channels)
{
  g_return_if_fail (OGMRIP_IS_AUDIO (audio));
  
  audio->priv->channels = MIN (channels, 
      ogmdvd_audio_stream_get_channels (audio->priv->stream));
}

gint
ogmrip_audio_get_channels (OGMRipAudio *audio)
{
  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), -1);

  return audio->priv->channels;
}

gint
ogmrip_audio_get_samples_per_frame (OGMRipAudio *audio)
{
  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), -1);

  return audio->priv->samples_per_frame * (audio->priv->channels + 1);
}

void
ogmrip_audio_set_sample_rate (OGMRipAudio *audio, guint srate)
{
  g_return_if_fail (OGMRIP_IS_AUDIO (audio));

  audio->priv->srate = srate;
}

gint
ogmrip_audio_get_sample_rate (OGMRipAudio *audio)
{
  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), -1);

  return audio->priv->srate;
}

